package io.dingodb.sdk.service.entity.common;

import com.google.protobuf.CodedInputStream;
import com.google.protobuf.CodedOutputStream;
import io.dingodb.sdk.grpc.serializer.Reader;
import io.dingodb.sdk.grpc.serializer.SizeUtils;
import io.dingodb.sdk.grpc.serializer.Writer;
import io.dingodb.sdk.service.entity.Message;
import io.dingodb.sdk.service.entity.Numeric;
import java.util.List;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.FieldNameConstants;
import lombok.experimental.SuperBuilder;

@Data
@SuperBuilder
@NoArgsConstructor
@FieldNameConstants
public class VectorSearchParameter implements Message {
    private boolean useBruteForce;

    private CoprocessorV2 vectorCoprocessor;

    private boolean withoutTableData;

    private VectorFilter vectorFilter;

    private boolean withoutVectorData;

    private VectorFilterType vectorFilterType;

    private int topN;

    private boolean isSorted;

    private List<Long> vectorIds;

    private List<String> selectedKeys;

    private boolean isNegation;

    private float radius;

    private boolean enableRangeSearch;

    private SearchNest search;

    private boolean withoutScalarData;

    private Object ext$;

    @Override
    public void write(CodedOutputStream out) {
        Writer.write(1, topN, out);
        Writer.write(2, withoutVectorData, out);
        Writer.write(3, withoutScalarData, out);
        Writer.write(4, selectedKeys, (n, v) -> Writer.write(n, v, out));
        Writer.write(5, withoutTableData, out);
        Writer.write(6, enableRangeSearch, out);
        Writer.write(7, radius, out);
        Writer.write(search, search, out);
        Writer.write(21, vectorFilter, out);
        Writer.write(22, vectorFilterType, out);
        Writer.write(23, vectorCoprocessor, out);
        Writer.write(24, vectorIds, out, Writer::write, SizeUtils::sizeOf);
        Writer.write(25, useBruteForce, out);
        Writer.write(26, isNegation, out);
        Writer.write(27, isSorted, out);
    }

    @Override
    public boolean read(CodedInputStream input) {
        int number = 0;
        boolean hasValue = false;
        while((number = Reader.readNumber(input)) != 0) {
            switch(number) {
                case 1: topN = Reader.readInt(input); hasValue = true; break;
                case 2: withoutVectorData = Reader.readBoolean(input); hasValue = true; break;
                case 3: withoutScalarData = Reader.readBoolean(input); hasValue = true; break;
                case 4: selectedKeys = Reader.readList(selectedKeys, input, Reader::readString); hasValue = true; break;
                case 5: withoutTableData = Reader.readBoolean(input); hasValue = true; break;
                case 6: enableRangeSearch = Reader.readBoolean(input); hasValue = true; break;
                case 7: radius = Reader.readFloat(input); hasValue = true; break;
                case 11: search = Reader.readMessage(new SearchNest.Flat(), input); hasValue = hasValue ? hasValue : search != null; break;
                case 12: search = Reader.readMessage(new SearchNest.IvfFlat(), input); hasValue = hasValue ? hasValue : search != null; break;
                case 13: search = Reader.readMessage(new SearchNest.IvfPq(), input); hasValue = hasValue ? hasValue : search != null; break;
                case 14: search = Reader.readMessage(new SearchNest.Hnsw(), input); hasValue = hasValue ? hasValue : search != null; break;
                case 15: search = Reader.readMessage(new SearchNest.Diskann(), input); hasValue = hasValue ? hasValue : search != null; break;
                case 21: vectorFilter = VectorFilter.forNumber(Reader.readInt(input));hasValue = true; break;
                case 22: vectorFilterType = VectorFilterType.forNumber(Reader.readInt(input));hasValue = true; break;
                case 23: vectorCoprocessor = Reader.readMessage(new CoprocessorV2(), input); hasValue = hasValue ? hasValue : vectorCoprocessor != null; break;
                case 24: vectorIds = Reader.readPack(input, Reader::readLong); hasValue = true; break;
                case 25: useBruteForce = Reader.readBoolean(input); hasValue = true; break;
                case 26: isNegation = Reader.readBoolean(input); hasValue = true; break;
                case 27: isSorted = Reader.readBoolean(input); hasValue = true; break;
                case 30: search = Reader.readMessage(new SearchNest.BinaryFlat(), input); hasValue = hasValue ? hasValue : search != null; break;
                case 31: search = Reader.readMessage(new SearchNest.BinaryIvfFlat(), input); hasValue = hasValue ? hasValue : search != null; break;
                default: Reader.skip(input);
            }
        }
        return hasValue;
    }

    @Override
    public int sizeOf() {
        int size = 0;
        size += SizeUtils.sizeOf(1, topN);
        size += SizeUtils.sizeOf(2, withoutVectorData);
        size += SizeUtils.sizeOf(3, withoutScalarData);
        size += SizeUtils.sizeOf(4, selectedKeys, SizeUtils::sizeOf);
        size += SizeUtils.sizeOf(5, withoutTableData);
        size += SizeUtils.sizeOf(6, enableRangeSearch);
        size += SizeUtils.sizeOf(7, radius);
        size += SizeUtils.sizeOf(search, search);
        size += SizeUtils.sizeOf(21, vectorFilter);
        size += SizeUtils.sizeOf(22, vectorFilterType);
        size += SizeUtils.sizeOf(23, vectorCoprocessor);
        size += SizeUtils.sizeOfPack(24, vectorIds, SizeUtils::sizeOf);
        size += SizeUtils.sizeOf(25, useBruteForce);
        size += SizeUtils.sizeOf(26, isNegation);
        size += SizeUtils.sizeOf(27, isSorted);
        return size;
    }

    public interface SearchNest extends Message, Numeric {
        int number();

        Nest nest();

        @NoArgsConstructor
        @SuperBuilder
        class Flat extends SearchFlatParam implements SearchNest {
            public static final int number = 11;

            @Override
            public int number() {
                return 11;
            }

            @Override
            public Nest nest() {
                return Nest.FLAT;
            }
        }

        @NoArgsConstructor
        @SuperBuilder
        class IvfFlat extends SearchIvfFlatParam implements SearchNest {
            public static final int number = 12;

            @Override
            public int number() {
                return 12;
            }

            @Override
            public Nest nest() {
                return Nest.IVF_FLAT;
            }
        }

        @NoArgsConstructor
        @SuperBuilder
        class IvfPq extends SearchIvfPqParam implements SearchNest {
            public static final int number = 13;

            @Override
            public int number() {
                return 13;
            }

            @Override
            public Nest nest() {
                return Nest.IVF_PQ;
            }
        }

        @NoArgsConstructor
        @SuperBuilder
        class Hnsw extends SearchHNSWParam implements SearchNest {
            public static final int number = 14;

            @Override
            public int number() {
                return 14;
            }

            @Override
            public Nest nest() {
                return Nest.HNSW;
            }
        }

        @NoArgsConstructor
        @SuperBuilder
        class Diskann extends SearchDiskAnnParam implements SearchNest {
            public static final int number = 15;

            @Override
            public int number() {
                return 15;
            }

            @Override
            public Nest nest() {
                return Nest.DISKANN;
            }
        }

        @NoArgsConstructor
        @SuperBuilder
        class BinaryFlat extends SearchBinaryFlatParam implements SearchNest {
            public static final int number = 30;

            @Override
            public int number() {
                return 30;
            }

            @Override
            public Nest nest() {
                return Nest.BINARY_FLAT;
            }
        }

        @NoArgsConstructor
        @SuperBuilder
        class BinaryIvfFlat extends SearchBinaryIvfFlatParam implements SearchNest {
            public static final int number = 31;

            @Override
            public int number() {
                return 31;
            }

            @Override
            public Nest nest() {
                return Nest.BINARY_IVF_FLAT;
            }
        }

        enum Nest implements Numeric {
            FLAT(11),

            IVF_FLAT(12),

            IVF_PQ(13),

            HNSW(14),

            DISKANN(15),

            BINARY_FLAT(30),

            BINARY_IVF_FLAT(31);

            public final Integer number;

            private Nest(Integer number) {
                this.number = number;
            }

            @Override
            public int number() {
                return number;
            }
        }
    }
}
