/**
 * Copyright (c) 2016-2022 Deephaven Data Labs and Patent Pending
 */
package io.deephaven.lang.completion;

import io.deephaven.base.Lazy;
import io.deephaven.engine.context.ExecutionContext;
import io.deephaven.engine.table.TableDefinition;
import io.deephaven.engine.context.QueryLibrary;
import io.deephaven.engine.util.ScriptSession;

import java.util.Collection;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ForkJoinPool;

/**
 * A lookup object for various values that the {@link ChunkerCompleter} might be interested in.
 *
 * This is extracted into its own class, so preloading can start as soon as the console session starts.
 *
 */
public class CompletionLookups {

    private static final WeakHashMap<ScriptSession, CompletionLookups> lookups = new WeakHashMap<>();

    private final Lazy<Collection<Class<?>>> statics;
    private final Map<String, TableDefinition> referencedTables;

    public CompletionLookups() {
        final QueryLibrary ql = ExecutionContext.getContext().getQueryLibrary();
        statics = new Lazy<>(ql::getStaticImports);
        referencedTables = new ConcurrentHashMap<>();

        // This can be slow, so lets start it on a background thread right away.
        final ForkJoinPool pool = ForkJoinPool.commonPool();
        pool.execute(statics::get);
    }

    public static CompletionLookups preload(ScriptSession session) {
        return lookups.computeIfAbsent(session, s -> new CompletionLookups());
    }

    public Collection<Class<?>> getStatics() {
        return statics.get();
    }

    public Map<String, TableDefinition> getReferencedTables() {
        return referencedTables;
    }
}
