/*
 * Decompiled with CFR 0.152.
 */
package io.atomix.copycat.server.state;

import io.atomix.catalyst.concurrent.Futures;
import io.atomix.catalyst.concurrent.Listener;
import io.atomix.catalyst.concurrent.Listeners;
import io.atomix.catalyst.concurrent.Scheduled;
import io.atomix.catalyst.transport.Address;
import io.atomix.catalyst.util.Assert;
import io.atomix.copycat.error.CopycatError;
import io.atomix.copycat.protocol.Response;
import io.atomix.copycat.server.CopycatServer;
import io.atomix.copycat.server.cluster.Cluster;
import io.atomix.copycat.server.cluster.Member;
import io.atomix.copycat.server.protocol.JoinRequest;
import io.atomix.copycat.server.protocol.LeaveRequest;
import io.atomix.copycat.server.protocol.ReconfigureRequest;
import io.atomix.copycat.server.state.LeaderState;
import io.atomix.copycat.server.state.MemberState;
import io.atomix.copycat.server.state.ServerContext;
import io.atomix.copycat.server.state.ServerMember;
import io.atomix.copycat.server.storage.system.Configuration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class ClusterState
implements Cluster,
AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(ClusterState.class);
    private final ServerContext context;
    private final ServerMember member;
    private volatile Configuration configuration;
    private final Map<Integer, MemberState> membersMap = new ConcurrentHashMap<Integer, MemberState>();
    private final Map<Address, MemberState> addressMap = new ConcurrentHashMap<Address, MemberState>();
    private final Set<Member> members = new CopyOnWriteArraySet<Member>();
    private final List<MemberState> remoteMembers = new CopyOnWriteArrayList<MemberState>();
    private List<MemberState> assignedMembers = new ArrayList<MemberState>();
    private final Map<Member.Type, List<MemberState>> memberTypes = new HashMap<Member.Type, List<MemberState>>();
    private volatile Scheduled joinTimeout;
    private volatile CompletableFuture<Void> joinFuture;
    private volatile Scheduled leaveTimeout;
    private volatile CompletableFuture<Void> leaveFuture;
    private final Listeners<Member> joinListeners = new Listeners();
    private final Listeners<Member> leaveListeners = new Listeners();

    ClusterState(Member.Type type, Address serverAddress, Address clientAddress, ServerContext context) {
        Instant time = Instant.now();
        this.member = new ServerMember(type, serverAddress, clientAddress, time).setCluster(this);
        this.context = Assert.notNull(context, "context");
        this.configuration = context.getMetaStore().loadConfiguration();
        if (this.configuration != null) {
            Instant updateTime = Instant.ofEpochMilli(this.configuration.time());
            for (Member member : this.configuration.members()) {
                if (member.equals(this.member)) {
                    this.member.update(member.type(), updateTime).update(member.clientAddress(), updateTime);
                    this.members.add(this.member);
                    continue;
                }
                MemberState state = new MemberState(new ServerMember(member.type(), member.serverAddress(), member.clientAddress(), updateTime), this);
                state.resetState(context.getLog());
                this.members.add(state.getMember());
                this.remoteMembers.add(state);
                this.membersMap.put(member.id(), state);
                this.addressMap.put(member.address(), state);
                List<MemberState> memberType = this.memberTypes.get((Object)member.type());
                if (memberType == null) {
                    memberType = new CopyOnWriteArrayList<MemberState>();
                    this.memberTypes.put(member.type(), memberType);
                }
                memberType.add(state);
            }
        }
    }

    ServerContext getContext() {
        return this.context;
    }

    Configuration getConfiguration() {
        return this.configuration;
    }

    @Override
    public Member leader() {
        return this.context.getLeader();
    }

    @Override
    public long term() {
        return this.context.getTerm();
    }

    @Override
    public Listener<Member> onLeaderElection(Consumer<Member> callback) {
        return this.context.onLeaderElection(callback);
    }

    @Override
    public Member member() {
        return this.member;
    }

    @Override
    public Collection<Member> members() {
        return new ArrayList<Member>(this.members);
    }

    @Override
    public ServerMember member(int id) {
        if (this.member.id() == id) {
            return this.member;
        }
        return this.getRemoteMember(id);
    }

    @Override
    public Member member(Address address) {
        if (this.member.address().equals(address)) {
            return this.member;
        }
        MemberState member = this.addressMap.get(address);
        return member != null ? member.getMember() : null;
    }

    @Override
    public Listener<Member> onJoin(Consumer<Member> callback) {
        return this.joinListeners.add(callback);
    }

    @Override
    public Listener<Member> onLeave(Consumer<Member> callback) {
        return this.leaveListeners.add(callback);
    }

    int getQuorum() {
        return (int)Math.floor((double)(this.getActiveMemberStates().size() + 1) / 2.0) + 1;
    }

    public MemberState getMemberState(int id) {
        return this.membersMap.get(id);
    }

    public ServerMember getRemoteMember(int id) {
        MemberState member = this.membersMap.get(id);
        return member != null ? member.getMember() : null;
    }

    public List<MemberState> getRemoteMemberStates() {
        return this.remoteMembers;
    }

    public List<MemberState> getRemoteMemberStates(Member.Type type) {
        List members = this.memberTypes.get((Object)type);
        return members != null ? members : Collections.EMPTY_LIST;
    }

    List<MemberState> getActiveMemberStates() {
        return this.getRemoteMemberStates(Member.Type.ACTIVE);
    }

    List<MemberState> getActiveMemberStates(Comparator<MemberState> comparator) {
        ArrayList<MemberState> activeMembers = new ArrayList<MemberState>(this.getActiveMemberStates());
        Collections.sort(activeMembers, comparator);
        return activeMembers;
    }

    List<MemberState> getPassiveMemberStates() {
        return this.getRemoteMemberStates(Member.Type.PASSIVE);
    }

    List<MemberState> getPassiveMemberStates(Comparator<MemberState> comparator) {
        ArrayList<MemberState> passiveMembers = new ArrayList<MemberState>(this.getPassiveMemberStates());
        Collections.sort(passiveMembers, comparator);
        return passiveMembers;
    }

    List<MemberState> getReserveMemberStates() {
        return this.getRemoteMemberStates(Member.Type.RESERVE);
    }

    List<MemberState> getReserveMemberStates(Comparator<MemberState> comparator) {
        ArrayList<MemberState> reserveMembers = new ArrayList<MemberState>(this.getReserveMemberStates());
        Collections.sort(reserveMembers, comparator);
        return reserveMembers;
    }

    List<MemberState> getAssignedPassiveMemberStates() {
        return this.assignedMembers;
    }

    @Override
    public CompletableFuture<Void> bootstrap(Collection<Address> cluster) {
        if (this.joinFuture != null) {
            return this.joinFuture;
        }
        if (this.configuration == null) {
            if (this.member.type() != Member.Type.ACTIVE) {
                return Futures.exceptionalFuture(new IllegalStateException("only ACTIVE members can bootstrap the cluster"));
            }
            Set<Member> activeMembers = cluster.stream().filter(m -> !m.equals(this.member.serverAddress())).map(m -> new ServerMember(Member.Type.ACTIVE, (Address)m, null, this.member.updated())).collect(Collectors.toSet());
            activeMembers.add(this.member);
            this.configure(new Configuration(0L, 0L, this.member.updated().toEpochMilli(), activeMembers));
        }
        return this.join();
    }

    @Override
    public synchronized CompletableFuture<Void> join(Collection<Address> cluster) {
        if (this.joinFuture != null) {
            return this.joinFuture;
        }
        if (this.configuration == null) {
            Set<Member> activeMembers = cluster.stream().filter(m -> !m.equals(this.member.serverAddress())).map(m -> new ServerMember(Member.Type.ACTIVE, (Address)m, null, this.member.updated())).collect(Collectors.toSet());
            if (activeMembers.isEmpty()) {
                return Futures.exceptionalFuture(new IllegalStateException("cannot join empty cluster"));
            }
            this.configure(new Configuration(0L, 0L, this.member.updated().toEpochMilli(), activeMembers));
        }
        return this.join();
    }

    private synchronized CompletableFuture<Void> join() {
        this.joinFuture = new CompletableFuture();
        this.context.getThreadContext().executor().execute(() -> {
            this.context.transition(this.member.type());
            List<MemberState> activeMembers = this.getActiveMemberStates();
            if (!activeMembers.isEmpty()) {
                this.join(this.getActiveMemberStates().iterator());
            } else {
                this.joinFuture.complete(null);
            }
        });
        return this.joinFuture.whenComplete((result, error) -> {
            this.joinFuture = null;
        });
    }

    private void join(Iterator<MemberState> iterator) {
        if (iterator.hasNext()) {
            this.cancelJoinTimer();
            this.joinTimeout = this.context.getThreadContext().schedule(this.context.getElectionTimeout().multipliedBy(2L), () -> this.join(iterator));
            MemberState member = iterator.next();
            LOGGER.debug("{} - Attempting to join via {}", (Object)this.member().address(), (Object)member.getMember().serverAddress());
            ((CompletableFuture)this.context.getConnections().getConnection(member.getMember().serverAddress()).thenCompose(connection -> {
                JoinRequest request = (JoinRequest)((JoinRequest.Builder)JoinRequest.builder().withMember(new ServerMember(this.member().type(), this.member().serverAddress(), this.member().clientAddress(), this.member().updated()))).build();
                return connection.sendAndReceive(request);
            })).whenComplete((response, error) -> {
                this.cancelJoinTimer();
                if (error == null) {
                    if (response.status() == Response.Status.OK) {
                        LOGGER.info("{} - Successfully joined via {}", (Object)this.member().address(), (Object)member.getMember().serverAddress());
                        Configuration configuration = new Configuration(response.index(), response.term(), response.timestamp(), response.members());
                        this.configure(configuration).commit();
                        if (!this.members.contains(this.member)) {
                            this.joinFuture.completeExceptionally(new IllegalStateException("not a member of the cluster"));
                        } else if (this.joinFuture != null) {
                            this.joinFuture.complete(null);
                        }
                    } else if (response.error() == null || response.error() == CopycatError.Type.CONFIGURATION_ERROR) {
                        LOGGER.debug("{} - Failed to join {}", (Object)this.member().address(), (Object)member.getMember().address());
                        this.resetJoinTimer();
                    } else {
                        LOGGER.debug("{} - Failed to join {}", (Object)this.member().address(), (Object)member.getMember().address());
                        this.join(iterator);
                    }
                } else {
                    LOGGER.debug("{} - Failed to join {}", (Object)this.member().address(), (Object)member.getMember().address());
                    this.join(iterator);
                }
            });
        } else {
            LOGGER.debug("{} - Failed to join cluster, retrying...", (Object)this.member.address());
            this.resetJoinTimer();
        }
    }

    CompletableFuture<Void> identify() {
        ServerMember leader = this.context.getLeader();
        if (this.joinFuture != null && leader != null) {
            if (this.context.getLeader().equals(this.member())) {
                if (this.context.getState() == CopycatServer.State.LEADER && !((LeaderState)this.context.getServerState()).configuring()) {
                    if (this.joinFuture != null) {
                        this.joinFuture.complete(null);
                    }
                } else {
                    this.cancelJoinTimer();
                    this.joinTimeout = this.context.getThreadContext().schedule(this.context.getElectionTimeout().multipliedBy(2L), this::identify);
                }
            } else {
                this.cancelJoinTimer();
                this.joinTimeout = this.context.getThreadContext().schedule(this.context.getElectionTimeout().multipliedBy(2L), this::identify);
                LOGGER.debug("{} - Sending server identification to {}", (Object)this.member().address(), (Object)leader.address());
                ((CompletableFuture)this.context.getConnections().getConnection(leader.serverAddress()).thenCompose(connection -> {
                    ReconfigureRequest request = (ReconfigureRequest)((ReconfigureRequest.Builder)ReconfigureRequest.builder().withIndex(this.configuration.index()).withTerm(this.configuration.term()).withMember(this.member())).build();
                    LOGGER.trace("{} - Sending {} to {}", this.member.address(), request, leader.address());
                    return connection.sendAndReceive(request);
                })).whenComplete((response, error) -> {
                    this.cancelJoinTimer();
                    if (error == null) {
                        LOGGER.trace("{} - Received {}", (Object)this.member.address(), response);
                        if (response.status() == Response.Status.OK) {
                            if (this.joinFuture != null) {
                                this.joinFuture.complete(null);
                            }
                        } else if (response.error() == null || response.error() == CopycatError.Type.CONFIGURATION_ERROR) {
                            LOGGER.debug("{} - Failed to update configuration: configuration change already in progress", (Object)this.member.address());
                            this.joinTimeout = this.context.getThreadContext().schedule(this.context.getElectionTimeout().multipliedBy(2L), this::identify);
                        }
                    } else {
                        LOGGER.warn("{} - Failed to update configuration: {}", (Object)this.member.address(), (Object)error.getMessage());
                        this.joinTimeout = this.context.getThreadContext().schedule(this.context.getElectionTimeout().multipliedBy(2L), this::identify);
                    }
                });
            }
        }
        return this.joinFuture;
    }

    private void resetJoinTimer() {
        this.cancelJoinTimer();
        this.joinTimeout = this.context.getThreadContext().schedule(this.context.getElectionTimeout().multipliedBy(2L), () -> this.join(this.getActiveMemberStates().iterator()));
    }

    private void cancelJoinTimer() {
        if (this.joinTimeout != null) {
            LOGGER.trace("{} - Cancelling join timeout", (Object)this.member().address());
            this.joinTimeout.cancel();
            this.joinTimeout = null;
        }
    }

    @Override
    public synchronized CompletableFuture<Void> leave() {
        if (this.leaveFuture != null) {
            return this.leaveFuture;
        }
        this.leaveFuture = new CompletableFuture();
        this.context.getThreadContext().executor().execute(() -> {
            this.cancelJoinTimer();
            if (this.joinFuture != null) {
                this.joinFuture.completeExceptionally(new IllegalStateException("failed to join cluster"));
            }
            if (this.getActiveMemberStates().isEmpty() && this.configuration.index() <= this.context.getCommitIndex()) {
                LOGGER.trace("{} - Single member cluster. Transitioning directly to inactive.", (Object)this.member().address());
                this.context.transition(CopycatServer.State.INACTIVE);
                this.leaveFuture.complete(null);
            } else {
                this.leave(this.leaveFuture);
            }
        });
        return this.leaveFuture.whenComplete((result, error) -> {
            this.leaveFuture = null;
        });
    }

    private void leave(CompletableFuture<Void> future) {
        this.leaveTimeout = this.context.getThreadContext().schedule(this.context.getElectionTimeout(), () -> this.leave(future));
        this.context.getServerState().leave((LeaveRequest)((LeaveRequest.Builder)LeaveRequest.builder().withMember(this.member())).build()).whenComplete((response, error) -> {
            this.cancelLeaveTimer();
            if (error == null && response.status() == Response.Status.OK) {
                Configuration configuration = new Configuration(response.index(), response.term(), response.timestamp(), response.members());
                this.configure(configuration).commit();
                future.complete(null);
            } else {
                this.leaveTimeout = this.context.getThreadContext().schedule(this.context.getElectionTimeout(), () -> this.leave(future));
            }
        });
    }

    private void cancelLeaveTimer() {
        if (this.leaveTimeout != null) {
            LOGGER.trace("{} - Cancelling leave timeout", (Object)this.member().address());
            this.leaveTimeout.cancel();
            this.leaveTimeout = null;
        }
    }

    ClusterState reset() {
        this.configure(this.context.getMetaStore().loadConfiguration());
        return this;
    }

    ClusterState commit() {
        this.context.transition(this.member.type());
        if (!this.configuration.members().contains(this.member) && this.leaveFuture != null) {
            this.leaveFuture.complete(null);
        }
        if (this.context.getMetaStore().loadConfiguration().index() < this.configuration.index()) {
            this.context.getMetaStore().storeConfiguration(this.configuration);
        }
        return this;
    }

    ClusterState configure(Configuration configuration) {
        Assert.notNull(configuration, "configuration");
        if (this.configuration != null && configuration.index() <= this.configuration.index()) {
            return this;
        }
        Instant time = Instant.ofEpochMilli(configuration.time());
        boolean transition = false;
        for (Member member : configuration.members()) {
            if (member.equals(this.member)) {
                transition = this.member.type().ordinal() < member.type().ordinal();
                this.member.update(member.type(), time).update(member.clientAddress(), time);
                this.members.add(this.member);
                continue;
            }
            MemberState state = this.membersMap.get(member.id());
            if (state == null) {
                state = new MemberState(new ServerMember(member.type(), member.serverAddress(), member.clientAddress(), time), this);
                state.resetState(this.context.getLog());
                this.members.add(state.getMember());
                this.remoteMembers.add(state);
                this.membersMap.put(member.id(), state);
                this.addressMap.put(member.address(), state);
                this.joinListeners.accept(state.getMember());
            }
            state.getMember().update(member.clientAddress(), time);
            if (state.getMember().type() != member.type()) {
                state.getMember().update(member.type(), time);
                state.resetState(this.context.getLog());
            }
            if (state.getMember().status() != member.status()) {
                state.getMember().update(member.status(), time);
            }
            for (List<MemberState> memberType : this.memberTypes.values()) {
                memberType.remove(state);
            }
            List<MemberState> memberType = this.memberTypes.get((Object)member.type());
            if (memberType == null) {
                memberType = new CopyOnWriteArrayList<MemberState>();
                this.memberTypes.put(member.type(), memberType);
            }
            memberType.add(state);
        }
        if (transition) {
            this.context.transition(this.member.type());
        }
        int i = 0;
        while (i < this.remoteMembers.size()) {
            MemberState memberState = this.remoteMembers.get(i);
            if (!configuration.members().contains(memberState.getMember())) {
                this.members.remove(memberState.getMember());
                this.remoteMembers.remove(i);
                for (List<MemberState> memberType : this.memberTypes.values()) {
                    memberType.remove(memberState);
                }
                this.membersMap.remove(memberState.getMember().id());
                this.addressMap.remove(memberState.getMember().address());
                this.leaveListeners.accept(memberState.getMember());
                continue;
            }
            ++i;
        }
        if (!configuration.members().contains(this.member)) {
            this.members.remove(this.member);
        }
        this.configuration = configuration;
        if (this.context.getCommitIndex() >= configuration.index()) {
            this.context.getMetaStore().storeConfiguration(configuration);
        }
        this.reassign();
        return this;
    }

    private void reassign() {
        if (this.member.type() == Member.Type.ACTIVE && !this.member.equals(this.context.getLeader())) {
            int index = 1;
            for (MemberState member : this.getActiveMemberStates((m1, m2) -> m1.getMember().id() - m2.getMember().id())) {
                if (member.getMember().equals(this.context.getLeader())) continue;
                if (this.member.id() >= member.getMember().id()) break;
                ++index;
            }
            List<MemberState> sortedPassiveMembers = this.getPassiveMemberStates((m1, m2) -> m1.getMember().id() - m2.getMember().id());
            this.assignedMembers = this.assignMembers(index, sortedPassiveMembers);
        } else {
            this.assignedMembers = new ArrayList<MemberState>(0);
        }
    }

    private List<MemberState> assignMembers(int index, List<MemberState> sortedMembers) {
        ArrayList<MemberState> members = new ArrayList<MemberState>(sortedMembers.size());
        for (int i = 0; i < sortedMembers.size(); ++i) {
            if ((i + 1) % index != 0) continue;
            members.add(sortedMembers.get(i));
        }
        return members;
    }

    @Override
    public void close() {
        for (MemberState member : this.remoteMembers) {
            member.getMember().close();
        }
        this.member.close();
        this.cancelJoinTimer();
    }
}

