/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.server.common;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import org.apache.kafka.common.utils.Utils;

public class CheckpointFile<T> {
    private final int version;
    private final EntryFormatter<T> formatter;
    private final Object lock = new Object();
    private final Path absolutePath;
    private final Path tempPath;

    public CheckpointFile(File file, int version, EntryFormatter<T> formatter) throws IOException {
        this.version = version;
        this.formatter = formatter;
        try {
            Files.createFile(file.toPath(), new FileAttribute[0]);
        }
        catch (FileAlreadyExistsException fileAlreadyExistsException) {
            // empty catch block
        }
        this.absolutePath = file.toPath().toAbsolutePath();
        this.tempPath = Paths.get(this.absolutePath + ".tmp", new String[0]);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void write(Collection<T> entries) throws IOException {
        Object object = this.lock;
        synchronized (object) {
            try (FileOutputStream fileOutputStream = new FileOutputStream(this.tempPath.toFile());
                 BufferedWriter writer = new BufferedWriter(new OutputStreamWriter((OutputStream)fileOutputStream, StandardCharsets.UTF_8));){
                CheckpointWriteBuffer<T> checkpointWriteBuffer = new CheckpointWriteBuffer<T>(writer, this.version, this.formatter);
                checkpointWriteBuffer.write(entries);
                writer.flush();
                fileOutputStream.getFD().sync();
            }
            Utils.atomicMoveWithFallback(this.tempPath, this.absolutePath);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public List<T> read() throws IOException {
        Object object = this.lock;
        synchronized (object) {
            try (BufferedReader reader = Files.newBufferedReader(this.absolutePath);){
                CheckpointReadBuffer<T> checkpointBuffer = new CheckpointReadBuffer<T>(this.absolutePath.toString(), reader, this.version, this.formatter);
                List<T> list = checkpointBuffer.read();
                return list;
            }
        }
    }

    public static interface EntryFormatter<T> {
        public String toString(T var1);

        public Optional<T> fromString(String var1);
    }

    public static class CheckpointReadBuffer<T> {
        private final String location;
        private final BufferedReader reader;
        private final int version;
        private final EntryFormatter<T> formatter;

        public CheckpointReadBuffer(String location, BufferedReader reader, int version, EntryFormatter<T> formatter) {
            this.location = location;
            this.reader = reader;
            this.version = version;
            this.formatter = formatter;
        }

        public List<T> read() throws IOException {
            String line = this.reader.readLine();
            if (line == null) {
                return Collections.emptyList();
            }
            int readVersion = this.toInt(line);
            if (readVersion != this.version) {
                throw new IOException("Unrecognised version:" + readVersion + ", expected version: " + this.version + " in checkpoint file at: " + this.location);
            }
            line = this.reader.readLine();
            if (line == null) {
                return Collections.emptyList();
            }
            int expectedSize = this.toInt(line);
            ArrayList<T> entries = new ArrayList<T>(expectedSize);
            line = this.reader.readLine();
            while (line != null) {
                Optional<T> maybeEntry = this.formatter.fromString(line);
                if (!maybeEntry.isPresent()) {
                    throw this.buildMalformedLineException(line);
                }
                entries.add(maybeEntry.get());
                line = this.reader.readLine();
            }
            if (entries.size() != expectedSize) {
                throw new IOException("Expected [" + expectedSize + "] entries in checkpoint file [" + this.location + "], but found only [" + entries.size() + "]");
            }
            return entries;
        }

        private int toInt(String line) throws IOException {
            try {
                return Integer.parseInt(line);
            }
            catch (NumberFormatException e) {
                throw this.buildMalformedLineException(line);
            }
        }

        private IOException buildMalformedLineException(String line) {
            return new IOException(String.format("Malformed line in checkpoint file [%s]: %s", this.location, line));
        }
    }

    public static class CheckpointWriteBuffer<T> {
        private final BufferedWriter writer;
        private final int version;
        private final EntryFormatter<T> formatter;

        public CheckpointWriteBuffer(BufferedWriter writer, int version, EntryFormatter<T> formatter) {
            this.writer = writer;
            this.version = version;
            this.formatter = formatter;
        }

        public void write(Collection<T> entries) throws IOException {
            this.writer.write(Integer.toString(this.version));
            this.writer.newLine();
            this.writer.write(Integer.toString(entries.size()));
            this.writer.newLine();
            for (T entry : entries) {
                this.writer.write(this.formatter.toString(entry));
                this.writer.newLine();
            }
        }
    }
}

