package tech.harmonysoft.oss.micrometer.registry

import io.micrometer.core.instrument.Meter
import io.micrometer.core.instrument.MeterRegistry
import io.micrometer.core.instrument.Tag
import tech.harmonysoft.oss.common.execution.ExecutionContextManager
import tech.harmonysoft.oss.micrometer.util.StatsValue
import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.atomic.AtomicReference

/**
 * Facilitates standard Micrometer [Meter] usage when we need to use them with parameters.
 * For example, there might be common 'processingTime' timer and we want to use it for `action=action1`,
 * `action=action2`, etc.
 *
 * Also it has a feature that context data for the target tag names is automatically retrieved from
 * [ExecutionContextManager] and attached to data points generated by this registry's meters.
 *
 * **Note:** we introduce multiple overloaded [getMeter] methods here (`getMeter(tag)`, `getMeter(tag1, tag2)`, etc)
 * instead of providing generic varargs-based processing because stats collection is expected to be performed
 * quite often. Hence, we want to avoid new array construction every time [getMeter] is called.
 *
 * @param tagNames                  list of the tag names to use for the target meter. Note that the order is
 *                                  important - it must be the same as the order of values in the correspondnig
 *                                  [getMeter] call
 * @param staticTags                tags to be applied to every data point collected by this registry's meter
 * @param registry                  meter registry to use
 * @param executionContextManager   context values holder
 * @param contextTagNames           context tag names, their values are picked from [ExecutionContextManager.getFromCurrentContext]
 *                                  and are attached to data points produced by this registry's meters
 * @param meterCreator          meter's initializer
 */
class StatsRegistry<T : Meter>(
    private val tagNames: List<String>,
    private val staticTags: Set<Tag>,
    private val registry: MeterRegistry,
    private val executionContextManager: ExecutionContextManager,

    contextTagNames: Set<String>,
    meterCreator: (MeterRegistry, Iterable<Tag>) -> T
) {

    // we use array here in order to avoid creating Iterator objects during iteration (that might happen often)
    private val _contextTagNames = contextTagNames.toTypedArray()

    private val cacheEntryCreator: (Iterable<Tag>) -> T = {
        meterCreator(registry, it)
    }

    private val cache = CacheEntry(staticTags.toList()) {
        cacheEntryCreator(staticTags)
    }

    fun getMeter(): T {
        if (tagNames.isNotEmpty()) {
            throw IllegalArgumentException(
                "expected to get ${tagNames.size} tag value(s) for tag name(s) $tagNames but got nothing"
            )
        }
        val entry = _contextTagNames.fold(cache) { cacheEntry, tagName ->
            val value = executionContextManager.getFromCurrentContext<Any>(tagName) ?: StatsValue.NO_VALUE
            cacheEntry.getFor(tagName, value, cacheEntryCreator)
        }

        return entry.meter
    }

    fun getMeter(tagValue: Any): T {
        if (tagNames.size != 1) {
            throw IllegalArgumentException(
                "expected to get ${tagNames.size} tag value(s) for tag name(s) $tagNames but got one: '$tagValue'"
            )
        }
        val entry = _contextTagNames.fold(cache) { cacheEntry, tagName ->
            val value = when (tagName) {
                tagNames[0] -> tagValue
                else -> executionContextManager.getFromCurrentContext<Any>(tagName) ?: StatsValue.NO_VALUE
            }
            cacheEntry.getFor(tagName, value, cacheEntryCreator)
        }.getFor(tagNames[0], tagValue, cacheEntryCreator)

        return entry.meter
    }

    fun getMeter(tag1Value: Any, tag2value: Any): T {
        if (tagNames.size != 2) {
            throw IllegalArgumentException(
                "expected to get ${tagNames.size} tag value(s) for tag name(s) $tagNames but gut two: '$tag1Value' "
                + "and '$tag2value'"
            )
        }

        val entry = _contextTagNames.fold(cache) { cacheEntry, tagName ->
            val value = when (tagName) {
                tagNames[0] -> tag1Value
                tagNames[1] -> tag2value
                else -> executionContextManager.getFromCurrentContext<Any>(tagName) ?: StatsValue.NO_VALUE
            }
            cacheEntry.getFor(tagName, value, cacheEntryCreator)
        }.getFor(tagNames[0], tag1Value, cacheEntryCreator).getFor(tagNames[1], tag2value, cacheEntryCreator)

        return entry.meter
    }

    fun getMeter(tag1Value: Any, tag2value: Any, tag3value: Any): T {
        if (tagNames.size != 3) {
            throw IllegalArgumentException(
                "expected to get ${tagNames.size} tag value(s) for tag name(s) $tagNames but gut three: "
                + "'$tag1Value', '$tag2value' and '$tag3value'"
            )
        }

        val entry = _contextTagNames.fold(cache) { cacheEntry, tagName ->
            val value = when (tagName) {
                tagNames[0] -> tag1Value
                tagNames[1] -> tag2value
                tagNames[2] -> tag3value
                else -> executionContextManager.getFromCurrentContext<Any>(tagName) ?: StatsValue.NO_VALUE
            }
            cacheEntry.getFor(tagName, value, cacheEntryCreator)
        }.getFor(tagNames[0], tag1Value, cacheEntryCreator)
            .getFor(tagNames[1], tag2value, cacheEntryCreator)
            .getFor(tagNames[2], tag3value, cacheEntryCreator)

        return entry.meter
    }

    fun getMeter(tag1Value: Any, tag2value: Any, tag3value: Any, tag4value: Any): T {
        if (tagNames.size != 4) {
            throw IllegalArgumentException(
                "expected to get ${tagNames.size} tag value(s) for tag name(s) $tagNames but gut four: "
                + "'$tag1Value', '$tag2value', '$tag3value' and '$tag4value'"
            )
        }

        val entry = _contextTagNames.fold(cache) { cacheEntry, tagName ->
            val value = when (tagName) {
                tagNames[0] -> tag1Value
                tagNames[1] -> tag2value
                tagNames[2] -> tag3value
                tagNames[3] -> tag4value
                else -> executionContextManager.getFromCurrentContext<Any>(tagName) ?: StatsValue.NO_VALUE
            }
            cacheEntry.getFor(tagName, value, cacheEntryCreator)
        }.getFor(tagNames[0], tag1Value, cacheEntryCreator)
            .getFor(tagNames[1], tag2value, cacheEntryCreator)
            .getFor(tagNames[2], tag3value, cacheEntryCreator)
            .getFor(tagNames[3], tag4value, cacheEntryCreator)

        return entry.meter
    }

    private class CacheEntry<T : Meter>(
        private val tags: List<Tag>,
        private val meterCreator: () -> T
    ) {
        private val meterRef = AtomicReference<T>()
        private val tagNames = tags.map { it.key }.toSet()
        private val childrenByValue = ConcurrentHashMap<Any, CacheEntry<T>>()

        // we use lazy initialization here because an instance with empty custom tags is always created by
        // micrometer infrastructure and StatsRegistry is created. That leads to an empty tag value which
        // litters the stats
        val meter: T
            get() {
                return meterRef.get() ?: run {
                    meterRef.compareAndSet(null, meterCreator())
                    meterRef.get()
                }
            }

        fun getFor(tagName: String, tagValue: Any, meterCreator: (Iterable<Tag>) -> T): CacheEntry<T> {
            return if (tagNames.contains(tagName)) {
                this
            } else {
                childrenByValue[tagValue] ?: childrenByValue.computeIfAbsent(tagValue) {
                    val childTags = tags + Tag.of(tagName, tagValue.toString())
                    CacheEntry(childTags) { meterCreator(childTags) }
                }
            }
        }
    }
}