/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.lambda.powertools.idempotency.internal;

import com.amazonaws.services.lambda.runtime.Context;
import com.fasterxml.jackson.databind.JsonNode;
import java.time.Instant;
import java.util.OptionalInt;
import java.util.function.BiFunction;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.lambda.powertools.idempotency.Idempotency;
import software.amazon.lambda.powertools.idempotency.exceptions.IdempotencyAlreadyInProgressException;
import software.amazon.lambda.powertools.idempotency.exceptions.IdempotencyInconsistentStateException;
import software.amazon.lambda.powertools.idempotency.exceptions.IdempotencyItemAlreadyExistsException;
import software.amazon.lambda.powertools.idempotency.exceptions.IdempotencyItemNotFoundException;
import software.amazon.lambda.powertools.idempotency.exceptions.IdempotencyKeyException;
import software.amazon.lambda.powertools.idempotency.exceptions.IdempotencyPersistenceLayerException;
import software.amazon.lambda.powertools.idempotency.exceptions.IdempotencyValidationException;
import software.amazon.lambda.powertools.idempotency.persistence.BasePersistenceStore;
import software.amazon.lambda.powertools.idempotency.persistence.DataRecord;
import software.amazon.lambda.powertools.utilities.JsonConfig;

public class IdempotencyHandler {
    private static final Logger LOG = LoggerFactory.getLogger(IdempotencyHandler.class);
    private static final int MAX_RETRIES = 2;
    private final ProceedingJoinPoint pjp;
    private final JsonNode data;
    private final BasePersistenceStore persistenceStore;
    private final Context lambdaContext;

    public IdempotencyHandler(ProceedingJoinPoint pjp, String functionName, JsonNode payload, Context lambdaContext) {
        this.pjp = pjp;
        this.data = payload;
        this.lambdaContext = lambdaContext;
        this.persistenceStore = Idempotency.getInstance().getPersistenceStore();
        this.persistenceStore.configure(Idempotency.getInstance().getConfig(), functionName);
    }

    public Object handle() throws Throwable {
        int i = 0;
        while (true) {
            try {
                return this.processIdempotency();
            }
            catch (IdempotencyInconsistentStateException e) {
                if (i == 2) {
                    throw e;
                }
                ++i;
                continue;
            }
            break;
        }
    }

    private Object processIdempotency() throws Throwable {
        try {
            this.persistenceStore.saveInProgress(this.data, Instant.now(), this.getRemainingTimeInMillis());
        }
        catch (IdempotencyItemAlreadyExistsException iaee) {
            DataRecord dr = iaee.getDataRecord().orElseGet(this::getIdempotencyRecord);
            if (dr != null) {
                return this.handleForStatus(dr);
            }
        }
        catch (IdempotencyKeyException ike) {
            throw ike;
        }
        catch (Exception e) {
            throw new IdempotencyPersistenceLayerException("Failed to save in progress record to idempotency store. If you believe this is a Powertools for AWS Lambda (Java) bug, please open an issue.", e);
        }
        return this.getFunctionResponse();
    }

    private OptionalInt getRemainingTimeInMillis() {
        if (this.lambdaContext != null) {
            return OptionalInt.of(this.lambdaContext.getRemainingTimeInMillis());
        }
        LOG.warn("Couldn't determine the remaining time left. Did you call registerLambdaContext on Idempotency?");
        return OptionalInt.empty();
    }

    private DataRecord getIdempotencyRecord() {
        try {
            return this.persistenceStore.getRecord(this.data, Instant.now());
        }
        catch (IdempotencyItemNotFoundException e) {
            LOG.debug("An existing idempotency record was deleted before we could fetch it");
            throw new IdempotencyInconsistentStateException("saveInProgress and getRecord return inconsistent results", e);
        }
        catch (IdempotencyKeyException | IdempotencyValidationException vke) {
            throw vke;
        }
        catch (Exception e) {
            throw new IdempotencyPersistenceLayerException("Failed to get record from idempotency store. If you believe this is a Powertools for AWS Lambda (Java) bug, please open an issue.", e);
        }
    }

    private Object handleForStatus(DataRecord record) {
        if (DataRecord.Status.EXPIRED.equals((Object)record.getStatus())) {
            throw new IdempotencyInconsistentStateException("saveInProgress and getRecord return inconsistent results");
        }
        if (DataRecord.Status.INPROGRESS.equals((Object)record.getStatus())) {
            if (record.getInProgressExpiryTimestamp().isPresent() && record.getInProgressExpiryTimestamp().getAsLong() < Instant.now().toEpochMilli()) {
                throw new IdempotencyInconsistentStateException("Item should have been expired in-progress because it already time-outed.");
            }
            throw new IdempotencyAlreadyInProgressException("Execution already in progress with idempotency key: " + record.getIdempotencyKey());
        }
        Class returnType = ((MethodSignature)this.pjp.getSignature()).getReturnType();
        try {
            LOG.debug("Response for key '{}' retrieved from idempotency store, skipping the function", (Object)record.getIdempotencyKey());
            BiFunction<Object, DataRecord, Object> responseHook = Idempotency.getInstance().getConfig().getResponseHook();
            Object responseData = returnType.equals(String.class) ? record.getResponseData() : JsonConfig.get().getObjectMapper().reader().readValue(record.getResponseData(), returnType);
            if (responseHook != null) {
                LOG.debug("Applying user-defined response hook to idempotency data before returning.");
                return responseHook.apply(responseData, record);
            }
            return responseData;
        }
        catch (Exception e) {
            throw new IdempotencyPersistenceLayerException("Unable to get function response as " + returnType.getSimpleName(), e);
        }
    }

    private Object getFunctionResponse() throws Throwable {
        Object response;
        try {
            response = this.pjp.proceed(this.pjp.getArgs());
        }
        catch (Throwable handlerException) {
            try {
                this.persistenceStore.deleteRecord(this.data, handlerException);
            }
            catch (IdempotencyKeyException ke) {
                throw ke;
            }
            catch (Exception e) {
                throw new IdempotencyPersistenceLayerException("Failed to delete record from idempotency store. If you believe this is a Powertools for AWS Lambda (Java) bug, please open an issue.", e);
            }
            throw handlerException;
        }
        try {
            this.persistenceStore.saveSuccess(this.data, response, Instant.now());
        }
        catch (Exception e) {
            throw new IdempotencyPersistenceLayerException("Failed to update record state to success in idempotency store. If you believe this is a Powertools for AWS Lambda (Java) bug, please open an issue.", e);
        }
        return response;
    }
}

