package com.particle.mpc

import com.particle.base.utils.HexUtils
import com.particle.mpc.data.AuthParam
import com.particle.mpc.data.EdDsaData
import com.particle.mpc.data.EdDsaDataBackup
import com.particle.mpc.data.EdDsaSign
import com.particle.mpc.data.KeyAgg
import com.particle.mpc.utils.JSON
import com.particle.mpc.utils.ServerErrorHandler
import com.particle.mpc.utils.parse
import com.particle.mpc.utils.stringify
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext

/**
 * Created by chaichuanfa on 2023/5/11
 */
class EdDsa(private val id: String, private val keyAgg: String, private val keyPair: String) {

    companion object {
        suspend fun gen(endpoint: String, authParam: AuthParam): EdDsa =
            withContext(Dispatchers.IO) {
                val key = MPCNative.eddsaKeygen(endpoint, JSON.stringify(authParam))
                ServerErrorHandler.checkAndThrow(key)
                val genData = JSON.parse<EdDsaData>(key)
                EdDsa(genData.id, genData.keyAgg, genData.keyPair)
            }

        fun from(key: String): EdDsa {
            val data = JSON.parse<EdDsaDataBackup>(key)
            return EdDsa(data.id, JSON.stringify(data.keyAgg), JSON.stringify(data.keyPair))
        }
    }

    /**
     * @return p2 key
     */
    fun to(): String {
        return JSON.stringify(EdDsaDataBackup(id, JSON.parse(keyAgg), JSON.parse(keyPair)))
    }

    fun pub(): ByteArray {
        val keyAgg = JSON.parse<KeyAgg>(keyAgg)
        return HexUtils.decode(keyAgg.apk.bytesStr)
    }

    suspend fun sign(
        endpoint: String, authParams: AuthParam, message: String
    ): EdDsaSign = withContext(Dispatchers.IO) {
        val key = JSON.stringify(EdDsaData(id, keyAgg, keyPair))
        val result = MPCNative.eddsaSign(endpoint, JSON.stringify(authParams), message, key)
        ServerErrorHandler.checkAndThrow(result)
        JSON.parse(result)
    }
}