package com.particle.mpc

import android.util.Log
import com.particle.base.utils.HexUtils
import com.particle.mpc.data.AuthParam
import com.particle.mpc.data.EcdsaData
import com.particle.mpc.data.EcdsaDataBackup
import com.particle.mpc.data.Signature
import com.particle.mpc.utils.JSON
import com.particle.mpc.utils.ServerErrorHandler
import com.particle.mpc.utils.parse
import com.particle.mpc.utils.stringify
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext

/**
 * Created by chaichuanfa on 2023/5/11
 * evm
 */
class Ecdsa(private val id: String, private val masterKey: String, private val childKey: String) {

    companion object {
        suspend fun gen(endpoint: String, authParam: AuthParam): Ecdsa =
            withContext(Dispatchers.IO) {
                val key = MPCNative.ecdsaKeygen(endpoint, JSON.stringify(authParam))
                ServerErrorHandler.checkAndThrow(key)
                val genData = JSON.parse<EcdsaData>(key)
                Ecdsa(
                    genData.id,
                    genData.masterKey,
                    genData.childKey,
                )
            }

        /**
         * @param key p2 key
         */
        fun from(key: String): Ecdsa {
            val data = JSON.parse<EcdsaDataBackup>(key)
            return Ecdsa(data.id, JSON.stringify(data.masterKey), JSON.stringify(data.childKey))
        }
    }

    fun to(): String {
        return JSON.stringify(
            EcdsaDataBackup(
                id,
                JSON.parse(masterKey),
                JSON.parse(childKey)
            )
        )
    }

    suspend fun pub(): ByteArray =
        withContext(Dispatchers.IO) {
            val result = MPCNative.ecdsaPub(getKey())
            HexUtils.decode(result)
        }

    suspend fun sign(
        endpoint: String,
        authParam: AuthParam,
        message: String,
        uniq: Boolean = false,
    ): Signature =
        withContext(Dispatchers.IO) {
            val result = MPCNative.ecdsaSign(
                endpoint,
                JSON.stringify(authParam),
                message,
                getKey(),
                uniq.toString()
            )
            ServerErrorHandler.checkAndThrow(result)
            JSON.parse(result)
        }

    suspend fun signTx(endpoint: String, authParam: AuthParam, txData: String): String =
        withContext(Dispatchers.IO) {
            val key = getKey()
            Log.i("Ecdsa", "key:$key")
            val result =
                MPCNative.ecdsaSignTx(endpoint, JSON.stringify(authParam), txData, key)
            ServerErrorHandler.checkAndThrow(result)
            result
        }

    suspend fun refresh(endpoint: String, authParam: AuthParam): Ecdsa =
        withContext(Dispatchers.IO) {
            val key = getKey()
            val newKey = MPCNative.ecdsaRotate(endpoint, JSON.stringify(authParam), key)
            ServerErrorHandler.checkAndThrow(newKey)
            val genData = JSON.parse<EcdsaData>(newKey)
            Ecdsa(genData.id, genData.masterKey, genData.childKey)
        }

    private fun getKey(): String = JSON.stringify(EcdsaData(id, masterKey, childKey))
}