package com.particle.api.service

import android.text.TextUtils
import androidx.annotation.Keep
import com.google.gson.JsonObject
import com.particle.api.BuildConfig
import com.particle.api.infrastructure.db.table.SplToken
import com.particle.api.infrastructure.db.table.TransInfo
import com.particle.api.infrastructure.net.CommonApi
import com.particle.api.infrastructure.net.SolanaRpcApi
import com.particle.api.infrastructure.net.data.*
import com.particle.api.infrastructure.net.data.Encoding
import com.particle.api.infrastructure.net.data.RequestConfiguration
import com.particle.api.infrastructure.net.data.req.*
import com.particle.api.infrastructure.net.data.resp.*
import com.particle.api.logic.TokenNFTListLogic
import com.particle.api.service.data.TokenNFTResult
import com.particle.api.solana
import com.particle.base.ParticleNetwork
import network.particle.chains.ChainInfo
import okhttp3.ResponseBody
import org.p2p.solanaj.model.types.*
import org.p2p.solanaj.programs.TokenProgram
import java.math.BigInteger


import java.util.*

@Keep
class SolanaService(private val solanaApi: SolanaRpcApi, private val commonApi: CommonApi) :
    ChainService {

    override suspend fun getTokenList(chainInfo: ChainInfo?): List<SplToken> {
        return try {
            val tokenUrl = "${BuildConfig.PN_SOLANA_TOKEN_LIST}${ParticleNetwork.chainName}.json"
            val splTokens = commonApi.getTokenList(tokenUrl)

            DBService.splTokenDao.deleteByChainName(
                ParticleNetwork.chainName
            )

            SplTokenService.insertSplToken(splTokens, chainInfo)
            DBService.splTokenDao.getTokenList(
                ParticleNetwork.chainName, ParticleNetwork.chainId
            )
        } catch (e: Exception) {
            DBService.splTokenDao.getTokenList(
                ParticleNetwork.chainName, ParticleNetwork.chainId
            )
        }

    }

    override suspend fun rpc(method: String, params: List<Any>?): ResponseBody {
        return solanaApi.rpc(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                method,
                params,
            )
        )
    }

    /**
     * Get the real-time exchange rate of the EVM token
     */
    suspend fun getPrice(
        addresses: List<String>,
        currencies: List<String>,
    ): RpcOutput<List<RateResult>> {
        return solanaApi.enhancedGetPrice(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.enhancedGetPrice.value,
                arrayListOf(addresses, currencies)
            )
        )
    }

    /**
     * Get token list and NFT list from server by giving an address
     */
    override suspend fun getTokensAndNFTs(address: String): TokenNFTResult {
        TokenNFTListLogic.updateTokenNFTInfo(address, true)
        val tokenList = DBService.tokenInfoDao.getAllTokenInfo(
            address, ParticleNetwork.chainName, ParticleNetwork.chainId
        )
        val nftList =
            DBService.nftInfoDao.getAll(ParticleNetwork.chainName, ParticleNetwork.chainId)
        return TokenNFTResult(tokenList, nftList)
    }

    /**
     * Get token list and NFT list from database by giving an address
     */
    override suspend fun getTokensAndNFTsFromDB(address: String): TokenNFTResult {
        val tokenList = DBService.tokenInfoDao.getAllTokenInfo(
            address, ParticleNetwork.chainName, ParticleNetwork.chainId
        )
        val nftList =
            DBService.nftInfoDao.getAll(ParticleNetwork.chainName, ParticleNetwork.chainId)
        return TokenNFTResult(tokenList, nftList)
    }

    /**
     * Get parsed transaction history from server by giving an address
     */
    suspend fun getTransactionsByAddress(
        address: String,
        transBody: TransOptBody,
    ): List<TransInfo> {
        fetchTransData2Db(address, transBody)
        return DBService.transDao.getAllSolTrans(
            address, ParticleNetwork.chainName, ParticleNetwork.chainId
        )
    }

    /**
     * Get parsed token transaction history from server by giving an address
     */
    suspend fun getTokenTransactionsByAddress(
        address: String,
        mint: String,
        transBody: TransOptBody,
    ): List<TransInfo> {
        fetchTransData2Db(address, transBody, mint)
        return DBService.transDao.getAllTokenTrans(
            address, ParticleNetwork.chainName, ParticleNetwork.chainId, mint
        )
    }

    /**
     * Get parsed transaction history from database by giving an address
     */
    suspend fun getTokenTransactionsByAddressFromDB(
        address: String,
        mint: String,
    ): List<TransInfo> {
        return DBService.transDao.getAllTokenTrans(
            address, ParticleNetwork.chainName, ParticleNetwork.chainId, mint
        )
    }

    /**
     * Get parsed transaction history from database by giving an address
     */
    suspend fun getTransactionsByAddressFromDB(address: String): List<TransInfo> {
        return DBService.transDao.getAllSolTrans(
            address, ParticleNetwork.chainName, ParticleNetwork.chainId
        )
    }

    suspend fun serializeTransaction(solTransReq: SerializeSOLTransReq): RpcOutput<SerializeTransResult> {
        return solanaApi.enhancedSerializeTransaction(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.enhancedSerializeTransaction.value,
                arrayListOf(SerializeTransactionParams.transferSol.value, solTransReq)
            )
        )
    }

    suspend fun serializeTransaction(tokenTransReq: SerializeTokenTransReq): RpcOutput<SerializeTransResult> {
        return solanaApi.enhancedSerializeTransaction(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.enhancedSerializeTransaction.value,
                arrayListOf(SerializeTransactionParams.transferToken.value, tokenTransReq)
            )
        )
    }

    /**
     * search mint trans if not mint empty
     * search native trans if mint empty
     */
    private suspend fun fetchTransData2Db(
        address: String, transBody: TransOptBody, mint: String = "",
    ) {
        var response: RpcOutput<List<TransResult>>
        if (TextUtils.isEmpty(mint)) {
            response = ParticleNetwork.solana.getTransactionsByAddressFromServer(
                address, transBody
            )
        } else {
            response = ParticleNetwork.solana.getTokenTransactionsFromServer(
                address, mint, transBody
            )
        }
        val transResult = response.result
        val allTrans = arrayListOf<TransInfo>()
        for (trans in transResult) {
            var from: String? = ""
            var to: String? = ""
            trans.data.let {
                from = it.sender
                to = it.receiver
            }
            if (!TextUtils.isEmpty(mint)) {
                if (trans.type != "transfer-token")
                    continue
            }
            allTrans.add(
                TransInfo(
                    address,
                    ParticleNetwork.chainName,
                    ParticleNetwork.chainId,
                    trans.type,
                    trans.lamportsChange,
                    trans.lamportsFee,
                    trans.signature,
                    trans.blockTime,
                    trans.status,
                    trans.data,
                    mint,
                    from,
                    to,
                    trans.data.amountTransfered,
                    trans.data.lamportsTransfered
                )
            )

        }
        if (allTrans.count() > 0) {
            DBService.transDao.insertAll(allTrans)
        }
    }

    suspend fun getTokensAndNFTsFromServer(
        address: String,
        chainInfo: ChainInfo? = null
    ): RpcOutput<TokenResult> {
        return solanaApi.enhancedGetTokensAndNFTs(
            ReqBody(
                chainInfo?.id ?: ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.enhancedGetTokensAndNFTs.toString(),
                arrayListOf(address, mutableMapOf(Pair("parseMetadataUri", true)))
            )
        )
    }

    /**
     * get native trans
     */
    private suspend fun getTransactionsByAddressFromServer(
        address: String,
        body: TransOptBody,
    ): RpcOutput<List<TransResult>> {
        return solanaApi.enhancedGetTransactionsByAddress(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.enhancedGetTransactionsByAddress.toString(),
                arrayListOf(address, body)
            )
        )
    }

    /**
     * get token trans
     */
    private suspend fun getTokenTransactionsFromServer(
        address: String,
        mint: String,
        body: TransOptBody?,
    ): RpcOutput<List<TransResult>> {

        val map = mutableMapOf<String, String>()
        map["address"] = address
        map["mint"] = mint
        val paramsList = if (body != null) arrayListOf(map, body) else arrayListOf(address)
        return solanaApi.enhancedGetTransactionsByAddress(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.enhancedGetTokenTransactionsByAddress.toString(),
                paramsList
            )
        )
    }

    /**
     *  get token info by address
     */
    suspend fun getTokensByTokenAddresses(
        pubKey: String, tokenAddresses: List<String>,
    ): RpcOutput<List<CustomTokenResult>> {
        return solanaApi.enhancedGetTokensByTokenAddresses(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.enhancedGetTokensByTokenAddresses.value,
                listOf(pubKey, tokenAddresses)
            )
        )
    }

    suspend fun addCustomTokens(address: String, tokenAddresses: List<String>): List<String> {
        return TokenNFTListLogic.addCustomToken(address, tokenAddresses)
    }


    /**
     * @param transaction : fully-signed Transaction, as encoded string
     * @param preflightCommitment: (optional) Commitment level to use for preflight (default: "finalized").
     * @param  encoding: (optional) Encoding used for the transaction data. Either "base58" (slow, DEPRECATED), or "base64". (default: "base58").
     */
    suspend fun sendTransaction(
        transaction: String, encoding: String = "base64", preflightCommitment: String = "confirmed",
    ): String {
        val map = mutableMapOf<String, String>()
        map["encoding"] = encoding
        map["preflightCommitment"] = preflightCommitment
        return solanaApi.sendTransactions(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.sendTransaction.value,
                listOf(transaction, map)
            )
        ).result
    }

    /**
     * @param transaction: transaction signature as base-58 encoded string
     */
    suspend fun checkTransactionConfirmed(
        transaction: String, commitment: String = "confirmed",
    ): Boolean {
        val map = mutableMapOf<String, String>()
        map["commitment"] = commitment
        val rs = solanaApi.getTransactionRespAny(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getTransaction.value,
                listOf(transaction, map)
            )
        ).result
        return rs != null
    }

    /**
     * Returns all information associated with the account of provided Pubkey
     * @param address
     * @return
     */
    suspend fun getAccountInfo(account: String): AccountInfo? {
        return solanaApi.getAccountInfo(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                method = ReqBodyMethod.getAccountInfo.value,
                params = listOf(
                    account, RequestConfiguration(encoding = Encoding.BASE64.encoding)
                )
            )
        ).result
    }

    suspend fun getAccountInfo(
        account: String, requestConfiguration: RequestConfiguration? = null,
    ): AccountInfo? {
        return solanaApi.getAccountInfo(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                method = ReqBodyMethod.getAccountInfo.value,
                params = listOf(
                    account, requestConfiguration
                )
            )
        ).result
    }


    suspend fun getMinimumBalanceForRentExemption(dataLength: Long): RpcOutput<Long> {
        return solanaApi.getMinimumBalanceForRentExemption(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                method = "getMinimumBalanceForRentExemption",
                params = listOf(dataLength)
            )
        )
    }

    //getFees
    suspend fun getFees(): RpcOutput<FeesResponse> {
        return solanaApi.getFees(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                method = "getFees",
            )
        )
    }

    suspend fun getTokenAccountsByOwner(owner: String): RpcOutput<TokenAccounts> {
        val programId = TokenProgram.PROGRAM_ID
        val programIdParam = HashMap<String, String>()
        programIdParam["programId"] = programId.toBase58()

        val encoding = HashMap<String, String>()
        encoding["encoding"] = "jsonParsed"
        encoding["commitment"] = "confirmed"
        val params = listOf(
            owner, programIdParam, encoding
        )
        return solanaApi.getTokenAccountsByOwner(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                method = "getTokenAccountsByOwner",
                params = params,
            )
        )
    }

    suspend fun getRecentBlockHash(commitment: String = "finalized"): String {
        val response = solanaApi.getRecentBlockHash(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                method = "getRecentBlockhash",
                params = listOf(ConfigObjects.Commitment(commitment))
            )
        )
        return response.result.recentBlockhash
    }

    suspend fun getQuote(
        pubKey: String, inputMint: String, outputMint: String, amount: BigInteger,
    ): RpcOutput<List<QuoteInfoSOL>> {
        val map = mutableMapOf<String, String>()
        map["inputMint"] = inputMint
        map["outputMint"] = outputMint
        map["amount"] = amount.toString()
        return solanaApi.swapGetQuote(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.swapGetQuote.value,
                listOf(pubKey, map)
            )
        )
    }

    suspend fun getQuoteJSONList(
        pubKey: String, inputMint: String, outputMint: String, amount: BigInteger,
    ): RpcOutput<List<JsonObject>> {
        val map = mutableMapOf<String, String>()
        map["inputMint"] = inputMint
        map["outputMint"] = outputMint
        map["amount"] = amount.toString()
        return solanaApi.swapGetQuoteJSONList(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.swapGetQuote.value,
                listOf(pubKey, map)
            )
        )
    }

    suspend fun swapGetTransactions(pubKey: String, quoteInfoSOL: QuoteInfoSOL): List<QuoteTrans> {
        return solanaApi.swapGetTransactions(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.swapGetTransactions.value,
                listOf(pubKey, quoteInfoSOL)
            )
        ).result
    }

    suspend fun swapGetTransactions(pubKey: String, json: JsonObject): List<QuoteTrans> {
        return solanaApi.swapGetTransactions(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.swapGetTransactions.value,
                listOf(pubKey, json)
            )
        ).result
    }

    suspend fun getBalance(pubKey: String): SOLBalanceInfo {
        val map = mutableMapOf<String, String>()
        map["commitment"] = "confirmed"
        return solanaApi.getBalance(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getBalance.value,
                listOf(pubKey, map)
            )
        ).result
    }

    /**
     * Returns the current block height of the node
     *
     */
    suspend fun getBlockHeight(
        commitment: String? = null, minContextSlot: Int? = null,
    ): BigInteger {
        return solanaApi.rpcRespBigInteger(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getBlockHeight.value,
                listOf(commitment, minContextSlot)
            )
        ).result
    }

    suspend fun getBlockProduction(req: BlockProductionReq): BlockProductionResp {
        return solanaApi.getBlockProduction(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getBlockProduction.value,
                listOf(req)
            )
        ).result
    }

    suspend fun getBlockCommitment(blockId: BigInteger): BlockCommitmentResp {
        return solanaApi.getBlockCommitment(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getBlockCommitment.value,
                listOf(blockId)
            )
        ).result
    }

    suspend fun getBlocks(
        startSlot: BigInteger, endSlot: BigInteger? = null, commitmentInfo: CommitmentInfo? = null,
    ): List<BigInteger> {
        return solanaApi.rpcRespListBigInteger(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getBlocks.value,
                listOf(startSlot, endSlot, commitmentInfo)
            )
        ).result
    }

    suspend fun getBlocksWithLimit(
        startSlot: BigInteger, limit: BigInteger, commitmentInfo: CommitmentInfo?,
    ): List<BigInteger> {
        return solanaApi.rpcRespListBigInteger(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getBlocksWithLimit.value,
                listOf(startSlot, limit, commitmentInfo)
            )
        ).result
    }

    suspend fun getBlockTime(block: BigInteger): BigInteger? {
        return solanaApi.rpcRespNullableBigInteger(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getBlockTime.value,
                listOf(block)
            )
        ).result
    }

    suspend fun getClusterNodes(): List<ClusterNodesResp> {
        return solanaApi.getClusterNodes(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getClusterNodes.value,
                listOf()
            )
        ).result
    }

    suspend fun getEpochInfo(req: EpochInfoReq? = null): EpochInfoResp {
        return solanaApi.getEpochInfo(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getEpochInfo.value,
                listOf(req)
            )
        ).result
    }

    suspend fun getEpochSchedule(): EpochScheduleResp {
        return solanaApi.getEpochSchedule(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getEpochSchedule.value,
                listOf()
            )
        ).result
    }

    suspend fun getFeeForMessage(message: String, req: FeeForMessageReq? = null): List<Any> {
        return solanaApi.getFeeForMessage(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getFeeForMessage.value,
                listOf()
            )
        ).result
    }

    suspend fun getFirstAvailableBlock(): BigInteger {
        return solanaApi.rpcRespBigInteger(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getFirstAvailableBlock.value,
                listOf()
            )
        ).result
    }

    suspend fun getGenesisHash(): String {
        return solanaApi.rpcRespString(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getGenesisHash.value,
                listOf()
            )
        ).result
    }

    suspend fun getHealth(): HealthResp {
        return solanaApi.getHealth(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getHealth.value,
                listOf()
            )
        ).result
    }

    suspend fun getHighestSnapshotSlot(): HighestSnapshotSlotResp {
        return solanaApi.getHighestSnapshotSlot(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getHighestSnapshotSlot.value,
                listOf()
            )
        ).result
    }

    suspend fun getIdentity(): IdentityResp {
        return solanaApi.getIdentity(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getIdentity.value,
                listOf()
            )
        ).result
    }

    suspend fun getInflationGovernor(commitmentInfo: CommitmentInfo? = null): InflationGovernorResp {
        return solanaApi.getInflationGovernor(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getInflationGovernor.value,
                listOf(commitmentInfo)
            )
        ).result
    }

    suspend fun getInflationRate(): InflationRateResp {
        return solanaApi.getInflationRate(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getInflationRate.value,
                listOf()
            )
        ).result
    }

    suspend fun getInflationReward(
        addresses: List<String>, req: InflationRewardReq? = null,
    ): List<Any> {
        return solanaApi.getInflationReward(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getInflationReward.value,
                listOf(addresses, req)
            )
        ).result
    }

    suspend fun getLargestAccounts(req: LargestAccountsReq? = null): LargestAccountsResp {
        return solanaApi.getLargestAccounts(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getLargestAccounts.value,
                listOf(req)
            )
        ).result
    }

    suspend fun getLatestBlockhash(req: LatestBlockhashReq? = null): LatestBlockhashResp {
        return solanaApi.getLatestBlockhash(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getLatestBlockhash.value,
                listOf(req)
            )
        ).result
    }

    suspend fun getLeaderSchedule(
        leader: BigInteger? = null, req: LeaderScheduleReq? = null,
    ): Any {
        return solanaApi.rpcRespAny(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getLeaderSchedule.value,
                listOf(leader, req)
            )
        ).result
    }

    suspend fun getMaxRetransmitSlot(): BigInteger {
        return solanaApi.rpcRespBigInteger(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getMaxRetransmitSlot.value,
                listOf()
            )
        ).result
    }

    suspend fun getMaxShredInsertSlot(): BigInteger {
        return solanaApi.rpcRespBigInteger(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getMaxShredInsertSlot.value,
                listOf()
            )
        ).result
    }

    suspend fun getMinimumBalanceForRentExemption(): BigInteger {
        return solanaApi.rpcRespBigInteger(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getMinimumBalanceForRentExemption.value,
                listOf()
            )
        ).result
    }

    suspend fun getMultipleAccounts() {
        //todo:
    }

    suspend fun getProgramAccounts() {
        //todo:
    }

    suspend fun getRecentPerformanceSamples(limit: Int): List<PerformanceSampleResp> {
        return solanaApi.getRecentPerformanceSamples(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getRecentPerformanceSamples.value,
                listOf(limit)
            )
        ).result
    }

    suspend fun getSignaturesForAddress(
        address: String, req: SignaturesForAddressReq? = null,
    ): List<SignaturesForAddressResp> {
        return solanaApi.getSignaturesForAddress(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getSignaturesForAddress.value,
                listOf(address, req)
            )
        ).result
    }

    suspend fun getSignatureStatuses(
        signatures: List<String>, req: SignatureStatusesReq? = null,
    ): SignatureStatusesResp? {
        return solanaApi.getSignatureStatuses(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getSignatureStatuses.value,
                listOf(signatures, req)
            )
        ).result
    }

    suspend fun getSlot(req: SlotReq? = null): BigInteger {
        return solanaApi.rpcRespBigInteger(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getSlot.value,
                listOf(req)
            )
        ).result
    }

    suspend fun getSlotLeader(req: SlotReq? = null): String {
        return solanaApi.rpcRespString(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getSlotLeader.value,
                listOf(req)
            )
        ).result
    }

    suspend fun getSlotLeaders(startSlot: BigInteger, limit: Int): List<String> {
        return solanaApi.rpcRespListString(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getSlotLeaders.value,
                listOf(startSlot, limit)
            )
        ).result
    }

    suspend fun getStakeActivation(
        pubKey: String, req: StakeActivationReq? = null,
    ): StakeActivationResp {
        return solanaApi.getStakeActivation(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getStakeActivation.value,
                listOf(pubKey, req)
            )
        ).result
    }

    suspend fun getStakeMinimumDelegation(commitmentInfo: CommitmentInfo? = null): StakeMinimumDelegationResp {
        return solanaApi.getStakeMinimumDelegation(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getStakeMinimumDelegation.value,
                listOf(commitmentInfo)
            )
        ).result
    }

    suspend fun getSupply(req: SupplyReq? = null): SupplyResp {
        return solanaApi.getSupply(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getSupply.value,
                listOf(req)
            )
        ).result
    }

    suspend fun getTokenAccountBalance(
        pubKey: String, commitment: CommitmentInfo? = null,
    ): TokenAccountBalanceResp {
        return solanaApi.getTokenAccountBalance(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getTokenAccountBalance.value,
                listOf(pubKey, commitment)
            )
        ).result
    }


    suspend fun getTokenAccountsByDelegate(
        pubKey: String, mint: String, programId: String, req: TokenAccountsByDelegateReq? = null,
    ): TokenAccountsByDelegateResp {
        val map = mapOf("mint" to mint, "programId" to programId)
        return solanaApi.getTokenAccountsByDelegate(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getTokenAccountsByDelegate.value,
                listOf(pubKey, map, req)
            )
        ).result
    }

    suspend fun getTokenLargestAccounts(
        pubKey: String, commitmentInfo: CommitmentInfo? = null,
    ): TokenLargestAccountsResp {
        return solanaApi.getTokenLargestAccounts(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getTokenLargestAccounts.value,
                listOf(pubKey, commitmentInfo)
            )
        ).result
    }

    suspend fun getTokenSupply(
        pubKey: String, commitmentInfo: CommitmentInfo? = null,
    ): TokenSupplyResp {
        return solanaApi.getTokenSupply(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getTokenSupply.value,
                listOf(pubKey, commitmentInfo)
            )
        ).result
    }

    suspend fun getTransaction(signature: String, req: TransactionReq? = null): TransactionResp? {
        return solanaApi.getTransaction(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getTransaction.value,
                listOf(signature, req)
            )
        ).result
    }

    suspend fun getTransactionCount(req: TransactionCountReq? = null): BigInteger {
        return solanaApi.rpcRespBigInteger(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getTransactionCount.value,
                listOf(req)
            )
        ).result
    }

    suspend fun getVersion(): SolanaVersion {
        return solanaApi.getVersion(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getVersion.value,
                listOf()
            )
        ).result
    }

    suspend fun getVoteAccounts(req: VoteAccountsReq? = null): VoteAccountsResp {
        return solanaApi.getVoteAccounts(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.getVoteAccounts.value,
                listOf(req)
            )
        ).result
    }

    suspend fun isBlockhashValid(blockhash: String, req: BlockhashReq? = null): BlockhashResp {
        return solanaApi.isBlockhashValid(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.isBlockhashValid.value,
                listOf(blockhash, req)
            )
        ).result
    }

    suspend fun minimumLedgerSlot(): BigInteger {
        return solanaApi.rpcRespBigInteger(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.minimumLedgerSlot.value,
                listOf()
            )
        ).result
    }

    suspend fun requestAirdrop(
        pubKey: String, lamports: BigInteger, commitmentInfo: CommitmentInfo? = null,
    ): String {
        return solanaApi.rpcRespString(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                ReqBodyMethod.requestAirdrop.value,
                listOf(pubKey, lamports, commitmentInfo)
            )
        ).result
    }


}
