package com.particle.api.service

import com.blankj.utilcode.util.GsonUtils
import com.blankj.utilcode.util.LogUtils
import com.blankj.utilcode.util.Utils
import com.particle.api.BuildConfig
import com.particle.api.infrastructure.net.CommonApi
import com.particle.api.infrastructure.net.EvmRpcApi
import com.particle.api.infrastructure.net.SolanaRpcApi
import com.particle.api.infrastructure.net.interceptor.BasicAuthInterceptor
import com.particle.api.infrastructure.net.interceptor.ContentTypeInterceptor
import com.particle.api.infrastructure.net.interceptor.ErrorInterceptor
import com.particle.base.ParticleNetwork

import okhttp3.Cache
import okhttp3.Interceptor
import okhttp3.OkHttpClient
import okhttp3.logging.HttpLoggingInterceptor
import retrofit2.Retrofit
import retrofit2.converter.gson.GsonConverterFactory
import java.util.concurrent.TimeUnit

internal object NetService {

    const val RPC_VERSION = "2.0"
    private const val DEFAULT_CONNECT_TIMEOUT_SECONDS = 60L
    private const val DEFAULT_READ_TIMEOUT_SECONDS = 60L
    private const val DEFAULT_RESPONSE_CACHE_SIZE = 1024 * 1024 * 10L
    private val okHttpClient: OkHttpClient = OkHttpClient.Builder().build()

    val solana: SolanaService
    val evmService: EvmService
    val commonApiService: CommonAPIService

    private val solanaApi: SolanaRpcApi
    private val evmApi: EvmRpcApi
    private val commonApi: CommonApi

    init {
        solanaApi = getRetrofit(
            BuildConfig.PN_RPC_API_URL
        ).create(SolanaRpcApi::class.java)

        evmApi = getRetrofit(
            BuildConfig.PN_RPC_API_URL
        ).create(EvmRpcApi::class.java)

        commonApi = getRetrofit(
            com.particle.base.BuildConfig.PN_API_BASE_URL, true
        ).create(CommonApi::class.java)

        solana = SolanaService(solanaApi, commonApi)
        evmService = EvmService(evmApi, commonApi)
        commonApiService = CommonAPIService(commonApi)
        LogUtils.d("NetService init")
    }

    private fun getRetrofit(baseUrl: String, useCache: Boolean = false): Retrofit {
        return Retrofit.Builder()
            .baseUrl(baseUrl)
            .client(getClient(ErrorInterceptor(), useCache))
            .addConverterFactory(GsonConverterFactory.create(GsonUtils.getGson()))
            .build()
    }

    private fun getClient(
        interceptor: Interceptor? = null,
        useCache: Boolean = false
    ): OkHttpClient {
        return okHttpClient.newBuilder()
            .readTimeout(DEFAULT_CONNECT_TIMEOUT_SECONDS, TimeUnit.SECONDS)
            .connectTimeout(DEFAULT_READ_TIMEOUT_SECONDS, TimeUnit.SECONDS)
            .apply {
//                if (ParticleNetwork.isDebug()) {
//                    val httpLoggingInterceptor = HttpLoggingInterceptor()
//                    httpLoggingInterceptor.setLevel(HttpLoggingInterceptor.Level.HEADERS)
//                    addInterceptor(httpLoggingInterceptor)
//                }
                if (interceptor != null) addInterceptor(interceptor)
                if (useCache) {
                    cache(Cache(Utils.getApp().cacheDir, DEFAULT_RESPONSE_CACHE_SIZE))
                }

                addInterceptor(
                    BasicAuthInterceptor(
                        ParticleNetwork.getProjectUUID(),
                        ParticleNetwork.getProjectClientID()
                    )
                )
            }
            .addNetworkInterceptor(ContentTypeInterceptor())
            .build()
    }
}