package com.particle.api.service

import android.text.TextUtils
import androidx.annotation.Keep
import com.google.gson.JsonElement
import com.google.gson.JsonObject
import com.particle.api.BuildConfig
import com.particle.api.infrastructure.db.table.SplToken
import com.particle.api.infrastructure.db.table.TransInfoEvm
import com.particle.api.infrastructure.net.CommonApi
import com.particle.api.infrastructure.net.EvmRpcApi
import com.particle.api.infrastructure.net.data.*
import com.particle.api.infrastructure.net.data.resp.*
import com.particle.api.logic.TokenNFTListLogic
import com.particle.api.logic.TransactionCreator
import com.particle.api.service.DBService.nftInfoDao
import com.particle.api.service.DBService.tokenInfoDao
import com.particle.api.service.SplTokenService.insertSplToken
import com.particle.api.service.data.ContractParams
import com.particle.api.service.data.TokenNFTResult
import com.particle.api.service.data.TransGasFeeMode
import com.particle.base.ParticleNetwork
import com.particle.base.isEvmNativeAddress
import com.particle.base.model.*
import com.particle.base.utils.Constants
import com.particle.base.utils.prefixedHexToBigInteger
import network.particle.chains.ChainInfo
import okhttp3.ResponseBody
import java.math.BigInteger
import java.util.*

@Keep
class EvmService(private val evmApi: EvmRpcApi, private val commonApi: CommonApi) : ChainService {


    override suspend fun getTokenList(chainInfo: ChainInfo?): List<SplToken> {
        val chainName = chainInfo?.name?.lowercase() ?: ParticleNetwork.chainName
        val chainId: Long = chainInfo?.id ?: ParticleNetwork.chainId
        return try {
            val tokenUrl = "${BuildConfig.PN_SOLANA_TOKEN_LIST}${chainName}.json"
            val splTokens = commonApi.getTokenList(tokenUrl)
            DBService.splTokenDao.deleteByChainName(
                chainName
            )
            insertSplToken(splTokens, chainInfo)
            DBService.splTokenDao.getTokenList(chainName, chainId)
        } catch (e: Exception) {
            e.printStackTrace()
            DBService.splTokenDao.getTokenList(chainName, chainId)
        }

    }

    override suspend fun rpc(method: String, params: List<Any>?): ResponseBody {
        return evmApi.rpc(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                method,
                params,
            )
        )
    }

    /**
     * Get the real-time exchange rate of the EVM token
     */
    suspend fun getPrice(
        addresses: List<String>,
        currencies: List<String>,
        chainInfo: ChainInfo? = null
    ): RpcOutput<List<RateResult>> {
        val chainId = chainInfo?.id ?: ParticleNetwork.chainId
        val newAddress = addresses.map { if (it.isEvmNativeAddress()) "native" else it }
        return evmApi.enhancedGetPrice(
            ReqBody(
                chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.particleGetPrice.value,
                arrayListOf(newAddress, currencies)
            )
        )
    }

    /**
     * Get token list and NFT list from server by giving an address
     */
    override suspend fun getTokensAndNFTs(address: String): TokenNFTResult {
        return getTokensAndNFTsWithTokenAddress(address, listOf())
    }

    /**
     * Get token list and NFT list from server by giving an address and token address
     */
    suspend fun getTokensAndNFTsWithTokenAddress(
        address: String, tokenAddresses: List<String> = listOf(), chainInfo: ChainInfo? = null
    ): TokenNFTResult {
        TokenNFTListLogic.updateTokenNFTInfo(address, true, tokenAddresses, chainInfo)
        val tokenList = tokenInfoDao.getAllTokenInfo(
            address, ParticleNetwork.chainName, ParticleNetwork.chainId
        )
        val nftList = nftInfoDao.getAll(ParticleNetwork.chainName, ParticleNetwork.chainId)
        return TokenNFTResult(tokenList, nftList)
    }

    /**
     *
     */
    suspend fun getTokensWithAddress(
        address: String, chainInfo: ChainInfo?
    ): TokenNFTResult {
        TokenNFTListLogic.syncTokenInfo(address, chainInfo)
        val tokenList = tokenInfoDao.getAllTokenInfo(
            address, chainInfo?.name?.lowercase() ?: ParticleNetwork.chainName, chainInfo?.id ?: ParticleNetwork.chainId,
        )
        val nftList = nftInfoDao.getAll(ParticleNetwork.chainName, ParticleNetwork.chainId)
        return TokenNFTResult(tokenList, nftList)
    }

    /**
     * Get token list and NFT list from database by giving an address
     */
    override suspend fun getTokensAndNFTsFromDB(address: String): TokenNFTResult {
        val tokenList = tokenInfoDao.getAllTokenInfo(
            address, ParticleNetwork.chainName, ParticleNetwork.chainId
        )
        val nftList = nftInfoDao.getAll(ParticleNetwork.chainName, ParticleNetwork.chainId)
        return TokenNFTResult(tokenList, nftList)
    }

    /**
     * Get parsed transaction history from server by giving an address
     */
    suspend fun getTransactionsByAddress(
        address: String,
    ): List<TransInfoEvm> {
        fetchTransData2Db(address)
        return DBService.evmTransDao.getAllEvmTrans(
            address, ParticleNetwork.chainName, ParticleNetwork.chainId
        )
    }

    /**
     * Get parsed transaction history from database by giving an address
     */
    suspend fun getTransactionsByAddressFromDB(address: String): List<TransInfoEvm> {
        return DBService.evmTransDao.getAllEvmTrans(
            address, ParticleNetwork.chainName, ParticleNetwork.chainId
        )
    }

    /**
     * Get suggested GasFees
     */
    suspend fun suggestedGasFees(): RpcOutput<EvmGasResult> {
        val resp = evmApi.particleSuggestedGasFees(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.particleSuggestedGasFees.value
            )
        )
        return resp
    }

    /**
     * Serialization erc20_transfer data of contract
     */
    suspend fun erc20Transfer(
        contractAddress: String,
        to: String,
        amount: BigInteger,
    ): String {
        val list2 = arrayListOf(to, amount.toString())
        val list1 = arrayListOf(contractAddress, AbiEncodeFunction.erc20Transfer.value, list2)
        return abiEncodeFunctionCall(list1)
    }

    /**
     * Serialization erc20_approve data of contract
     */
    suspend fun erc20Approve(
        contractAddress: String,
        sender: String,
        amount: BigInteger,
    ): String {
        val list2 = arrayListOf(sender, amount.toString())
        val list1 = arrayListOf(contractAddress, AbiEncodeFunction.erc20Approve.value, list2)
        return abiEncodeFunctionCall(list1)
    }

    /**
     * Serialization erc20_transferFrom data of contract
     */
    suspend fun erc20TransferFrom(
        contractAddress: String,
        from: String,
        to: String,
        amount: BigInteger,
    ): String {
        val list2 = arrayListOf(from, to, amount.toString())
        val list1 = arrayListOf(contractAddress, AbiEncodeFunction.erc20TransferFrom.value, list2)
        return abiEncodeFunctionCall(list1)
    }

    /**
     * Serialization erc721SafeTransferFrom data of contract
     */
    suspend fun erc721SafeTransferFrom(
        contractAddress: String,
        from: String,
        to: String,
        tokenId: String,
    ): String {
        val list2 = arrayListOf(from, to, tokenId)
        val list1 =
            arrayListOf(contractAddress, AbiEncodeFunction.erc721SafeTransferFrom.value, list2)
        return abiEncodeFunctionCall(list1)
    }

    /**
     * Serialization erc1155SafeTransferFrom data of contract
     */
    suspend fun erc1155SafeTransferFrom(
        contractAddress: String,
        from: String,
        to: String,
        id: String,
        amount: BigInteger,
        data: String,
    ): String {
        val list2 = arrayListOf(from, to, id, amount.toString(), data)
        val list1 =
            arrayListOf(contractAddress, AbiEncodeFunction.erc1155SafeTransferFrom.value, list2)
        return abiEncodeFunctionCall(list1)
    }

    /**
     * abi encode
     */
    suspend fun abiEncodeFunctionCall(
        contractAddress: String, method: String, params: List<Any>, json: String = ""
    ): String {
        val list1 = arrayListOf(contractAddress, method, params)
        if (!TextUtils.isEmpty(json)) list1.add(json)
        return evmApi.particleAbiEncodeFunctionCall(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.particleAbiEncodeFunctionCall.value,
                list1
            )
        ).result
    }

    /**
     * read contract data
     * test data: readContract("0xd000f000aa1f8accbd5815056ea32a54777b2fc4","balanceOf",listOf("0xBbc1CA8776EfDeC12C75e218C64e96ce52aC6671"))
     */
    suspend fun readContract(
        contractAddress: String, method: String, params: List<Any>, json: String = ""
    ): String {
        val data = abiEncodeFunctionCall(contractAddress, "custom_$method", params, json)
        return ethCall(contractAddress, data)
    }

    suspend fun writeContract(from: String, contractParams: ContractParams): ITxData? {
        val data = abiEncodeFunctionCall(
            contractParams.contractAddress,
            contractParams.method,
            contractParams.params,
            contractParams.abiJsonString ?: ""
        )
        return createTransaction(
            from,
            contractParams.contractAddress,
            contractParams = contractParams,
            gasFeeLevel = TransGasFeeMode.high,
            abiEncodeData = data,
            value = "0x0",
            type = if (ParticleNetwork.chainInfo.isEIP1559Supported()) "0x2" else "0x0",
        )
    }


    suspend fun ethCall(contractAddress: String, data: String): String {
        val params1 = mapOf(
            "to" to contractAddress, "data" to data
        )
        val resp = evmApi.ethCall(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.ethCall.value,
                listOf(params1, "latest")
            )
        )
        return resp.result
    }

    /**
     * abi encode
     */
    suspend fun abiEncodeFunctionCall(params: List<Any>): String {
        return evmApi.particleAbiEncodeFunctionCall(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.particleAbiEncodeFunctionCall.value,
                params
            )
        ).result
    }


    private suspend fun fetchTransData2Db(address: String) {
        val response = getTransactionsByAddressFromServer(address)
        val evmTransResult: List<EvmTransResult> = response.result
        val allTrans = arrayListOf<TransInfoEvm>()
        for (trans in evmTransResult) {
            allTrans.add(
                TransInfoEvm(
                    address.uppercase(),
                    ParticleNetwork.chainName,
                    ParticleNetwork.chainId,
                    trans.from,
                    trans.to,
                    trans.hash.uppercase(),
                    trans.value,
                    trans.data,
                    trans.gasLimit,
                    trans.gasSpent,
                    trans.gasPrice,
                    trans.fees,
                    trans.type,
                    trans.nonce,
                    trans.maxPriorityFeePerGas,
                    trans.maxFeePerGas,
                    trans.timestamp,
                    trans.status,
                    trans.deserializedData
                )
            )
        }

        if (allTrans.isNotEmpty()) {
            DBService.evmTransDao.deleteAllTempTrans()
            DBService.evmTransDao.insertAll(allTrans)
        }
    }

    internal suspend fun getTokensAndNFTsFromServer(
        address: String, tokenAddresses: List<String> = listOf(),
        chainInfo: ChainInfo? = null
    ): RpcOutput<TokenResult> {
        return evmApi.particleGetTokensAndNFTs(
            ReqBody(
                chainInfo?.id ?: ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.particleGetTokensAndNFTs.value,
                arrayListOf(address, tokenAddresses)
            )
        )
    }

    private suspend fun getTransactionsByAddressFromServer(
        address: String,
    ): RpcOutput<List<EvmTransResult>> {
        return evmApi.particleGetTransactionsByAddress(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.particleGetTransactionsByAddress.value,
                arrayListOf(address)
            )
        )
    }

    /**
     * Create a transaction
     * @param from: sender address.
     * @param to: receiver address or contract address.
     * @param value: (optional) hex of the value sent with this transaction.
     * @param contractParams: (optional) call contract function params.
     * @param type: default value "0x2"(EIP1559), "0x1"(EIP2930), "0x0"(Legacy)
     * @param nonce: default value "0x0", particle auth manage nonce without cancel or speed transaction.
     * @param gasFeeLevel: default value TransGasFeeMode.medium, transaction gas fee level.
     * @param action:  default TxAction.normal. if you cancel/speed tracsaction, set the vaule.
     * @param abiEncodeData: the abi_encodeFunctionCall response, default value is null.
     */
    suspend fun createTransaction(
        from: PrefixedHexString,
        to: PrefixedHexString? = null,
        value: PrefixedHexString? = null,
        contractParams: ContractParams? = null,
        type: PrefixedHexString = "0x2",
        nonce: PrefixedHexString = "0x0",
        gasFeeLevel: TransGasFeeMode = TransGasFeeMode.medium,
        action: String = TxAction.normal.toString(),
        abiEncodeData: String? = null
    ): ITxData? {
        return TransactionCreator.createTrans(
            from, to, value ?: "0x0", contractParams, type, nonce, gasFeeLevel, action, abiEncodeData
        )
    }


    /**
     *  get token info by address
     */
    suspend fun getTokensByTokenAddresses(
        address: String, tokenAddresses: List<String>
    ): RpcOutput<List<CustomTokenResult>> {
        return evmApi.particleGetTokensByTokenAddresses(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.particleGetTokensByTokenAddresses.value,
                arrayListOf(address, tokenAddresses)
            )
        )
    }

    suspend fun addCustomTokens(address: String, tokenAddresses: List<String>): List<String> {
        return TokenNFTListLogic.addCustomToken(address, tokenAddresses)
    }

    /**
     * @param provider :all | 1inch | izi-swap default value is 1inch
     */
    @Throws(ServerException::class)
    suspend fun getQuote(
        address: String, fromTokenAddress: String, toTokenAddress: String, amount: BigInteger, provider: String = "1inch"
    ): RpcOutput<QuoteInfoEVM?> {
        val map = mutableMapOf<String, String>()
        map["fromTokenAddress"] = fromTokenAddress
        map["toTokenAddress"] = toTokenAddress
        map["amount"] = amount.toString()
        map["provider"] = provider
        map["version"] = "v2"
        return evmApi.particleGetQuote(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.particleSwapGetQuote.value,
                arrayListOf(address, map)
            )
        )
    }

    @Throws(ServerException::class)
    suspend fun particleSwapCreateOrder(
        address: String, fromTokenAddress: String, toTokenAddress: String, amount: BigInteger
    ): RpcOutput<JsonObject> {
        val map = mutableMapOf<String, String>()
        map["fromTokenAddress"] = fromTokenAddress
        map["toTokenAddress"] = toTokenAddress
        map["amount"] = amount.toString()
        map["version"] = "v2"
        return evmApi.particleSwapCreateOrder(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.particleSwapCreateOrder.value,
                arrayListOf(address, map)
            )
        )
    }

    @Throws(ServerException::class)
    suspend fun particleSwapSubmitOrder(
        address: String, jobj: JsonObject,
    ): RpcOutput<JsonElement> {
        return evmApi.particleSwapSubmitOrder(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.particleSwapSubmitOrder.value,
                arrayListOf(address, jobj)
            )
        )
    }


    suspend fun particleSwapGetSwap(
        address: String, inputMint: String, outMint: String, amount: BigInteger, slippage: Float, provider: String = "1inch"
    ): ParticleGetSwapResp? {
        val map = mutableMapOf<String, Any>()
        map["fromTokenAddress"] = inputMint
        map["toTokenAddress"] = outMint
        map["amount"] = amount.toString()
        map["slippage"] = slippage
        map["provider"] = provider
        map["version"] = "v2"
        return evmApi.particleSwapGetSwap(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.particleSwapGetSwap.value,
                arrayListOf(address, map)
            )
        ).result
    }

    @Deprecated("use ethGetBalance", ReplaceWith("ethGetBalance(pubKey)"))
    suspend fun getBalance(address: String): BigInteger {
        return ethGetBalance(address).prefixedHexToBigInteger()
    }

    suspend fun swapCheckApprove(
        address: String, tokenAddress: String, amount: BigInteger, provider: String
    ): ApprovedInfo {
        val map = mutableMapOf<String, String>()
        map["tokenAddress"] = tokenAddress
        map["amount"] = amount.toString()
        map["provider"] = provider
        map["version"] = "v2"
        return evmApi.swapCheckApprove(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.particleSwapCheckApprove.value,
                listOf(address, map)
            )
        ).result
    }

    suspend fun checkTransactionConfirmed(transaction: String): Boolean {
        val rs = evmApi.getTransactionReceipt(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.ethGetTransactionReceipt.value,
                listOf(transaction)
            )
        ).result
        return rs != null
    }

    /**
     * Returns the current client version.
     * @return String - The current client version
     */
    suspend fun web3ClientVersion(): String {
        return evmApi.particleRespStringFunctionCall(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.web3ClientVersion.value,
                listOf()
            )
        ).result
    }

    /**
     * Returns Keccak-256 (not the standardized SHA3-256) of the given data.
     * @param DATA - The SHA3 result of the given string.
     * @return DATA - The SHA3 result of the given string.
     */
    suspend fun web3Sha3(data: String): String {
        return evmApi.particleRespStringFunctionCall(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.web3Sha3.value,
                listOf(data)
            )
        ).result
    }

    /**
     * Returns the current network id.
     * @return String - The current network id.
     */
    suspend fun netVersion(): String {
        return evmApi.particleRespStringFunctionCall(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.netVersion.value,
                listOf()
            )
        ).result
    }


    /**
     * Returns the current Ethereum protocol version.
     * @return String - The current Ethereum protocol version
     * // Result
     *  {
     *      "id":67,
     *      "jsonrpc": "2.0",
     *      "result": "54"
     *  }
     */
    suspend fun ethProtocolVersion(): String {
        return evmApi.particleRespStringFunctionCall(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.ethProtocolVersion.value,
                listOf()
            )
        ).result
    }

    /**
     * Returns an object with data about the sync status or false.
     * // Result
     *   {
     *   "id":1,
     *   "jsonrpc": "2.0",
     *   "result": {
     *       startingBlock: '0x384',
     *       currentBlock: '0x386',
     *       highestBlock: '0x454'
     *   }
     *   }
     *   // Or when not syncing
     *   {
     *       "id":1,
     *       "jsonrpc": "2.0",
     *       "result": false
     *   }
     */
    suspend fun ethSyncing(): Any {
        return evmApi.particleRespAnyFunctionCall(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.ethSyncing.value,
                listOf()
            )
        ).result
    }


    /**
     * Returns true if client is actively mining new blocks.
     * @return Boolean - true when mining, otherwise false.
     */
    suspend fun ethMining(): Boolean {
        return evmApi.particleRespBoolFunctionCall(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.ethMining.value,
                listOf()
            )
        ).result
    }


    /**
     * Returns the current price per gas in wei.
     * @return QUANTITY - integer of the current gas price in wei.
     */
    suspend fun ethGasPrice(): PrefixedHexString {
        return evmApi.particleRespHexStringFunctionCall(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.ethGasPrice.value,
                listOf()
            )
        ).result
    }

    /**
     * Returns a list of addresses owned by client.
     * @return DATA, 20 Bytes - address of the account.
     * // Result
     *   {
     *       "id":83,
     *       "jsonrpc": "2.0",
     *       "result": "0x4b7" // 1207
     *   }
     */
    suspend fun ethBlockNumber(): PrefixedHexString {
        return evmApi.particleRespHexStringFunctionCall(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.ethBlockNumber.value,
                listOf()
            )
        ).result
    }

    /**
     * Returns the balance of the account of given address.
     * @param address: DATA, 20 Bytes - address to check for balance.
     * @param block: QUANTITY|TAG - integer block number, or the string "latest", "earliest" or "pending", see the default block parameter.
     * @return QUANTITY - integer of the current balance in wei.
     * // Result
     *  {
     *      "id":1,
     *      "jsonrpc": "2.0",
     *      "result": "0x0234c8a3397aab58" // 158972490234375000
     *  }
     */
    suspend fun ethGetBalance(address: String, blockParams: String = "latest"): PrefixedHexString {
        return evmApi.particleGetBalance(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.ethGetBalance.value,
                listOf(address, blockParams)
            )
        ).result
    }

    /**
     * Returns the value from a storage position at a given address.
     * @param address: DATA, 20 Bytes - address of the storage.
     * @param position: QUANTITY - integer of the position in the storage.
     * @param block: QUANTITY|TAG - integer block number, or the string "latest", "earliest" or "pending", see the default block parameter.
     * @return DATA - the value at this storage position.
     */
    suspend fun ethGetStorageAt(
        address: String, position: PrefixedHexString, block: String = "latest"
    ): PrefixedHexString {
        return evmApi.particleRespHexStringFunctionCall(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.ethGetStorageAt.value,
                listOf(address, position, block)
            )
        ).result
    }

    /**
     * Returns the number of transactions sent from an address.
     * @param address: DATA, 20 Bytes - address.
     * @param block: QUANTITY|TAG - integer block number, or the string "latest", "earliest" or "pending", see the default block parameter.
     * @return QUANTITY - integer of the number of transactions send from this address.
     * // Result
     *   {
     *      "id":1,
     *      "jsonrpc": "2.0",
     *      "result": "0x1" // 1
     *   }
     */
    suspend fun ethGetTransactionCount(
        address: String, block: String = "latest"
    ): PrefixedHexString {
        return evmApi.particleRespHexStringFunctionCall(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.ethGetTransactionCount.value,
                listOf(address, block)
            )
        ).result
    }

    /**
     * Returns the number of transactions in a block from a block matching the given block hash.
     * @param blockHash: DATA, 32 Bytes - hash of a block.
     * @return QUANTITY - integer of the number of transactions in this block.
     */
    suspend fun ethGetBlockTransactionCountByHash(hashBlock: String): PrefixedHexString {
        return evmApi.particleRespHexStringFunctionCall(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.ethGetBlockTransactionCountByHash.value,
                listOf(hashBlock)
            )
        ).result
    }

    /**
     * Returns the number of transactions in a block matching the given block number.
     * @param block: QUANTITY|TAG - integer of a block number, or the string "earliest", "latest" or "pending", as in the default block parameter.
     * @return QUANTITY - integer of the number of transactions in this block.
     */
    suspend fun ethGetBlockTransactionCountByNumber(block: String): PrefixedHexString {
        return evmApi.particleRespHexStringFunctionCall(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.ethGetBlockTransactionCountByNumber.value,
                listOf(block)
            )
        ).result
    }

    /**
     * Returns the number of uncles in a block from a block matching the given block hash.
     * @param blockHash: DATA, 32 Bytes - hash of a block.
     * @return QUANTITY - integer of the number of uncles in this block.
     *
     */
    suspend fun ethGetUncleCountByBlockHash(hashBlock: String): PrefixedHexString {
        return evmApi.particleRespHexStringFunctionCall(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.ethGetUncleCountByBlockHash.value,
                listOf(hashBlock)
            )
        ).result
    }

    /**
     * Returns the number of uncles in a block from a block matching the given block number.
     * @param block: QUANTITY|TAG - integer of a block number, or the string "earliest", "latest" or "pending", as in the default block parameter.
     * @return QUANTITY - integer of the number of uncles in this block.
     *
     */
    suspend fun ethGetUncleCountByBlockNumber(block: String): PrefixedHexString {
        return evmApi.particleRespHexStringFunctionCall(
            ReqBody(
                ParticleNetwork.chainId, UUID.randomUUID().toString(), NetService.RPC_VERSION,

                EvmReqBodyMethod.ethGetUncleCountByBlockNumber.value, listOf(block)
            )
        ).result
    }


    /**
     * Returns code at a given address.
     *  @param address: DATA, 20 Bytes - address.
     *  @param block: QUANTITY|TAG - integer block number, or the string "latest", "earliest" or "pending", see the default block parameter.
     */
    suspend fun ethGetCode(address: String, block: String): PrefixedHexString {
        return evmApi.particleRespHexStringFunctionCall(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.ethGetCode.value,
                listOf(address, block)
            )
        ).result
    }

    /**
     * Returns information about a block by hash.
     * @param blockHash: DATA, 32 Bytes - hash of a block.
     * @param fullTransactionObjects: If true it returns the full transaction objects, if false only the hashes of the transactions.
     * @return Object - A block object, or null when no block was found:
     *
     */
    suspend fun ethGetBlockByHash(
        blockhash: String, fullTransactionObjects: Boolean
    ): BlockByHashInfo? {
        return evmApi.ethGetBlockByHash(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.ethGetBlockByHash.value,
                listOf(blockhash, fullTransactionObjects)
            )
        ).result
    }

    /**
     * Returns information about a block by block number.
     * @param block: QUANTITY|TAG - integer of a block number, or the string "earliest", "latest" or "pending", as in the default block parameter.
     * @param fullTransactionObjects: If true it returns the full transaction objects, if false only the hashes of the transactions.
     */
    suspend fun ethGetBlockByNumber(
        block: String, fullTransactionObjects: Boolean
    ): BlockByHashInfo? {
        return evmApi.ethGetBlockByHash(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.ethGetBlockByNumber.value,
                listOf(block, fullTransactionObjects)
            )
        ).result
    }

    suspend fun ethGetBlockByNumber(
        params: List<Any>
    ): BlockByHashInfo? {
        return evmApi.ethGetBlockByHash(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.ethGetBlockByNumber.value,
                params
            )
        ).result
    }

    /**
     * Returns the information about a transaction requested by transaction hash.
     * @param transactionHash: DATA, 32 Bytes - hash of a transaction.
     * @return Object - A transaction object, or null when no transaction was found:
     */
    suspend fun ethGetTransactionByHash(transactionHash: String): TransactionByHashInfo? {
        return evmApi.ethGetTransaction(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.ethGetTransactionByHash.value,
                listOf(transactionHash)
            )
        ).result
    }

    /**
     * Returns information about a transaction by block hash and transaction index position.
     * @param blockHash: DATA, 32 Bytes - hash of a block.
     * @param transactionIndex: QUANTITY - integer of the transaction index position.
     * @return Object - A transaction object, or null when no transaction was found:
     *
     */
    suspend fun ethGetTransactionByBlockHashAndIndex(
        blockHash: String,
        transactionIndex: String
    ): TransactionByHashInfo? {
        return evmApi.ethGetTransaction(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.ethGetTransactionByBlockHashAndIndex.value,
                listOf(blockHash, transactionIndex)
            )
        ).result
    }

    /**
     * Returns information about a transaction by block number and transaction index position.
     * @param blockHash: QUANTITY|TAG - integer of a block number, or the string "earliest", "latest" or "pending", as in the default block parameter.
     * @param transactionIndex: QUANTITY - integer of the transaction index position.
     */
    suspend fun ethGetTransactionByBlockNumberAndIndex(
        blockHash: String,
        transactionIndex: String
    ): TransactionByHashInfo? {
        return evmApi.ethGetTransaction(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.ethGetTransactionByBlockNumberAndIndex.value,
                listOf(blockHash, transactionIndex)
            )
        ).result
    }

    /**
     * Returns the receipt of a transaction by transaction hash.
     * @param transactionHash: DATA, 32 Bytes - hash of a transaction.
     * @return Object - A transaction receipt object, or null when no receipt was found:
     */
    suspend fun ethGetTransactionReceipt(transactionHash: String): Any? {
        return evmApi.getTransactionReceipt(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.ethGetTransactionReceipt.value,
                listOf(transactionHash)
            )
        ).result
    }

    /**
     *Returns information about a uncle of a block by hash and uncle index position.
     * @param blockHash: DATA, 32 Bytes - hash of a block.
     * @param uncleIndex: QUANTITY - the uncle's index position.
     */
    suspend fun ethGetUncleByBlockHashAndIndex(
        blockHash: String,
        uncleIndex: String
    ): BlockByHashInfo? {
        return evmApi.ethGetBlockByHash(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.ethGetUncleByBlockHashAndIndex.value,
                listOf(blockHash, uncleIndex)
            )
        ).result
    }

    /**
     * Returns information about a uncle of a block by number and uncle index position.
     * @param block: QUANTITY|TAG - integer of a block number, or the string "earliest", "latest" or "pending", as in the default block parameter.
     * @param uncleIndex: QUANTITY - the uncle's index position.
     */
    suspend fun ethGetUncleByBlockNumberAndIndex(
        block: String,
        uncleIndex: String
    ): BlockByHashInfo? {
        return evmApi.ethGetBlockByHash(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.ethGetUncleByBlockHashAndIndex.value,
                listOf(block, uncleIndex)
            )
        ).result
    }


    suspend fun ethGetWork(): List<String> {
        return evmApi.particleRespListStringFunctionCall(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.ethGetWork.value,
                listOf()
            )
        ).result
    }

    suspend fun ethSubmitWork(nonce: String, powHash: String, mixDigest: String): Boolean {
        return evmApi.particleRespBoolFunctionCall(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.ethSubmitWork.value,
                listOf(nonce, powHash, mixDigest)
            )
        ).result
    }


    suspend fun ethSubmitHashrate(hashrate: String, randomId: String): Boolean {
        return evmApi.particleRespBoolFunctionCall(
            ReqBody(
                ParticleNetwork.chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.ethSubmitHashrate.value,
                listOf(hashrate, randomId)
            )
        ).result
    }


    suspend fun particleBridgeGetTokens(chainId: Long): List<BridgeSupportedToken> {
        return evmApi.particleBridgeGetTokens(
            ReqBody(
                chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.particleBridgeGetTokens.value,
                listOf()
            )
        ).result
    }

    @Throws(ServerException::class)
    suspend fun particleBridgeGetQuote(
        pubAddress: String,
        fromChainId: Long,
        fromTokenAddress: String,
        toChainId: Long,
        toTokenAddress: String,
        amount: BigInteger,
        slippage: Float
    ): RpcOutput<BridgeGetQuoteResp> {
        val map = mutableMapOf<String, Any>()
        map["toChainId"] = toChainId
        map["fromTokenAddress"] = if (fromTokenAddress.isEvmNativeAddress()) "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee" else fromTokenAddress
        map["toTokenAddress"] = if (toTokenAddress.isEvmNativeAddress()) "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee" else toTokenAddress
        map["amount"] = amount.toString()
        map["slippage"] = slippage
        return evmApi.particleBridgeGetQuote(
            ReqBody(
                fromChainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.particleBridgeGetQuote.value,
                listOf(pubAddress, map)
            )
        )
    }

    @Throws(ServerException::class)
    suspend fun particleBridgeCheckApprove(
        pubAddress: String,
        chainId: Long,
        tokenAddress: String,
        amount: BigInteger
    ): RpcOutput<BridgeApproveResult> {
        val map = mutableMapOf<String, Any>()
        map["tokenAddress"] = tokenAddress
        map["amount"] = amount.toString()
        map["provider"] = "lifi"
        return evmApi.particleBridgeCheckApprove(
            ReqBody(
                chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.particleBridgeCheckApprove.value,
                listOf(pubAddress, map)
            )
        )
    }

    @Throws(ServerException::class)
    suspend fun particleBridgeHistory(
        pubAddress: String,
        chainId: Long
    ): RpcOutput<List<BridgeHistory>> {
        return evmApi.particleBridgeHistory(
            ReqBody(
                chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.particleBridgeHistory.value,
                listOf(pubAddress)
            )
        )
    }

    @Throws(ServerException::class)
    suspend fun particleBridgeStatus(
        signature: String,
        chainId: Long
    ): RpcOutput<JsonElement> {
        return evmApi.particleBridgeStatus(
            ReqBody(
                chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.particleBridgeStatus.value,
                listOf(signature)
            )
        )
    }

    @Throws(ServerException::class)
    suspend fun particleDidGetAddress(
        pubAddress: String,
        chainId: Long
    ): RpcOutput<String?> {
        return evmApi.particleDidGetAddress(
            ReqBody(
                chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.particleDidGetAddress.value,
                listOf(pubAddress)
            )
        )
    }

    @Throws(ServerException::class)
    suspend fun particleDidGetDID(
        pubAddress: String,
        chainId: Long
    ): RpcOutput<List<String>> {
        return evmApi.particleDidGetDID(
            ReqBody(
                chainId,
                UUID.randomUUID().toString(),
                NetService.RPC_VERSION,
                EvmReqBodyMethod.particleDidGetDID.value,
                listOf(pubAddress)
            )
        )
    }
}
