package com.particle.api.logic

import android.text.TextUtils
import com.particle.api.evm
import com.particle.api.infrastructure.net.data.EvmReqBodyMethod
import com.particle.api.service.data.ContractParams
import com.particle.api.service.data.TransGasFeeMode
import com.particle.base.ParticleNetwork
import com.particle.base.model.*
import com.particle.base.utils.gweiToHexStr
import com.particle.base.utils.toDecimalInt
import com.particle.base.utils.toHexStr
import org.json.JSONObject
import java.util.HashMap

object TransactionCreator {

    suspend fun createTrans(
        from: String,
        to: String?,
        value: String?,
        contractParams: ContractParams?,
        type: String = "0x2",
        nonce: String = "0x0",
        gasFeeLevel: TransGasFeeMode = TransGasFeeMode.medium,
        action: String = TxAction.normal.toString(),
        abiEncodeData: String? = null
    ): ITxData? {
        val ethValue = value ?: "0x0"
        val data: String = if (abiEncodeData != null && !TextUtils.isEmpty(abiEncodeData)) {
            abiEncodeData
        } else {
            if (contractParams == null) {
                "0x"
            } else {
                val result = ParticleNetwork.evm.abiEncodeFunctionCall(
                    contractParams.contractAddress,
                    contractParams.method,
                    contractParams.params,
                    contractParams.abiJsonString ?: ""
                )
                result
            }
        }

        val gasFees = ParticleNetwork.evm.suggestedGasFees().result
        val gasInfo = when (gasFeeLevel) {
            TransGasFeeMode.low -> {
                gasFees.low
            }

            TransGasFeeMode.medium -> {
                gasFees.medium
            }

            else -> {
                gasFees.high
            }
        }
        val gasLimit = getGasLimit(from, to, ethValue, data)
        var txData: ITxData? = null
        when (type) {
            "0x0" -> {
                txData = TxData(
                    gasPrice = gasInfo.maxFeePerGas.gweiToHexStr(),
                    nonce = nonce,
                    gasLimit.toString().toHexStr(),
                    from,
                    to ?: "", value, data,
                    chainId = ParticleNetwork.chainId.toString().toHexStr(),
                    action = action
                )
            }

            "0x1" -> {
                txData = AccessListEIP2930TxData(
                    gasInfo.maxFeePerGas.gweiToHexStr(),
                    null,
                    nonce = nonce,
                    gasLimit.toString().toHexStr(),
                    from,
                    to ?: "",
                    value ?: "0x0",
                    data,
                    ParticleNetwork.chainId.toString().toHexStr(),
                    action = action
                )
            }

            "0x2" -> {
                txData = FeeMarketEIP1559TxData(
                    gasInfo.maxPriorityFeePerGas.gweiToHexStr(),
                    gasInfo.maxFeePerGas.gweiToHexStr(),
                    nonce = nonce,
                    gasLimit.toString().toHexStr(),
                    from,
                    to ?: "",
                    value ?: "0x0",
                    data,
                    chainId = ParticleNetwork.chainId.toString().toHexStr(),
                    action = action
                )
            }
        }
        return txData
    }

    private suspend fun getGasLimit(
        from: String,
        to: String?,
        value: String?,
        data: String
    ): Int {
        val map = HashMap<String, String>()
        map["from"] = from
        if (!TextUtils.isEmpty(to)) {
            map["to"] = to!!
        }
        map["value"] = if (TextUtils.isEmpty(value)) "0x0" else value!!
        map["data"] = data
        val params = arrayListOf(map)
        val resp =
            ParticleNetwork.evm.rpc(EvmReqBodyMethod.ethEstimateGas.value, params)
        val jobj = JSONObject(resp.string())
        val limit = jobj.getString("result")
        val gasLimit = limit.toDecimalInt()
        return gasLimit
    }
}