package com.particle.api.logic

import android.text.TextUtils
import com.blankj.utilcode.util.LogUtils
import com.particle.api.ParticleNetworkApi
import com.particle.api.evm
import com.particle.api.infrastructure.db.table.NftInfo
import com.particle.api.infrastructure.db.table.SplToken
import com.particle.api.infrastructure.db.table.SplTokenSource
import com.particle.api.infrastructure.db.table.TokenInfo
import com.particle.api.infrastructure.net.data.resp.CustomTokenResult
import com.particle.api.infrastructure.net.data.resp.Token

import com.particle.api.infrastructure.net.data.resp.TokenResult
import com.particle.api.service.DBService
import com.particle.api.solana
import com.particle.base.ParticleNetwork
import network.particle.chains.ChainInfo
import java.math.BigDecimal
import java.math.BigInteger
import kotlin.math.pow

internal object TokenNFTListLogic {
    val NATIVE = "native"
    val maxTime = 9000000000000

    private suspend fun getTokenNFTList(
        address: String,
        tokenAddresses: List<String> = listOf(),
        chainInfo: ChainInfo? = null
    ): TokenResult? {
        return if (ParticleNetwork.isEvmChain()) {
            val response = ParticleNetwork.evm.getTokensAndNFTsFromServer(
                address, tokenAddresses, chainInfo = chainInfo
            )
            val reqChainId = chainInfo?.id ?: ParticleNetwork.chainId
            if (reqChainId != response.chainId) {
                null
            } else {
                response.result
            }
        } else {
            val response = ParticleNetwork.solana.getTokensAndNFTsFromServer(
                address,
                chainInfo = chainInfo
            )
            if (chainInfo != null && chainInfo.id != response.chainId) {
                null
            } else if (ParticleNetwork.chainId != response.chainId) {
                null
            } else {
                response.result
            }
        }

    }

    private suspend fun updateTokenInfo(
        address: String,
        tokenResult: TokenResult,
        chainInfo: ChainInfo? = null
    ) {
        try {
            val chainName: String = chainInfo?.name?.lowercase() ?: ParticleNetwork.chainName
            val chainId: Long = chainInfo?.id ?: ParticleNetwork.chainId
            val tokens = tokenResult.tokens
            initNative(address, tokenResult, chainInfo)
            if (tokens.isEmpty())
                return
            val isEvm = ParticleNetwork.isEvmChain()
            val updateList = arrayListOf<TokenInfo>()
            val addList = arrayListOf<TokenInfo>()
            val addUpdateList = arrayListOf<TokenInfo>()
            val pubKey = address
            for (token in tokens) {
                splTokenCheck(token, chainInfo)
                val tokenInfo = DBService.tokenInfoDao.getWalletTokenByAddress(
                    pubKey,
                    chainName,
                    chainId,
                    if (isEvm) token.address else token.mint
                )
                if (tokenInfo != null) {
                    tokenInfo.amount = token.amount
                    updateList.add(tokenInfo)
                } else {
                    val tmpTokenInfo = TokenInfo(
                        pubKey,
                        chainName,
                        chainId,
                        if (isEvm) token.address else token.mint,
                        token.symbol,
                        token.amount,
                        token.decimals,
                        1,
                        0,
                        BigDecimal.valueOf(10.0.pow(token.decimals.toDouble())).toBigInteger()
                    )
                    addList.add(tmpTokenInfo)
                }
            }
            if (updateList.isNotEmpty()) {
                DBService.tokenInfoDao.updateAll(updateList)
                addUpdateList.addAll(updateList)
            }
            if (addList.isNotEmpty()) {
                DBService.tokenInfoDao.insertAll(addList)
                addUpdateList.addAll(addList)
            }
            if (addUpdateList.isNotEmpty()) {
                val allTokenInfos = DBService.tokenInfoDao.getAllDisplayed(
                    pubKey,
                    chainName,
                    chainId
                )
                val allTokenInfosArray = mutableListOf<TokenInfo>().apply { addAll(allTokenInfos) }
                allTokenInfosArray.removeAll(addUpdateList)
                if (allTokenInfosArray.isNotEmpty()) {
                    for (resetToken in allTokenInfosArray) {
                        if (resetToken.address == NATIVE) continue
                        resetToken.amount = 0.toBigInteger()
                    }
                    DBService.tokenInfoDao.updateAll(allTokenInfosArray)
                }
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    private suspend fun initNative(address: String, tokenResult: TokenResult, chainInfo: ChainInfo? = null) {
        val chainName: String = chainInfo?.name?.lowercase() ?: ParticleNetwork.chainName
        val chainId: Long = chainInfo?.id ?: ParticleNetwork.chainId
        val decimals: Int = chainInfo?.nativeCurrency?.decimals ?: ParticleNetwork.chainInfo.nativeCurrency.decimals
        val symbol: String = chainInfo?.nativeCurrency?.symbol ?: ParticleNetwork.chainInfo.nativeCurrency.symbol
        val name: String = chainInfo?.nativeCurrency?.name ?: ParticleNetwork.chainInfo.nativeCurrency.name
        val pubKey: String = address
        val amount = tokenResult.native
        if (chainInfo == null) {
            DBService.tokenInfoDao.deleteToken(
                pubKey,
                chainName,
                chainId,
                NATIVE
            );
        }
        val tmpTokenInfo = TokenInfo(
            pubKey.uppercase(),
            chainName,
            chainId,
            NATIVE,
            symbol,
            amount,
            decimals,
            1,
            maxTime,
            BigDecimal.valueOf(10.0.pow(decimals.toDouble()))
                .toBigInteger()
        )
        DBService.tokenInfoDao.insert(tmpTokenInfo)
        DBService.splTokenDao.insert(
            SplToken(
                chainName,
                chainId,
                NATIVE,
                symbol,
                decimals,
                name,
                ""
            )
        )
    }


    private suspend fun updateNftInfo(tokenResult: TokenResult, chainInfo: ChainInfo? = null) {
        try {
            val chainName: String = chainInfo?.name?.lowercase() ?: ParticleNetwork.chainName
            val chainId: Long = chainInfo?.id ?: ParticleNetwork.chainId
            val nfts = tokenResult.nfts
            if (nfts.isEmpty()) {
                DBService.nftInfoDao.deleteAll()
                return
            }
            val isEvm = ParticleNetwork.isEvmChain()
            val nftList = arrayListOf<NftInfo>()
            for (nft in nfts) {
                val externalUrl =
                    if (isEvm) nft.data?.external_url
                        ?: nft.data?.external_link else nft.metadata?.data?.uriData?.external_url
                var strExternalUrl = ""
                if (externalUrl != null) {
                    strExternalUrl = externalUrl as String
                }
                var mintAddress: String
                if (ParticleNetwork.isEvmChain()) {
                    mintAddress = nft.address!!
                } else {
                    mintAddress = nft.mint!!
                }
                val image: String = if (!TextUtils.isEmpty(nft.image)) {
                    nft.image!!
                } else {
                    if (isEvm) nft.data?.image ?: "" else nft.metadata?.data?.uriData?.image ?: ""
                }
                nftList.add(
                    NftInfo(
                        chainName,
                        chainId,
                        mintAddress,
                        image,
                        nft.symbol!!,
                        nft.name,
                        nft.metadata?.data?.uriData?.seller_fee_basis_points,
                        nft.metadata?.data?.uriData?.description,
                        strExternalUrl,
                        if (isEvm) nft?.data?.animation_url else nft.metadata?.data?.uriData?.animation_url,
                        nft?.metadata,
                        nft?.data,
                        nft.isSemiFungible,
                        nft.tokenId ?: mintAddress,
                        nft.tokenBalance,
                        isScam = if (nft.isScam) 1 else 0
                    )
                )
            }
            if (nftList.isNotEmpty()) {
                DBService.nftInfoDao.deleteAll()
                DBService.nftInfoDao.insertAll(nftList)
            }
            LogUtils.d("update local NFT data success")
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    suspend fun updateTokenNFTInfo(
        address: String, isOverwriteSelected: Boolean, tokenAddresses: List<String> = listOf(),
        chainInfo: ChainInfo? = null
    ) {
        val tokenNFTList = getTokenNFTList(address, tokenAddresses, chainInfo) ?: return
        updateTokenInfo(address, tokenNFTList, chainInfo)
        updateNftInfo(tokenNFTList)
        RatesLogic.updateRates(address)
    }

    suspend fun syncTokenInfo(
        address: String,
        chainInfo: ChainInfo?
    ) {
        val tokenNFTList = getTokenNFTList(address, emptyList(), chainInfo) ?: return
        updateTokenInfo(address, tokenNFTList, chainInfo)
        RatesLogic.updateRates(address)
    }


    suspend fun addCustomToken(address: String, tokenAddresses: List<String>): List<String> {
        val result: List<CustomTokenResult> = if (ParticleNetwork.isEvmChain()) {
            ParticleNetwork.evm.getTokensByTokenAddresses(
                address,
                tokenAddresses
            ).result
        } else {
            ParticleNetwork.solana.getTokensByTokenAddresses(
                address,
                tokenAddresses
            ).result
        }
        val successInsertList = mutableListOf<String>()
        for (token in result) {
            val splToken = SplToken(
                ParticleNetwork.chainName,
                ParticleNetwork.chainId,
                token.address,
                token.symbol,
                token.decimals,
                token.name,
                token.image
            )
            DBService.splTokenDao.insert(splToken)
            val tokenInfo = TokenInfo(
                address,
                ParticleNetwork.chainName,
                ParticleNetwork.chainId,
                token.address,
                token.symbol,
                BigInteger(token.amount),
                token.decimals,
                1,
                System.currentTimeMillis(),
                BigDecimal.valueOf(10.0.pow(token.decimals.toDouble())).toBigInteger()
            )
            DBService.tokenInfoDao.insert(tokenInfo)
            successInsertList.add(token.address)
        }
        return successInsertList
    }

    private suspend fun splTokenCheck(token: Token, chainInfo: ChainInfo? = null) {
        val chainName = chainInfo?.name?.lowercase() ?: ParticleNetwork.chainName
        val chainId = chainInfo?.id ?: ParticleNetwork.chainId
        val isExist = DBService.splTokenDao.isAddressExist(
            chainName,
            chainId,
            token.address
        )
        if (!isExist) {
            val source = if (ParticleNetworkApi.priorityTokenAddresses.any { it.equals(token.address, ignoreCase = true) })
                SplTokenSource.Particle.index
            else
                SplTokenSource.SUSPICIOUS.index
            val splToken = SplToken(
                chainName,
                chainId,
                token.address,
                token.symbol ?: "",
                token.decimals,
                token.name ?: "",
                token.image ?: "",
                source
            )
            DBService.splTokenDao.insert(splToken)
        }

    }
}