package com.particle.api.logic

import com.particle.api.solana
import com.particle.base.ParticleNetwork
import org.p2p.solanaj.core.PublicKey
import org.p2p.solanaj.kits.TokenTransaction
import org.p2p.solanaj.model.types.*
import org.p2p.solanaj.programs.TokenProgram

data class TransactionAddressData(
    val associatedAddress: PublicKey,
    val shouldCreateAssociatedInstruction: Boolean
)

object SolanaRpcRepository {


    suspend fun getRecentBlockhash(commitment: String = "finalized"): String {
        return ParticleNetwork.solana.getRecentBlockHash()
    }

    suspend fun findSplTokenAddressData(
        destinationAddress: PublicKey,
        mintAddress: String,
    ): TransactionAddressData {
        val associatedAddress = try {
            findSplTokenAddress(destinationAddress, mintAddress)
        } catch (e: IllegalStateException) {
            throw IllegalStateException("Invalid owner address")
        }

        /* If account is not found, create one */
        val accountInfo = getAccountInfo(associatedAddress.toBase58())
        val value = accountInfo?.value
        val accountExists = value?.owner == TokenProgram.PROGRAM_ID.toString() && value.data != null
        return TransactionAddressData(
            associatedAddress,
            !accountExists
        )
    }

    suspend fun getAccountInfo(account: String): AccountInfo? {
        return ParticleNetwork.solana.getAccountInfo(account)
    }

    suspend fun findSplTokenAddress(
        destinationAddress: PublicKey,
        mintAddress: String
    ): PublicKey {
        val accountInfo = getAccountInfo(destinationAddress.toBase58())
        val info = TokenTransaction.parseAccountInfoData(accountInfo, TokenProgram.PROGRAM_ID)

        // create associated token address
        val value = accountInfo?.value
        if (value == null || value.data?.get(0).isNullOrEmpty()) {
            return TokenTransaction.getAssociatedTokenAddress(
                PublicKey(mintAddress),
                destinationAddress
            )
        }

        // detect if destination address is already a SPLToken address
        if (info?.mint == destinationAddress) {
            return destinationAddress
        }

        // detect if destination address is a SOL address
        if (info?.owner?.toBase58() == TokenProgram.PROGRAM_ID.toBase58()) {
            // create associated token address
            return TokenTransaction.getAssociatedTokenAddress(
                PublicKey(mintAddress),
                destinationAddress
            )
        }

        throw IllegalStateException("Wallet address is not valid")
    }

    suspend fun getTokenAccountsByOwner(owner: String): TokenAccounts {
        return ParticleNetwork.solana.getTokenAccountsByOwner(owner).result
    }

}