package com.particle.api.infrastructure.net.interceptor

import com.blankj.utilcode.util.GsonUtils
import com.blankj.utilcode.util.LogUtils
import com.particle.api.infrastructure.net.data.NetworkException
import com.particle.api.infrastructure.net.data.ServerError
import com.particle.api.infrastructure.net.data.ServerException
import okhttp3.Interceptor
import okhttp3.Request
import okhttp3.Response
import okhttp3.ResponseBody.Companion.toResponseBody
import okio.Buffer
import org.json.JSONException
import java.io.IOException
import java.net.URLDecoder
import java.nio.charset.Charset


internal class ErrorInterceptor : Interceptor {

    override fun intercept(chain: Interceptor.Chain): Response {
        val request = chain.request()
        val response = chain.proceed(request)
        return if (response.isSuccessful && response.networkResponse?.code != 304) {
            handleResponse(request, response)
        } else {
            throw extractGeneralException(response.body!!.string())
        }
    }

    private fun handleResponse(request: Request, response: Response): Response {
        val responseBody = try {
            response.body!!.string()
        } catch (e: Exception) {
            throw IOException("Error parsing response body", e)
        }

        if (responseBody.isEmpty()) {
            throw Exception("Data is empty")
        }
        val errorContent = GsonUtils.fromJson(responseBody, ServerError::class.java)
        errorContent.error?.reqUrl = response.request.url.toString()
        try {
            val buffer = Buffer()
            request.body?.writeTo(buffer)
            val requestData = URLDecoder.decode(buffer.readString(Charset.forName("UTF-8")).trim(), "utf-8")
            errorContent.error?.reqBody = requestData
        } catch (e: Exception) {
            //ignore
        }

        if (errorContent.error != null) {
            throw ServerException(errorContent.error)
        }

        return try {
            createResponse(response, responseBody)
        } catch (e: JSONException) {
            throw IllegalStateException("Error parsing data", e)
        }
    }

    private fun createResponse(response: Response, responseBody: String): Response =
        response.newBuilder().body(responseBody.toResponseBody()).build()

    private fun extractGeneralException(bodyString: String): Throwable {
        if (bodyString.isEmpty()) {
            return NetworkException("Network error")
        }
        if (bodyString.length > 1024) {
            return NetworkException("Network error")
        }
        return NetworkException(bodyString)
    }

}