package com.particle.api.infrastructure.net.download

import androidx.annotation.Keep
import com.blankj.utilcode.util.EncryptUtils
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.flowOn
import okhttp3.OkHttpClient
import okhttp3.Request
import java.io.BufferedInputStream
import java.io.File
import java.io.FileOutputStream

typealias download_error = suspend (Throwable) -> Unit
typealias download_process = suspend (downloadedSize: Long, length: Long, progress: Float) -> Unit
typealias download_success = suspend (uri: File) -> Unit

@Keep
object KCHttpV2 {
    private var okHttpClient: OkHttpClient = OkHttpClient()

    @JvmOverloads
    suspend fun download(
        url: String, outputDir: String,
        onError: download_error = {},
        onProcess: download_process = { _, _, _ -> },
        onSuccess: download_success = { }
    ) {
        flow {
            try {
                var outputFileName = EncryptUtils.encryptMD5ToString(url)
                val file = File(outputDir, "$outputFileName")
                if (file.exists()) {
                    emit(HttpResult.success(file))
                } else {
                    var request = Request.Builder().url(url).build()
                    var resp = okHttpClient.newCall(request).execute()
                    val body = resp.body!!
                    val contentLength = body.contentLength()
                    val inputStream = body.byteStream()

                    val fileTmp = File(outputDir, "$outputFileName.tmp")
                    if (fileTmp.exists()) fileTmp.delete()

                    val outputStream = FileOutputStream(fileTmp)
                    var currentLength = 0
                    val bufferSize = 1024 * 8
                    val buffer = ByteArray(bufferSize)
                    val bufferedInputStream = BufferedInputStream(inputStream, bufferSize)
                    var readLength: Int
                    while (bufferedInputStream.read(buffer, 0, bufferSize)
                            .also { readLength = it } != -1
                    ) {
                        outputStream.write(buffer, 0, readLength)
                        outputStream.flush()
                        currentLength += readLength
                        emit(
                            HttpResult.progress(
                                currentLength.toLong(),
                                contentLength,
                                currentLength.toFloat() / contentLength.toFloat()
                            )
                        )
                    }
                    bufferedInputStream.close()
                    outputStream.close()
                    inputStream.close()
                    fileTmp.renameTo(file)
                    emit(HttpResult.success(file))
                }
            } catch (e: Exception) {
                emit(HttpResult.failure<File>(e))
            }
        }.flowOn(Dispatchers.IO)
            .collect {
                it.fold(onFailure = { e ->
                    e?.let { it1 -> onError(it1) }
                }, onSuccess = { file ->
                    onSuccess(file)
                }, onLoading = { progress ->
                    onProcess(progress.currentLength, progress.length, progress.process)
                })
            }
    }
}
