package com.particle.api.infrastructure.db.dao

import androidx.lifecycle.LiveData
import androidx.paging.PagingSource
import androidx.room.*
import androidx.room.OnConflictStrategy.Companion.REPLACE
import com.particle.api.infrastructure.db.result.TokenInfoJoinSplTokenRates
import com.particle.api.infrastructure.db.table.TokenInfo

@Dao
interface TokenInfoDao {

    @Query(
        "SELECT token.*,spl.symbol,spl.logoURI,rates.rate,rates.change24 FROM token_info as token " +
                "LEFT JOIN spl_token spl on spl.chainName = token.chainName and spl.chainId = token.chainId and upper(token.address) = upper(spl.address) " +
                "LEFT JOIN rates on rates.chainName = token.chainName and rates.chainId = token.chainId and upper(token.address) = upper(rates.address)" +
                " where (token.displayState = 1 or token.displayState = 2) and upper(token.publicKey) = upper(:publicKey) and token.chainName=:chainName and token.chainId=:chainId and upper(token.address)=upper(:address)"
    )
    suspend fun getTokenInfoJoinByAddress(
        publicKey: String,
        chainName: String,
        chainId: Long,
        address: String
    ): TokenInfoJoinSplTokenRates

    @RewriteQueriesToDropUnusedColumns
    @Query(
        "SELECT token.*,spl.symbol,spl.logoURI,rates.rate,rates.change24 FROM token_info as token " +
                "LEFT JOIN spl_token spl on spl.chainName = token.chainName and spl.chainId = token.chainId and upper(token.address) = upper(spl.address) " +
                "LEFT JOIN rates on rates.chainName = token.chainName and rates.chainId = token.chainId and upper(token.address) = upper(rates.address)" +
                " where (token.displayState = 1 or token.displayState = 2 ) and upper(token.publicKey) = upper(:publicKey) and token.chainName=:chainName and token.chainId=:chainId and spl.source!=-1 "
    )
    fun getAllSelectedListLiveData(
        publicKey: String,
        chainName: String,
        chainId: Long
    ): LiveData<List<TokenInfoJoinSplTokenRates>>

    @Query(
        "SELECT token.*,spl.symbol,spl.logoURI,rates.rate,rates.change24 FROM token_info as token " +
                "LEFT JOIN spl_token spl on spl.chainName = token.chainName and spl.chainId = token.chainId and upper(token.address) = upper(spl.address) " +
                "LEFT JOIN rates on rates.chainName = token.chainName and rates.chainId = token.chainId and token.address = rates.address" +
                " where (token.displayState = 1 or token.displayState = 2 ) and upper(token.publicKey) = upper(:publicKey) and token.chainName=:chainName and token.chainId=:chainId"
    )
    suspend fun getAllSelectedListOnce(
        publicKey: String,
        chainName: String,
        chainId: Long
    ): List<TokenInfoJoinSplTokenRates>

    @RewriteQueriesToDropUnusedColumns
    @Query(
        "SELECT token.*,token.amount*rates.rate/token.divideDecimals total,CAST(IFNULL(token.amount,0) AS FLOAT)/IFNULL(token.divideDecimals,1) sortAmount,spl.symbol,spl.logoURI,rates.rate,rates.change24 FROM token_info as token " +
                "LEFT JOIN spl_token spl on spl.chainName = token.chainName and spl.chainId = token.chainId and upper(token.address) = upper(spl.address) " +
                "LEFT JOIN rates on rates.chainName = token.chainName and rates.chainId = token.chainId and upper(token.address) = upper(rates.address)" +
                " where (token.displayState = 1 or token.displayState = 2) and upper(token.publicKey) = upper(:publicKey) and token.chainName=:chainName and token.chainId=:chainId and spl.source in (:sources)" +
                " order by token.address ='native' desc,total desc, sortAmount desc,token.updateTime desc "

    )
    fun getAllSelectedList(
        publicKey: String,
        chainName: String,
        chainId: Long,
        sources: List<Int>
    ): PagingSource<Int, TokenInfoJoinSplTokenRates>

    @RewriteQueriesToDropUnusedColumns
    @Query(
        "SELECT token.*,token.amount*rates.rate/token.divideDecimals total,CAST(IFNULL(token.amount,0) AS FLOAT)/IFNULL(token.divideDecimals,1) sortAmount,spl.symbol,spl.logoURI,rates.rate,rates.change24 FROM token_info as token " +
                "LEFT JOIN spl_token spl on spl.chainName = token.chainName and spl.chainId = token.chainId and upper(token.address) = upper(spl.address) " +
                "LEFT JOIN rates on rates.chainName = token.chainName and rates.chainId = token.chainId and upper(token.address) = upper(rates.address)" +
                " where (token.displayState = 1 or token.displayState = 2) and upper(token.publicKey) = upper(:publicKey) and token.chainName=:chainName and token.chainId=:chainId" +
                " and token.address in  (:displayTokenList) and spl.source in (:sources)" +
                " order by token.address ='native' desc,total desc, sortAmount desc,token.updateTime desc "

    )
    fun getAllSelectedListByDisplayTokenList(
        publicKey: String,
        chainName: String,
        chainId: Long,
        displayTokenList: List<String>,
        sources: List<Int>
    ): PagingSource<Int, TokenInfoJoinSplTokenRates>


    @RewriteQueriesToDropUnusedColumns
    @Query(
        "SELECT token.*,token.amount*rates.rate/token.divideDecimals total,CAST(IFNULL(token.amount,0) AS FLOAT)/IFNULL(token.divideDecimals,1) sortAmount,spl.symbol,spl.logoURI,rates.rate,rates.change24 FROM token_info as token " +
                "LEFT JOIN spl_token spl on spl.chainName = token.chainName and spl.chainId = token.chainId and upper(token.address) = upper(spl.address) " +
                "LEFT JOIN rates on rates.chainName = token.chainName and rates.chainId = token.chainId and upper(token.address) = upper(rates.address)" +
                " where (token.displayState = 1 or token.displayState = 2) and upper(token.publicKey) = upper(:publicKey) and token.chainName=:chainName and token.chainId=:chainId  and spl.source in (:sources)" +
                " order by token.address ='native' desc,token.address in (:priorityTokenList) desc ,total desc, sortAmount desc,token.updateTime desc "

    )
    fun getAllSelectedListByPriorityTokenList(
        publicKey: String,
        chainName: String,
        chainId: Long,
        priorityTokenList: List<String>,
        sources: List<Int>
    ): PagingSource<Int, TokenInfoJoinSplTokenRates>


    @RewriteQueriesToDropUnusedColumns
    @Query(
        "SELECT token.*, token.amount*rates.rate total,symbol,spl.logoURI,rates.rate,rates.change24 FROM token_info as token " +
                "LEFT JOIN spl_token spl on spl.chainName = token.chainName and spl.chainId = token.chainId and upper(token.address) = upper(spl.address) " +
                "LEFT JOIN rates on rates.chainName = token.chainName and rates.chainId = token.chainId and upper(token.address) = upper(rates.address)" +
                " where (token.displayState = 1 or token.displayState = 2) and upper(token.address)!=upper(:ignoreAddress) and upper(token.publicKey) = upper(:publicKey) and token.chainName=:chainName and token.chainId=:chainId" +
                " and (spl.symbol like '%'||:symbolName||'%' or IFNULL(spl.name,token.address) like '%'||:symbolName||'%') " +
                "and spl.source in (:sources) " +
                " order by token.address ='native' desc,total desc, token.amount desc,token.updateTime desc "
    )
    fun getAllSelectedListByLike(
        publicKey: String,
        chainName: String,
        chainId: Long,
        symbolName: String,
        ignoreAddress: String,
        sources: List<Int>
    ): PagingSource<Int, TokenInfoJoinSplTokenRates>


    @Query(
        "SELECT token.*,spl.symbol,spl.logoURI,rates.rate,rates.change24 FROM token_info as token " +
                "LEFT JOIN spl_token as spl on spl.chainName = token.chainName and spl.chainId = token.chainId and upper(token.address) = upper(spl.address) " +
                "LEFT JOIN rates on rates.chainName = token.chainName and rates.chainId = token.chainId and token.address = rates.address" +
                " where upper(token.publicKey) = upper(:publicKey) and token.chainName=:chainName and token.chainId=:chainId" +
                " and (spl.symbol like '%'||:symbolName||'%' or IFNULL(spl.name,token.address) like '%'||:symbolName||'%') ORDER BY  token.updateTime desc"
    )
    fun getAllTokenListByLike(
        publicKey: String,
        chainName: String,
        chainId: Long,
        symbolName: String
    ): PagingSource<Int, TokenInfoJoinSplTokenRates>


    @Query("SELECT * FROM token_info where (displayState = 1 or displayState = 2) and upper(publicKey)=upper(:publicKey) and chainName=:chainName and chainId=:chainId")
    suspend fun getAllDisplayed(
        publicKey: String,
        chainName: String,
        chainId: Long,
    ): List<TokenInfo>

    @Query("update token_info set displayState =:state,updateTime=:updateTime where upper(publicKey) =upper( :publicKey)  and chainName = :chainName and chainId =:chainId and address =:address ")
    suspend fun updateState(
        publicKey: String,
        chainName: String,
        chainId: Long,
        address: String,
        state: Int,
        updateTime: Long
    )

    @Query("update token_info set displayState = -2 where displayState = -1 or displayState = 0 ")
    suspend fun updateDisplayStatus1()

    @Query("update token_info set displayState = 2 where displayState = 1")
    suspend fun updateDisplayStatus2()

    @Query("update token_info set displayState = 2 where displayState = -3")
    suspend fun updateHiddenStatus()

    @Query("update token_info set displayState = -2 where address = :address")
    suspend fun hidden(address: String)

    @Query("SELECT * FROM token_info where upper(publicKey) = upper(:publicKey) and chainName = :chainName and chainId = :chainId and address = :address ")
    suspend fun getWalletTokenByAddress(
        publicKey: String,
        chainName: String,
        chainId: Long,
        address: String
    ): TokenInfo


    @Update(onConflict = REPLACE)
    suspend fun updateAll(token: List<TokenInfo>)

    @Insert(onConflict = REPLACE)
    suspend fun insert(token: TokenInfo)

    @Insert(onConflict = REPLACE)
    suspend fun insertAll(tokens: List<TokenInfo>)

    @Query(
        "SELECT token.*,spl.symbol,spl.logoURI,rates.rate,rates.change24 FROM token_info as token " +
                "LEFT JOIN spl_token spl on spl.chainName = token.chainName and spl.chainId = token.chainId and upper(token.address) = upper(spl.address) " +
                "LEFT JOIN rates on rates.chainName = token.chainName and rates.chainId = token.chainId and token.address = rates.address" +
                " where (token.displayState = 1 or token.displayState = 2) and upper(token.publicKey) = upper(:publicKey) and token.chainName=:chainName and token.chainId=:chainId" +
                " ORDER BY  token.updateTime desc"
    )
    suspend fun getAllTokenInfo(
        publicKey: String,
        chainName: String,
        chainId: Long,
    ): List<TokenInfoJoinSplTokenRates>

    @Query("SELECT count(address) FROM token_info where (displayState = 1 or displayState = 2) and upper(publicKey)=upper(:publicKey) and chainName=:chainName and chainId=:chainId")
    suspend fun getAllDisplayedCount(
        publicKey: String,
        chainName: String,
        chainId: Long,
    ): Int

    @Query("SELECT EXISTS (SELECT address from token_info where chainName=:chainName and chainId=:chainId  and displayState = :displayState)")
    suspend fun isSeparateAddressExist(chainName: String, chainId: Long, displayState: Int): Boolean

    //delete token by chainName chainId publickey
    @Query("delete from token_info where chainName=:chainName and chainId=:chainId and upper(publicKey)=upper(:publicKey) and address=:address")
    suspend fun deleteToken(
        publicKey: String,
        chainName: String,
        chainId: Long,
        address: String
    )

    @Insert(onConflict = REPLACE)
    suspend fun insertSeparateAddress(token: TokenInfo)


    @RewriteQueriesToDropUnusedColumns
    @Query(
        "SELECT token.*, token.amount*rates.rate total,symbol,spl.logoURI,rates.rate,rates.change24 FROM token_info as token " +
                "LEFT JOIN spl_token spl on spl.chainName = token.chainName and spl.chainId = token.chainId and upper(token.address) = upper(spl.address) " +
                "LEFT JOIN rates on rates.chainName = token.chainName and rates.chainId = token.chainId and upper(token.address) = upper(rates.address)" +
                " where (token.displayState = 1 or token.displayState = 2)  and upper(token.publicKey) = upper(:publicKey) and token.chainName=:chainName and token.chainId=:chainId" +
                " and spl.source in (:sources) " +
                " order by token.address ='native' desc,total desc, token.amount desc,token.updateTime desc "
    )
    suspend fun getAllTokenChoiceList(
        publicKey: String,
        chainName: String,
        chainId: Long,
        sources: List<Int>
    ): List<TokenInfoJoinSplTokenRates>
}
