/*
	Apache License
	Version 2.0, January 2004
	http://www.apache.org/licenses/

	TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION

	1. Definitions.

	"License" shall mean the terms and conditions for use, reproduction, and
	distribution as defined by Sections 1 through 9 of this document.

	"Licensor" shall mean the copyright owner or entity authorized by the copyright
	owner that is granting the License.

	"Legal Entity" shall mean the union of the acting entity and all other entities
	that control, are controlled by, or are under common control with that entity.
	For the purposes of this definition, "control" means (i) the power, direct or
	indirect, to cause the direction or management of such entity, whether by
	contract or otherwise, or (ii) ownership of fifty percent (50%) or more of the
	outstanding shares, or (iii) beneficial ownership of such entity.

	"You" (or "Your") shall mean an individual or Legal Entity exercising
	permissions granted by this License.

	"Source" form shall mean the preferred form for making modifications, including
	but not limited to software source code, documentation source, and configuration
	files.

	"Object" form shall mean any form resulting from mechanical transformation or
	translation of a Source form, including but not limited to compiled object code,
	generated documentation, and conversions to other media types.

	"Work" shall mean the work of authorship, whether in Source or Object form, made
	available under the License, as indicated by a copyright notice that is included
	in or attached to the work (an example is provided in the Appendix below).

	"Derivative Works" shall mean any work, whether in Source or Object form, that
	is based on (or derived from) the Work and for which the editorial revisions,
	annotations, elaborations, or other modifications represent, as a whole, an
	original work of authorship. For the purposes of this License, Derivative Works
	shall not include works that remain separable from, or merely link (or bind by
	name) to the interfaces of, the Work and Derivative Works thereof.

	"Contribution" shall mean any work of authorship, including the original version
	of the Work and any modifications or additions to that Work or Derivative Works
	thereof, that is intentionally submitted to Licensor for inclusion in the Work
	by the copyright owner or by an individual or Legal Entity authorized to submit
	on behalf of the copyright owner. For the purposes of this definition,
	"submitted" means any form of electronic, verbal, or written communication sent
	to the Licensor or its representatives, including but not limited to
	communication on electronic mailing lists, source code control systems, and
	issue tracking systems that are managed by, or on behalf of, the Licensor for
	the purpose of discussing and improving the Work, but excluding communication
	that is conspicuously marked or otherwise designated in writing by the copyright
	owner as "Not a Contribution."

	"Contributor" shall mean Licensor and any individual or Legal Entity on behalf
	of whom a Contribution has been received by Licensor and subsequently
	incorporated within the Work.

	2. Grant of Copyright License.

	Subject to the terms and conditions of this License, each Contributor hereby
	grants to You a perpetual, worldwide, non-exclusive, no-charge, royalty-free,
	irrevocable copyright license to reproduce, prepare Derivative Works of,
	publicly display, publicly perform, sublicense, and distribute the Work and such
	Derivative Works in Source or Object form.

	3. Grant of Patent License.

	Subject to the terms and conditions of this License, each Contributor hereby
	grants to You a perpetual, worldwide, non-exclusive, no-charge, royalty-free,
	irrevocable (except as stated in this section) patent license to make, have
	made, use, offer to sell, sell, import, and otherwise transfer the Work, where
	such license applies only to those patent claims licensable by such Contributor
	that are necessarily infringed by their Contribution(s) alone or by combination
	of their Contribution(s) with the Work to which such Contribution(s) was
	submitted. If You institute patent litigation against any entity (including a
	cross-claim or counterclaim in a lawsuit) alleging that the Work or a
	Contribution incorporated within the Work constitutes direct or contributory
	patent infringement, then any patent licenses granted to You under this License
	for that Work shall terminate as of the date such litigation is filed.

	4. Redistribution.

	You may reproduce and distribute copies of the Work or Derivative Works thereof
	in any medium, with or without modifications, and in Source or Object form,
	provided that You meet the following conditions:

	You must give any other recipients of the Work or Derivative Works a copy of
	this License; and
	You must cause any modified files to carry prominent notices stating that You
	changed the files; and
	You must retain, in the Source form of any Derivative Works that You distribute,
	all copyright, patent, trademark, and attribution notices from the Source form
	of the Work, excluding those notices that do not pertain to any part of the
	Derivative Works; and
	If the Work includes a "NOTICE" text file as part of its distribution, then any
	Derivative Works that You distribute must include a readable copy of the
	attribution notices contained within such NOTICE file, excluding those notices
	that do not pertain to any part of the Derivative Works, in at least one of the
	following places: within a NOTICE text file distributed as part of the
	Derivative Works; within the Source form or documentation, if provided along
	with the Derivative Works; or, within a display generated by the Derivative
	Works, if and wherever such third-party notices normally appear. The contents of
	the NOTICE file are for informational purposes only and do not modify the
	License. You may add Your own attribution notices within Derivative Works that
	You distribute, alongside or as an addendum to the NOTICE text from the Work,
	provided that such additional attribution notices cannot be construed as
	modifying the License.
	You may add Your own copyright statement to Your modifications and may provide
	additional or different license terms and conditions for use, reproduction, or
	distribution of Your modifications, or for any such Derivative Works as a whole,
	provided Your use, reproduction, and distribution of the Work otherwise complies
	with the conditions stated in this License.

	5. Submission of Contributions.

	Unless You explicitly state otherwise, any Contribution intentionally submitted
	for inclusion in the Work by You to the Licensor shall be under the terms and
	conditions of this License, without any additional terms or conditions.
	Notwithstanding the above, nothing herein shall supersede or modify the terms of
	any separate license agreement you may have executed with Licensor regarding
	such Contributions.

	6. Trademarks.

	This License does not grant permission to use the trade names, trademarks,
	service marks, or product names of the Licensor, except as required for
	reasonable and customary use in describing the origin of the Work and
	reproducing the content of the NOTICE file.

	7. Disclaimer of Warranty.

	Unless required by applicable law or agreed to in writing, Licensor provides the
	Work (and each Contributor provides its Contributions) on an "AS IS" BASIS,
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied,
	including, without limitation, any warranties or conditions of TITLE,
	NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A PARTICULAR PURPOSE. You are
	solely responsible for determining the appropriateness of using or
	redistributing the Work and assume any risks associated with Your exercise of
	permissions under this License.

	8. Limitation of Liability.

	In no event and under no legal theory, whether in tort (including negligence),
	contract, or otherwise, unless required by applicable law (such as deliberate
	and grossly negligent acts) or agreed to in writing, shall any Contributor be
	liable to You for damages, including any direct, indirect, special, incidental,
	or consequential damages of any character arising as a result of this License or
	out of the use or inability to use the Work (including but not limited to
	damages for loss of goodwill, work stoppage, computer failure or malfunction, or
	any and all other commercial damages or losses), even if such Contributor has
	been advised of the possibility of such damages.

	9. Accepting Warranty or Additional Liability.

	While redistributing the Work or Derivative Works thereof, You may choose to
	offer, and charge a fee for, acceptance of support, warranty, indemnity, or
	other liability obligations and/or rights consistent with this License. However,
	in accepting such obligations, You may act only on Your own behalf and on Your
	sole responsibility, not on behalf of any other Contributor, and only if You
	agree to indemnify, defend, and hold each Contributor harmless for any liability
	incurred by, or claims asserted against, such Contributor by reason of your
	accepting any such warranty or additional liability.

	END OF TERMS AND CONDITIONS

	APPENDIX: How to apply the Apache License to your work

	To apply the Apache License to your work, attach the following boilerplate
	notice, with the fields enclosed by brackets "{}" replaced with your own
	identifying information. (Don't include the brackets!) The text should be
	enclosed in the appropriate comment syntax for the file format. We also
	recommend that a file or class name and description of purpose be included on
	the same "printed page" as the copyright notice for easier identification within
	third-party archives.

	   Copyright 2020 t-io

	   Licensed under the Apache License, Version 2.0 (the "License");
	   you may not use this file except in compliance with the License.
	   You may obtain a copy of the License at

	     http://www.apache.org/licenses/LICENSE-2.0

	   Unless required by applicable law or agreed to in writing, software
	   distributed under the License is distributed on an "AS IS" BASIS,
	   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
	   See the License for the specific language governing permissions and
	   limitations under the License.
*/
package org.tio.utils.hutool;

import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Objects;

/**
 * 代码移植自<a href="https://github.com/biezhi/blade">blade</a><br>
 * 快速缓冲，将数据存放在缓冲集中，取代以往的单一数组
 *
 * @author biezhi、looly、L.cm
 * @since 1.0
 */
public class FastByteBuffer {
	/**
	 * 一个缓冲区的最小字节数
	 */
	private final int minChunkLen;
	/**
	 * 缓冲集
	 */
	private byte[][] buffers = new byte[16][];
	/**
	 * 缓冲数
	 */
	private int buffersCount;
	/**
	 * 当前缓冲索引
	 */
	private int currentBufferIndex = -1;
	/**
	 * 当前缓冲
	 */
	private byte[] currentBuffer;
	/**
	 * 当前缓冲偏移量
	 */
	private int offset;
	/**
	 * 缓冲字节数
	 */
	private int size;

	public FastByteBuffer() {
		this.minChunkLen = 1024;
	}

	public FastByteBuffer(int size) {
		this.minChunkLen = Math.abs(size);
	}

	/**
	 * 分配下一个缓冲区，不会小于1024
	 *
	 * @param newSize 理想缓冲区字节数
	 */
	private void needNewBuffer(int newSize) {
		int delta = newSize - size;
		int newBufferSize = Math.max(minChunkLen, delta);
		currentBufferIndex++;
		currentBuffer = new byte[newBufferSize];
		offset = 0;
		// add buffer
		if (currentBufferIndex >= buffers.length) {
			int newLen = buffers.length << 1;
			byte[][] newBuffers = new byte[newLen][];
			System.arraycopy(buffers, 0, newBuffers, 0, buffers.length);
			buffers = newBuffers;
		}
		buffers[currentBufferIndex] = currentBuffer;
		buffersCount++;
	}

	/**
	 * 向快速缓冲加入数据
	 *
	 * @param array 数据
	 * @param off   偏移量
	 * @param len   字节数
	 * @return 快速缓冲自身 @see FastByteBuffer
	 */
	public FastByteBuffer append(byte[] array, int off, int len) {
		int end = off + len;
		if ((off < 0) || (len < 0) || (end > array.length)) {
			throw new IndexOutOfBoundsException();
		}
		if (len == 0) {
			return this;
		}
		int newSize = size + len;
		int remaining = len;

		if (currentBuffer != null) {
			// first try to fill current buffer
			int part = Math.min(remaining, currentBuffer.length - offset);
			System.arraycopy(array, end - remaining, currentBuffer, offset, part);
			remaining -= part;
			offset += part;
			size += part;
		}

		if (remaining > 0) {
			// still some data left
			// ask for new buffer
			needNewBuffer(newSize);

			// then copy remaining
			// but this time we are sure that it will fit
			int part = Math.min(remaining, currentBuffer.length - offset);
			System.arraycopy(array, end - remaining, currentBuffer, offset, part);
			offset += part;
			size += part;
		}
		return this;
	}

	/**
	 * 向快速缓冲加入数据
	 *
	 * @param buffer 数据
	 * @return 快速缓冲自身 @see FastByteBuffer
	 */
	public FastByteBuffer append(ByteBuffer buffer) {
		return append(buffer.array());
	}

	/**
	 * 向快速缓冲加入数据
	 *
	 * @param array 数据
	 * @return 快速缓冲自身 @see FastByteBuffer
	 */
	public FastByteBuffer append(byte[] array) {
		return append(array, 0, array.length);
	}

	/**
	 * 向快速缓冲加入一个字节
	 *
	 * @param element 一个字节的数据
	 * @return 快速缓冲自身 @see FastByteBuffer
	 */
	public FastByteBuffer append(byte element) {
		if ((currentBuffer == null) || (offset == currentBuffer.length)) {
			needNewBuffer(size + 1);
		}
		currentBuffer[offset] = element;
		offset++;
		size++;
		return this;
	}

	/**
	 * 将另一个快速缓冲加入到自身
	 *
	 * @param buff 快速缓冲
	 * @return 快速缓冲自身 @see FastByteBuffer
	 */
	public FastByteBuffer append(FastByteBuffer buff) {
		if (buff.size == 0) {
			return this;
		}
		for (int i = 0; i < buff.currentBufferIndex; i++) {
			append(buff.buffers[i]);
		}
		append(buff.currentBuffer, 0, buff.offset);
		return this;
	}

	/**
	 * 写出一个 boolean 值，true 为 1，false 为 0
	 *
	 * @param value boolean value
	 * @return FastByteBuffer
	 */
	public FastByteBuffer writeBoolean(boolean value) {
		return writeByte(value ? 1 : 0);
	}

	/**
	 * 写出一个byte类型的数据
	 *
	 * @param value 待写出数值
	 * @return FastByteBuffer
	 */
	public FastByteBuffer writeByte(int value) {
		return this.append((byte) value);
	}

	/**
	 * 写出 short，大端模式
	 *
	 * @param value 待写出数值
	 * @return FastByteBuffer
	 */
	public FastByteBuffer writeShortBE(int value) {
		byte[] bytes = new byte[2];
		bytes[0] = (byte) (value >>> 8);
		bytes[1] = (byte) (value);
		return this.append(bytes, 0, 2);
	}

	/**
	 * 写出 short，小端模式
	 *
	 * @param value 待写出数值
	 * @return FastByteBuffer
	 */
	public FastByteBuffer writeShortLE(int value) {
		byte[] bytes = new byte[2];
		bytes[0] = (byte) (value);
		bytes[1] = (byte) (value >>> 8);
		return this.append(bytes, 0, 2);
	}

	/**
	 * 写出 3 个字节，大端模式
	 *
	 * @param value 待写出数值
	 * @return FastByteBuffer
	 */
	public FastByteBuffer writeMediumBE(int value) {
		byte[] bytes = new byte[3];
		bytes[0] = (byte) (value >>> 16);
		bytes[1] = (byte) (value >>> 8);
		bytes[2] = (byte) (value);
		return this.append(bytes, 0, 3);
	}

	/**
	 * 写出 3 个字节，小端模式
	 *
	 * @param value 待写出数值
	 * @return FastByteBuffer
	 */
	public FastByteBuffer writeMediumLE(int value) {
		byte[] bytes = new byte[3];
		bytes[0] = (byte) (value);
		bytes[1] = (byte) (value >>> 8);
		bytes[2] = (byte) (value >>> 16);
		return this.append(bytes, 0, 3);
	}

	/**
	 * 写出 4 个字节，大端模式
	 *
	 * @param value 待写出数值
	 * @return FastByteBuffer
	 */
	public FastByteBuffer writeIntBE(long value) {
		byte[] bytes = new byte[4];
		bytes[0] = (byte) (value >>> 24);
		bytes[1] = (byte) (value >>> 16);
		bytes[2] = (byte) (value >>> 8);
		bytes[3] = (byte) (value);
		return this.append(bytes, 0, 4);
	}

	/**
	 * 写出 4 个字节，小端模式
	 *
	 * @param value 待写出数值
	 * @return FastByteBuffer
	 */
	public FastByteBuffer writeIntLE(long value) {
		byte[] bytes = new byte[4];
		bytes[0] = (byte) (value);
		bytes[1] = (byte) (value >>> 8);
		bytes[2] = (byte) (value >>> 16);
		bytes[3] = (byte) (value >>> 24);
		return this.append(bytes, 0, 4);
	}

	/**
	 * 写出 8 个字节，大端模式
	 *
	 * @param value 待写出数值
	 * @return FastByteBuffer
	 */
	public FastByteBuffer writeLongBE(long value) {
		byte[] bytes = new byte[8];
		bytes[0] = (byte) (value >>> 56);
		bytes[1] = (byte) (value >>> 48);
		bytes[2] = (byte) (value >>> 40);
		bytes[3] = (byte) (value >>> 32);
		bytes[4] = (byte) (value >>> 24);
		bytes[5] = (byte) (value >>> 16);
		bytes[6] = (byte) (value >>> 8);
		bytes[7] = (byte) value;
		return this.append(bytes, 0, 8);
	}

	/**
	 * 写出 8 个字节，小端模式
	 *
	 * @param value 待写出数值
	 * @return FastByteBuffer
	 */
	public FastByteBuffer writeLongLE(long value) {
		byte[] bytes = new byte[8];
		bytes[0] = (byte) value;
		bytes[1] = (byte) (value >>> 8);
		bytes[2] = (byte) (value >>> 16);
		bytes[3] = (byte) (value >>> 24);
		bytes[4] = (byte) (value >>> 32);
		bytes[5] = (byte) (value >>> 40);
		bytes[6] = (byte) (value >>> 48);
		bytes[7] = (byte) (value >>> 56);
		return this.append(bytes, 0, 8);
	}

	/**
	 * 写出单精度浮点
	 *
	 * @param value value
	 * @return FastByteBuffer
	 */
	public FastByteBuffer writeFloatBE(float value) {
		return writeIntBE(Float.floatToRawIntBits(value));
	}

	/**
	 * 写出单精度浮点
	 *
	 * @param value value
	 * @return FastByteBuffer
	 */
	public FastByteBuffer writeFloatLE(float value) {
		return writeIntLE(Float.floatToRawIntBits(value));
	}

	/**
	 * 写出双精度浮点
	 *
	 * @param value value
	 * @return FastByteBuffer
	 */
	public FastByteBuffer writeDoubleBE(double value) {
		return writeLongBE(Double.doubleToRawLongBits(value));
	}

	/**
	 * 写出双精度浮点
	 *
	 * @param value value
	 * @return FastByteBuffer
	 */
	public FastByteBuffer writeDoubleLE(double value) {
		return writeLongLE(Double.doubleToRawLongBits(value));
	}

	/**
	 * 写出一个bytes类型的数据
	 *
	 * @param bytes 待写出数值
	 * @return FastByteBuffer
	 */
	public FastByteBuffer writeBytes(byte[] bytes) {
		return this.append(bytes, 0, bytes.length);
	}

	/**
	 * 写出一个bytes类型的数据
	 *
	 * @param bytes 待写出数值
	 * @param off   off
	 * @param len   len
	 * @return FastByteBuffer
	 */
	public FastByteBuffer writeBytes(byte[] bytes, int off, int len) {
		return this.append(bytes, off, len);
	}

	/**
	 * 写可变长度整数
	 *
	 * @param num num
	 * @return FastByteBuffer
	 */
	public FastByteBuffer writeVarLengthInt(int num) {
		do {
			int digit = num % 128;
			num /= 128;
			if (num > 0) {
				digit |= 0x80;
			}
			this.append((byte) digit);
		} while (num > 0);
		return this;
	}

	/**
	 * 写出文本
	 *
	 * @param text 文本
	 */
	public void writeString(String text) {
		writeString(text, StandardCharsets.UTF_8);
	}

	/**
	 * 写出文本
	 *
	 * @param text    文本
	 * @param charset charset
	 */
	public void writeString(String text, String charset) {
		writeString(text, Charset.forName(charset));
	}

	/**
	 * 写出文本
	 *
	 * @param text    文本
	 * @param charset charset
	 */
	public void writeString(String text, Charset charset) {
		append(Objects.requireNonNull(text).getBytes(charset));
	}

	/**
	 * 设定指定位置的字节
	 *
	 * @param index index
	 * @param value value
	 * @return FastByteBuffer
	 */
	public FastByteBuffer setByte(int index, int value) {
		if ((index >= size) || (index < 0)) {
			throw new IndexOutOfBoundsException();
		}
		int ndx = 0;
		while (true) {
			byte[] b = buffers[ndx];
			if (index < b.length) {
				b[index] = (byte) value;
				return this;
			}
			ndx++;
			index -= b.length;
		}
	}

	/**
	 * 设置 boolean 值，true 为 1，false 为 0
	 *
	 * @param index index
	 * @param value boolean value
	 * @return FastByteBuffer
	 */
	public FastByteBuffer setBoolean(int index, boolean value) {
		return setByte(index, value ? 1 : 0);
	}


	/**
	 * 写出 short，大端模式
	 *
	 * @param index index
	 * @param value 待写出数值
	 * @return FastByteBuffer
	 */
	public FastByteBuffer setShortBE(int index, int value) {
		byte[] bytes = new byte[2];
		bytes[0] = (byte) (value >>> 8);
		bytes[1] = (byte) (value);
		return this.setBytes(index, bytes, 0, 2);
	}

	/**
	 * 写出 short，小端模式
	 *
	 * @param index index
	 * @param value 待写出数值
	 * @return FastByteBuffer
	 */
	public FastByteBuffer setShortLE(int index, int value) {
		byte[] bytes = new byte[2];
		bytes[0] = (byte) (value);
		bytes[1] = (byte) (value >>> 8);
		return this.setBytes(index, bytes, 0, 2);
	}

	/**
	 * 写出 3 个字节，大端模式
	 *
	 * @param index index
	 * @param value 待写出数值
	 * @return FastByteBuffer
	 */
	public FastByteBuffer setMediumBE(int index, int value) {
		byte[] bytes = new byte[3];
		bytes[0] = (byte) (value >>> 16);
		bytes[1] = (byte) (value >>> 8);
		bytes[2] = (byte) (value);
		return this.setBytes(index, bytes, 0, 3);
	}

	/**
	 * 写出 3 个字节，小端模式
	 *
	 * @param index index
	 * @param value 待写出数值
	 * @return FastByteBuffer
	 */
	public FastByteBuffer setMediumLE(int index, int value) {
		byte[] bytes = new byte[3];
		bytes[0] = (byte) (value);
		bytes[1] = (byte) (value >>> 8);
		bytes[2] = (byte) (value >>> 16);
		return this.setBytes(index, bytes, 0, 3);
	}

	/**
	 * 写出 4 个字节，大端模式
	 *
	 * @param index index
	 * @param value 待写出数值
	 * @return FastByteBuffer
	 */
	public FastByteBuffer setIntBE(int index, long value) {
		byte[] bytes = new byte[4];
		bytes[0] = (byte) (value >>> 24);
		bytes[1] = (byte) (value >>> 16);
		bytes[2] = (byte) (value >>> 8);
		bytes[3] = (byte) (value);
		return this.setBytes(index, bytes, 0, 4);
	}

	/**
	 * 写出 4 个字节，小端模式
	 *
	 * @param index index
	 * @param value 待写出数值
	 * @return FastByteBuffer
	 */
	public FastByteBuffer setIntLE(int index, long value) {
		byte[] bytes = new byte[4];
		bytes[0] = (byte) (value);
		bytes[1] = (byte) (value >>> 8);
		bytes[2] = (byte) (value >>> 16);
		bytes[3] = (byte) (value >>> 24);
		return this.setBytes(index, bytes, 0, 4);
	}

	/**
	 * 写出 8 个字节，大端模式
	 *
	 * @param index index
	 * @param value 待写出数值
	 * @return FastByteBuffer
	 */
	public FastByteBuffer setLongBE(int index, long value) {
		byte[] bytes = new byte[8];
		bytes[0] = (byte) (value >>> 56);
		bytes[1] = (byte) (value >>> 48);
		bytes[2] = (byte) (value >>> 40);
		bytes[3] = (byte) (value >>> 32);
		bytes[4] = (byte) (value >>> 24);
		bytes[5] = (byte) (value >>> 16);
		bytes[6] = (byte) (value >>> 8);
		bytes[7] = (byte) value;
		return this.setBytes(index, bytes, 0, 8);
	}

	/**
	 * 写出 8 个字节，小端模式
	 *
	 * @param index index
	 * @param value 待写出数值
	 * @return FastByteBuffer
	 */
	public FastByteBuffer setLongLE(int index, long value) {
		byte[] bytes = new byte[8];
		bytes[0] = (byte) value;
		bytes[1] = (byte) (value >>> 8);
		bytes[2] = (byte) (value >>> 16);
		bytes[3] = (byte) (value >>> 24);
		bytes[4] = (byte) (value >>> 32);
		bytes[5] = (byte) (value >>> 40);
		bytes[6] = (byte) (value >>> 48);
		bytes[7] = (byte) (value >>> 56);
		return this.setBytes(index, bytes, 0, 8);
	}

	/**
	 * 写出单精度浮点
	 *
	 * @param index index
	 * @param value value
	 * @return FastByteBuffer
	 */
	public FastByteBuffer setFloatBE(int index, float value) {
		return setIntBE(index, Float.floatToRawIntBits(value));
	}

	/**
	 * 写出单精度浮点
	 *
	 * @param index index
	 * @param value value
	 * @return FastByteBuffer
	 */
	public FastByteBuffer setFloatLE(int index, float value) {
		return setIntLE(index, Float.floatToRawIntBits(value));
	}

	/**
	 * 写出双精度浮点
	 *
	 * @param index index
	 * @param value value
	 * @return FastByteBuffer
	 */
	public FastByteBuffer setDoubleBE(int index, double value) {
		return setLongBE(index, Double.doubleToRawLongBits(value));
	}

	/**
	 * 写出双精度浮点
	 *
	 * @param index index
	 * @param value value
	 * @return FastByteBuffer
	 */
	public FastByteBuffer setDoubleLE(int index, double value) {
		return setLongLE(index, Double.doubleToRawLongBits(value));
	}

	/**
	 * 设置指定 index 的 byte
	 *
	 * @param index index
	 * @param src   src
	 * @return FastByteBuffer
	 */
	public FastByteBuffer setBytes(int index, byte[] src) {
		return setBytes(index, src, 0, src.length);
	}

	/**
	 * 设置字节
	 *
	 * @param index    index
	 * @param src      src
	 * @param srcIndex srcIndex
	 * @param length   length
	 * @return FastByteBuffer
	 */
	public FastByteBuffer setBytes(int index, byte[] src, int srcIndex, int length) {
		if ((index >= size) || (index < 0)) {
			throw new IndexOutOfBoundsException();
		}
		int ndx = 0;
		while (true) {
			byte[] buffer = buffers[ndx];
			int bufferLength = buffer.length;
			index = Math.max(index, 0);
			if (index < bufferLength) {
				if ((index + length) < bufferLength) {
					System.arraycopy(src, srcIndex, buffer, index, length);
					return this;
				} else {
					System.arraycopy(src, srcIndex, buffer, index, bufferLength - index);
					length = index + length - bufferLength;
				}
			}
			ndx++;
			index -= bufferLength;
		}
	}

	public int size() {
		return size;
	}

	public boolean isEmpty() {
		return size == 0;
	}

	/**
	 * 当前缓冲位于缓冲区的索引位
	 *
	 * @return {@link #currentBufferIndex}
	 */
	public int index() {
		return currentBufferIndex;
	}

	public int offset() {
		return offset;
	}

	/**
	 * 根据索引位返回缓冲集中的缓冲
	 *
	 * @param index 索引位
	 * @return 缓冲
	 */
	public byte[] array(int index) {
		return buffers[index];
	}

	/**
	 * 重置，用于复用
	 *
	 * @return FastByteBuffer
	 */
	public FastByteBuffer reset() {
		size = 0;
		offset = 0;
		currentBufferIndex = -1;
		currentBuffer = null;
		buffersCount = 0;
		return this;
	}

	/**
	 * 返回快速缓冲中的数据
	 *
	 * @return 快速缓冲中的数据
	 */
	public byte[] toArray() {
		int pos = 0;
		byte[] array = new byte[size];

		if (currentBufferIndex == -1) {
			return array;
		}

		for (int i = 0; i < currentBufferIndex; i++) {
			int len = buffers[i].length;
			System.arraycopy(buffers[i], 0, array, pos, len);
			pos += len;
		}

		System.arraycopy(buffers[currentBufferIndex], 0, array, pos, offset);

		return array;
	}

	/**
	 * 返回快速缓冲中的数据
	 *
	 * @param start 逻辑起始位置
	 * @param len   逻辑字节长
	 * @return 快速缓冲中的数据
	 */
	public byte[] toArray(int start, int len) {
		int remaining = len;
		int pos = 0;
		byte[] array = new byte[len];

		if (len == 0) {
			return array;
		}

		int i = 0;
		while (start >= buffers[i].length) {
			start -= buffers[i].length;
			i++;
		}

		while (i < buffersCount) {
			byte[] buf = buffers[i];
			int c = Math.min(buf.length - start, remaining);
			System.arraycopy(buf, start, array, pos, c);
			pos += c;
			remaining -= c;
			if (remaining == 0) {
				break;
			}
			start = 0;
			i++;
		}
		return array;
	}

	/**
	 * 转换成数组
	 *
	 * @return ByteBuffer
	 */
	public ByteBuffer toBuffer() {
		return ByteBuffer.wrap(this.toArray());
	}

	/**
	 * 根据索引位返回一个字节
	 *
	 * @param index 索引位
	 * @return 一个字节
	 */
	public byte get(int index) {
		if ((index >= size) || (index < 0)) {
			throw new IndexOutOfBoundsException();
		}
		int ndx = 0;
		while (true) {
			byte[] b = buffers[ndx];
			if (index < b.length) {
				return b[index];
			}
			ndx++;
			index -= b.length;
		}
	}

}
