package io.timothyheider.springrestdoc;

import com.thoughtworks.xstream.XStream;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.LineNumberReader;
import java.io.PrintWriter;
import java.io.StringWriter;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;

/**
 *
 * @author theider
 */
@Mojo(name = "write")
public class WriteWikiMojo extends AbstractMojo {

    @Parameter(defaultValue = "${project}", readonly = true, required = true)
    protected MavenProject project;

    @Parameter(name = "documentationSourcePath")
    private String documentationSourcePath;

    @Parameter(name = "outputPath")
    private String outputPath;

    @Override
    public void execute() throws MojoExecutionException, MojoFailureException {
        getLog().info("Spring REST doc write Wiki text to output " + this.outputPath + " for project " + project.getName() + " artifact:" + project.getArtifact());
        File sourceFile = new File(this.documentationSourcePath + File.separatorChar + "springrestdoc.xml");
        try (FileReader fin = new FileReader(sourceFile)) {
            XStream xstream = new XStream();
            xstream.processAnnotations(SpringRestDocIndex.class);
            xstream.processAnnotations(DocumentationPage.class);
            SpringRestDocIndex docIndex = (SpringRestDocIndex) xstream.fromXML(fin);
            getLog().info("read configuration " + docIndex);
            writeDocumentation(docIndex);
        } catch (IOException ex) {
            throw new MojoExecutionException("failed to read configuration file", ex);
        }
    }

    private void writeDocumentation(SpringRestDocIndex docIndex) throws MojoExecutionException {
        DocumentationPage homePage = docIndex.getIndexPage();
        File docOutputHome = new File(this.outputPath);
        if (!docOutputHome.exists()) {
            docOutputHome.mkdir();
            getLog().info("created new folder " + docOutputHome.getAbsolutePath());
        }
        writePage(0, null, homePage);

    }

    private void writePage(int pageLevel, PrintWriter inlineOutput, DocumentationPage page) throws MojoExecutionException {
        LineNumberReader lin = null;
        try {
            File pageFile;
            PrintWriter out;
            if (inlineOutput == null) {
                pageFile = new File(this.outputPath + File.separatorChar + page.getPath());
                try {
                    out = new PrintWriter(new FileWriter(pageFile));
                } catch (IOException ex) {
                    throw new MojoExecutionException("failed to write output", ex);
                }
            } else {
                out = inlineOutput;
            }
            getLog().info("writing page " + page.getPath());
            // open the matching file in source.
            File pageInputFile = new File(this.documentationSourcePath + File.separatorChar + page.getPath());
            lin = new LineNumberReader(new FileReader(pageInputFile));
            String text;
            do {
                text = lin.readLine();
                if (text != null) {
                    text = processTemplateText(text, page);
                    out.println(text);
                }
            } while (text != null);
            if (page.getPages() != null) {
                for (DocumentationPage p : page.getPages()) {
                    if ((p.getInline() != null) && p.getInline()) {
                        writePage(pageLevel + 1, out, p);
                    } else {
                        writePage(pageLevel + 1, null, p);
                    }
                }
            }
            out.flush();
            if (inlineOutput == null) {
                out.close();
            }
        } catch (IOException ex) {
            throw new MojoExecutionException("failed to write target", ex);
        } finally {
            try {
                if (lin != null) {
                    lin.close();
                }
            } catch (IOException ex) {
                getLog().error(ex);
            }
        }
    }

    private String processTemplateText(String text, DocumentationPage page) throws MojoExecutionException {
        int parserState = 0;
        String result = "";
        String fieldName = "";
        for (int i = 0; i < text.length(); i++) {
            char c = text.charAt(i);
            switch (parserState) {
                // SCANNING (INIT)
                case 0:
                    if (c == '{') {
                        parserState = 1;
                    } else {
                        result += c;
                    }
                    break;
                // OPENING LCURLY
                case 1:
                    if (c == '{') {
                        fieldName = "";
                        parserState = 2;
                    } else {
                        result += '{';
                        result += c;
                        parserState = 0;
                    }
                    break;
                // FIELD NAME
                case 2:
                    if (c == '}') {
                        // closing RCURLY
                        parserState = 3;
                        // process field name
                        getLog().info("including template: " + fieldName);
                        result += readIncludedFile(fieldName, page);
                    } else {
                        fieldName += c;
                    }
                    break;
                case 3:
                    if (c == '}') {
                        // yay!  the second RCURLY.  now back to scanning text.
                        parserState = 0;
                    } else {
                        throw new MojoExecutionException("unmatched LCURLY when specifying field name");
                    }
                    break;
            }
        }
        return result;
    }

    private String readIncludedFile(String fieldName, DocumentationPage page) throws MojoExecutionException {
        // build new page name based on page path
        StringWriter swriter = new StringWriter();
        PrintWriter out = new PrintWriter(swriter);        
        String sourceName = page.getPath();
        sourceName = sourceName.substring(0, sourceName.length() - 3);
        sourceName += ("_" + fieldName + ".md");
        getLog().info("template name = " + sourceName);
        File pageInputFile = new File(this.documentationSourcePath + File.separatorChar + sourceName);
        // fail softly        
        if(pageInputFile.exists()) {
            LineNumberReader lin = null;
            try {
                lin = new LineNumberReader(new FileReader(pageInputFile));
                String text;
                do {
                    text = lin.readLine();                    
                    if (text != null) {
                        // facade page for the purpose of recursive template processing.
                        DocumentationPage p = new DocumentationPage();
                        p.setInline(Boolean.TRUE);
                        p.setPath(sourceName);
                        text = processTemplateText(text, p);
                        out.println(text);
                    }
                } while (text != null);
                out.flush();
                out.close();            
            } catch (IOException ex) {
                throw new MojoExecutionException("failure reading template", ex);
            } finally {
                try {
                    if(lin != null) {
                        lin.close();
                    }
                } catch (IOException ex) {
                    throw new MojoExecutionException("failure reading template", ex);
                }
            }
            return swriter.toString();
        } else {
            return "";
        }
    }

}
