package io.timothyheider.springrestdoc;

import com.thoughtworks.xstream.XStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.List;
import org.apache.maven.artifact.DependencyResolutionRequiredException;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;
import org.springframework.web.bind.annotation.RequestMapping;

/**
 *
 * @author theider
 */
@Mojo(name = "generate")
public class GenerateTemplatesMojo extends AbstractMojo {

    @Parameter(defaultValue = "${project}", readonly = true, required = true)
    protected MavenProject project;

    @Parameter(name = "documentationSourcePath")
    private String documentationSourcePath;

    @Parameter(name = "outputPath")
    private String outputPath;

    @Parameter(name = "title")
    private String title;

    private void recurseFiles(File basePath, File f, ClassLoader targetClassLoader) throws MojoExecutionException {
        getLog().debug("file : " + f.getAbsolutePath());
        if (f.isFile()) {
            if (f.getAbsolutePath().endsWith(".class")) {
                parseClassFile(basePath, f, targetClassLoader);
            }
        } else {
            for (File subf : f.listFiles()) {
                recurseFiles(basePath, subf, targetClassLoader);
            }
        }
    }

    public void parseTargetClasses(ClassLoader targetClassLoader, List<String> classPathElements) throws MojoExecutionException {
        getLog().info("parse classes " + classPathElements);
        for (String classPathElement : classPathElements) {
            recurseFiles(new File(classPathElement), new File(classPathElement), targetClassLoader);
        }
    }

    private final List<RestControllerInstance> restControllers = new ArrayList<>();

    private SpringRestDocEndpointBlacklist blacklist = null;

    private void loadBlacklist() throws MojoExecutionException {
        File sourceFile = new File(this.documentationSourcePath + File.separatorChar + "springrestdoc-blacklist.xml");
        if (sourceFile.exists()) {
            try (FileReader fin = new FileReader(sourceFile)) {
                XStream xstream = new XStream();
                xstream.processAnnotations(SpringRestDocEndpointBlacklist.class);
                blacklist = (SpringRestDocEndpointBlacklist) xstream.fromXML(fin);
                getLog().info("endpoint blacklist " + blacklist);
            } catch (IOException ex) {
                throw new MojoExecutionException("failed to read blacklist file", ex);
            }
        }
    }

    @Override
    public void execute() throws MojoExecutionException, MojoFailureException {
        getLog().info("Spring REST doc parsing project " + project.getName() + " artifact:" + project.getArtifact());
        establishOutputPath();
        loadBlacklist();
        String jarsFolderPath = project.getBasedir().getAbsolutePath() + File.separatorChar + "target" + File.separatorChar + project.getArtifact().getArtifactId() + "-" + project.getArtifact().getVersion();
        jarsFolderPath += (File.separatorChar + "WEB-INF" + File.separatorChar + "lib");
        File jarsFolder = new File(jarsFolderPath);
        getLog().info("dependencies folder:" + jarsFolder);
        try {
            // add the jars to the URLClassLoader
            List<URL> jarPaths = new ArrayList<>();
            if (jarsFolder.isDirectory()) {
                for (File f : jarsFolder.listFiles()) {
                    if (f.getAbsolutePath().toUpperCase().endsWith(".JAR")) {
                        jarPaths.add(f.toURI().toURL());
                    }
                }
            }
            List rtElements = project.getDependencies();
            getLog().info("rte:" + rtElements + " size=" + rtElements.size());

            getLog().info(project.getFile().getAbsolutePath());
            List<String> classPathElements = project.getCompileClasspathElements();
            URL[] classPathURLList = new URL[classPathElements.size() + jarPaths.size()];
            int c = 0;
            for (String element : classPathElements) {
                URL classPathURL = new File(element).toURI().toURL();
                //getLog().info("target class path:" + classPathURL.toString());
                classPathURLList[c++] = classPathURL;
            }
            for (URL jarURL : jarPaths) {
                classPathURLList[c++] = jarURL;
            }
            URLClassLoader targetClassLoader = new URLClassLoader(classPathURLList);
            parseTargetClasses(targetClassLoader, classPathElements);
            getLog().info("resolved REST controllers:" + restControllers);
            writeTemplateOutput();
        } catch (DependencyResolutionRequiredException | MalformedURLException ex) {
            throw new MojoExecutionException("template generation failure", ex);
        }
    }

    private void parseClassFile(File basePath, File f, ClassLoader targetClassLoader) throws MojoExecutionException {
        try {
            // strip off basePath
            String prefix = basePath.getAbsolutePath();
            String fileName = f.getAbsolutePath();
            String className = fileName.substring(prefix.length() + 1);
            className = className.replace("/", ".");
            className = className.substring(0, className.length() - 6);
            Class klass = targetClassLoader.loadClass(className);
            parseClass(klass);
        } catch (ClassNotFoundException ex) {
            throw new MojoExecutionException("failed to parse class", ex);
        }
    }

    private void parseClass(Class klass) throws MojoExecutionException {
        getLog().debug(" == process CLASS :" + klass.getName());
        Annotation[] annotations = klass.getAnnotations();
        boolean isRestController = false;
        for (Annotation annotation : annotations) {
            if (annotation.annotationType().getName().equals(org.springframework.web.bind.annotation.RestController.class.getName())) {
                isRestController = true;
            }
        }
        if (!isRestController) {
            return;
        }
        RestControllerInstance rci = null;
        for (Annotation annotation : annotations) {
            if (annotation.annotationType().getName().equals(RequestMapping.class.getName())) {
                try {
                    Class mapping = annotation.annotationType();
                    Method methodValues = mapping.getMethod("value");
                    if (methodValues != null) {
                        String[] restControllerAddresses = (String[]) methodValues.invoke(annotation);
                        rci = new RestControllerInstance(klass, restControllerAddresses);
                        // make sure it's not on the blacklist
                        if (blacklist != null) {
                            for (String address : rci.getAddresses()) {
                                if (blacklist.getEndpoints().contains(address)) {
                                    getLog().info("omitting blacklisted endpoint " + address);
                                    return;
                                }
                            }
                        }
                        restControllers.add(rci);
                        getLog().info(" --> created new REST controller instance " + rci);
                    }
                } catch (NoSuchMethodException | SecurityException | IllegalAccessException | IllegalArgumentException | InvocationTargetException ex) {
                    throw new MojoExecutionException("failed to parse class", ex);
                }
            }
        }
        if (rci == null) {
            return;
        }
        for (Method method : klass.getMethods()) {
            for (Annotation annotation : method.getAnnotations()) {
                getLog().info("method annotation " + annotation + " on method " + method);
                if (annotation.annotationType().getName().equals(org.springframework.web.bind.annotation.RequestMapping.class.getName())) {
                    // found a rest method
                    parseMethod(method, annotation, rci);
                }
            }
        }
    }

    private void establishOutputPath() throws MojoFailureException {
        getLog().info("output path = " + this.outputPath);
        File f = new File(this.outputPath);
        if (!f.exists()) {
            f.mkdir();
        }
        if (!f.exists()) {
            throw new MojoFailureException("failed to create target path:" + f.getAbsolutePath());
        }
    }

    private void writeTemplateOutput() throws MojoExecutionException {
        XStream xstream = new XStream();
        xstream.processAnnotations(SpringRestDocIndex.class);
        xstream.processAnnotations(DocumentationPage.class);
        SpringRestDocIndex docIndex = getDocumentIndex();
        File out = new File(documentationSourcePath + File.separatorChar + "springrestdoc.xml");
        try (FileWriter fout = new FileWriter(out)) {
            xstream.toXML(docIndex, fout);
        } catch (IOException ex) {
            throw new MojoExecutionException("failed to write new template", ex);
        }
    }

    private SpringRestDocIndex getDocumentIndex() throws MojoExecutionException {
        SpringRestDocIndex docIndex = new SpringRestDocIndex();
        if (title == null) {
            docIndex.setProjectTitle("Spring REST Documentation");
        } else {
            docIndex.setProjectTitle(title);
        }
        DocumentationPage indexPage = generateHomePage();
        docIndex.setIndexPage(indexPage);
        return docIndex;
    }

    private DocumentationPage generateHomePage() throws MojoExecutionException {
        DocumentationPage indexPage = new DocumentationPage();
        indexPage.setPageTitle(this.title);
        indexPage.setPath("Home.md");
        indexPage.setInline(Boolean.TRUE);
        // generate Home.md
        File homePageFile = new File(this.documentationSourcePath + File.separatorChar + "Home.md");
        try (PrintWriter out = new PrintWriter(new FileOutputStream(homePageFile))) {
            out.println("# " + this.title);
            out.println("{{preamble}}");
            for (RestControllerInstance rci : this.restControllers) {
                // link: [test](urlpagename)
                String name = rci.getControllerClass().getSimpleName();
                out.printf("[%s](%s)\n\n", rci.getAddressesText(), name);
            }
            out.println("{{postfix}}");
        } catch (IOException ex) {
            throw new MojoExecutionException("filed to write output file", ex);
        }
        // generate controller pages        
        for (RestControllerInstance rci : this.restControllers) {
            // link: [test](urlpagename)
            generateControllerPage(rci, indexPage);
        }
        return indexPage;
    }

    private void parseMethod(Method method, Annotation annotation, RestControllerInstance rci) throws MojoExecutionException {
        try {
            String address = null;
            String restMethod = null;
            Class mapping = annotation.annotationType();
            Method methodValues = mapping.getMethod("method");
            Object[] methodValuesResult = (Object[]) methodValues.invoke(annotation);
            if (methodValuesResult != null) {
                getLog().info("length = " + Integer.toString(methodValuesResult.length));
                if (methodValuesResult.length != 0) {
                    restMethod = methodValuesResult[0].toString();
                }
            }
            Method methodAddress = mapping.getMethod("value");
            if (methodAddress != null) {
                String[] methodAddressesResult = (String[]) methodAddress.invoke(annotation);
                if (methodAddressesResult != null) {
                    if (methodAddressesResult.length != 0) {
                        address = methodAddressesResult[0];
                    }
                }
            }
            if ((address != null) && (restMethod != null)) {
                RestMethodInstance rmi = new RestMethodInstance(restMethod, address, method);
                rci.addMethod(rmi);
                getLog().info("added new REST method " + rmi);
            }
        } catch (NoSuchMethodException | SecurityException | IllegalAccessException | IllegalArgumentException | InvocationTargetException ex) {
            throw new MojoExecutionException("failed to parse method", ex);
        }
    }

    private void generateControllerPage(RestControllerInstance rci, DocumentationPage indexPage) throws MojoExecutionException {
        String name = rci.getControllerClass().getSimpleName();
        //String fileName = name.replace('.', '_');
        File pageFile = new File(this.documentationSourcePath + File.separatorChar + name + ".md");
        // table of methods:
//            | Tables        | Are           | Cool  |
//            | ------------- |:-------------:| -----:|
        DocumentationPage page = new DocumentationPage();
        page.setPageTitle(rci.getAddressesText());
        page.setPath(name + ".md");
        page.setInline(Boolean.FALSE);
        try (PrintWriter out = new PrintWriter(new FileOutputStream(pageFile))) {
            out.println("# " + rci.getAddressesText());
            out.println("{{preamble}}");
            out.println("| Method | URL | Description |");
            out.println("| -------| -------- | -------- |");
            // write methods.
            for (RestMethodInstance rmi : rci.getMethods()) {
                String methodFilename = rci.getControllerClass().getSimpleName() + "_" + rmi.getMethod().getName();
                String methodPageLink = String.format("[%s](%s)", rmi.getMethod().getName(), methodFilename);
                out.printf("| %s | %s | %s |\n", rmi.getHttpMethod(), rmi.getAddress(), methodPageLink);
                generateMethodPage(page, rci, rmi);
                // for each method generate a method page.
            }
            out.println("{{postfix}}");
        } catch (IOException ex) {
            throw new MojoExecutionException("failed to write output file", ex);
        }
        indexPage.getPages().add(page);
    }

    private String getSimpleTypeName(String typeName) {
        if (typeName.isEmpty()) {
            return typeName;
        } else {
            String[] x;
            if (typeName.contains("<")) {
                x = typeName.split("\\<");
                if (x.length != 0) {
                    typeName = x[0];
                }
            }
            x = typeName.split("\\.");
            if (x.length == 0) {
                return "";
            } else {
                return x[x.length - 1];
            }
        }
    }

    private void generateMethodPage(DocumentationPage page, RestControllerInstance rci, RestMethodInstance rmi) throws MojoExecutionException {
        String methodFilename = rci.getControllerClass().getSimpleName() + "_" + rmi.getMethod().getName();
        File methodFile = new File(this.documentationSourcePath + File.separatorChar + methodFilename + ".md");
        try (PrintWriter out = new PrintWriter(new FileOutputStream(methodFile))) {
            out.println("# " + rmi.getMethod().getName());
            out.println("{{preamble}}\n");
            // URL
            out.println("### URL");
            out.printf("%s\n", rmi.getAddress());
            int c = 0;
            for (java.lang.reflect.Parameter param : rmi.getMethod().getParameters()) {
                for (Annotation pa : param.getAnnotations()) {
                    if (pa.annotationType().getName().equals(org.springframework.web.bind.annotation.PathVariable.class.getName())) {
                        getLog().info(" PathVariable method annotation " + param.getName() + " : " + pa.toString() + " : " + param.getType().getName());
                        c++;
                    } else if (pa.annotationType().getName().equals(org.springframework.web.bind.annotation.RequestBody.class.getName())) {
                        getLog().info(" RequestBody method annotation " + param.getName() + " : " + pa.toString() + " : " + param.getType().getName());
                        c++;
                    } else if (pa.annotationType().getName().equals(org.springframework.web.bind.annotation.RequestHeader.class.getName())) {
                        getLog().info(" RequestHeader method annotation " + param.getName() + " : " + pa.toString() + " : " + param.getType().getName());
                        c++;
                    }
                }
            }
            if (c != 0) {
                out.println("### Parameters");
                out.println("| Parameter | Type | Description |");
                out.println("| -------| -------- | -------- |");
                for (java.lang.reflect.Parameter param : rmi.getMethod().getParameters()) {
                    for (Annotation pa : param.getAnnotations()) {
                        if (pa.annotationType().getName().equals(org.springframework.web.bind.annotation.PathVariable.class.getName())) {
                            getLog().info(" PathVariable method annotation " + param.getName() + " : " + pa.toString() + " : " + param.getType().getName());
                            try {
                                Method methodValue = pa.annotationType().getMethod("value");
                                String paramName = (String) methodValue.invoke(pa);
                                String paramDescFilename = "param" + Integer.toString(c++);
                                out.printf("| %-24s | %-32s | {{%s}} |\n", paramName, param.getType().getSimpleName(), paramDescFilename);
                            } catch (NoSuchMethodException | SecurityException | IllegalAccessException | IllegalArgumentException | InvocationTargetException ex) {
                                throw new MojoExecutionException("failed to get method parameter name", ex);
                            }
                        } else if (pa.annotationType().getName().equals(org.springframework.web.bind.annotation.RequestBody.class.getName())) {
                            getLog().info(" RequestBody method annotation " + param.getName() + " : " + pa.toString() + " : " + param.getType().getName());
                            String paramDescFilename = "param" + Integer.toString(c++);
                            out.printf("| JSON object | %-32s | {{%s}} |\n", param.getType().getSimpleName(), paramDescFilename);
                        } else if (pa.annotationType().getName().equals(org.springframework.web.bind.annotation.RequestHeader.class.getName())) {
                            getLog().info(" RequestHeader method annotation " + param.getName() + " : " + pa.toString() + " : " + param.getType().getName());
                            String paramDescFilename = "header" + Integer.toString(c++);
                            try {
                                Method methodValue = pa.annotationType().getMethod("value");
                                String paramName = (String) methodValue.invoke(pa);
                                out.printf("| HEADER %-24s | %-32s | {{%s}} |\n", paramName, param.getType().getSimpleName(), paramDescFilename);
                            } catch (NoSuchMethodException | SecurityException | IllegalAccessException | IllegalArgumentException | InvocationTargetException ex) {
                                throw new MojoExecutionException("failed to get method parameter name", ex);
                            }
                        }
                    }
                }
            }
            // return type
            String listTypeText = null;
            Type returnType = rmi.getMethod().getGenericReturnType();
            if (returnType != null) {
                out.println("### Returns");
                if (returnType instanceof ParameterizedType) {
                    ParameterizedType pReturnType = (ParameterizedType) returnType;
                    listTypeText = "";
                    int ac = 0;
                    for (Type typeArg : pReturnType.getActualTypeArguments()) {
                        if (ac != 0) {
                            listTypeText += ',';
                        } else {
                            listTypeText += getSimpleTypeName(typeArg.getTypeName());
                        }
                        ac++;
                    }
                }
                if (listTypeText != null) {
                    out.printf("%s of %s\n", getSimpleTypeName(returnType.getTypeName()), listTypeText);
                } else {
                    out.printf("%s\n", getSimpleTypeName(returnType.getTypeName()));
                }
            }
            out.println("{{postfix}}");
        } catch (IOException ex) {
            throw new MojoExecutionException("failed to write output file", ex);
        }
        DocumentationPage pageNode = new DocumentationPage();
        pageNode.setPageTitle(rci.getAddressesText());
        pageNode.setPath(methodFilename + ".md");
        pageNode.setInline(Boolean.FALSE);
        page.getPages().add(pageNode);
    }

}
