/**
 * MIT License
 * 
 * Copyright (c) 2018 b+m Informatik AG
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package io.tapirtest.execution.gui.application.listener;

import com.google.common.base.Objects;
import de.bmiag.tapir.execution.executor.AbstractExecutionListener;
import de.bmiag.tapir.execution.model.ExecutionPlan;
import de.bmiag.tapir.execution.model.Identifiable;
import de.bmiag.tapir.execution.model.TestClass;
import de.bmiag.tapir.execution.model.TestStep;
import de.bmiag.tapir.execution.model.TestSuite;
import io.tapirtest.execution.gui.application.components.AbstractCheckBoxTreeItem;
import io.tapirtest.execution.gui.application.data.ExecutionStatus;
import javafx.application.Platform;
import javafx.beans.property.ObjectProperty;
import javafx.collections.ObservableList;
import javafx.scene.control.TreeItem;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.springframework.stereotype.Component;

/**
 * This is the listener responsible for updating the view model during the execution of the tapir test cases.
 * 
 * @author Nils Christian Ehmke
 * 
 * @since 1.0.0
 */
@Component
@SuppressWarnings("all")
public class GUIExecutionListener extends AbstractExecutionListener {
  private final static Logger logger = LogManager.getLogger(GUIExecutionListener.class);
  
  private AbstractCheckBoxTreeItem<ExecutionPlan> executionPlanRoot;
  
  private ObjectProperty<Object> refreshTableObservable;
  
  @Override
  public void classSucceeded(final TestClass testClass) {
    this.setStatus(testClass, ExecutionStatus.SUCCEEDED);
  }
  
  @Override
  public void classFailed(final TestClass testClass, final Throwable throwable) {
    this.setStatus(testClass, ExecutionStatus.FAILED);
  }
  
  @Override
  public void classSkipped(final TestClass testClass) {
    this.setStatus(testClass, ExecutionStatus.SKIPPED);
  }
  
  @Override
  public void executionFailed(final ExecutionPlan executionPlan, final Throwable throwable) {
    this.setStatus(executionPlan, ExecutionStatus.FAILED);
  }
  
  @Override
  public void executionSucceeded(final ExecutionPlan executionPlan) {
    this.setStatus(executionPlan, ExecutionStatus.SUCCEEDED);
  }
  
  @Override
  public void stepFailed(final TestStep testStep, final Throwable throwable) {
    this.setStatus(testStep, ExecutionStatus.FAILED);
  }
  
  @Override
  public void stepSkipped(final TestStep testStep) {
    this.setStatus(testStep, ExecutionStatus.SKIPPED);
  }
  
  @Override
  public void stepSucceeded(final TestStep testStep) {
    this.setStatus(testStep, ExecutionStatus.SUCCEEDED);
  }
  
  @Override
  public void suiteFailed(final TestSuite testSuite, final Throwable throwable) {
    this.setStatus(testSuite, ExecutionStatus.FAILED);
  }
  
  @Override
  public void suiteSkipped(final TestSuite testSuite) {
    this.setStatus(testSuite, ExecutionStatus.SKIPPED);
  }
  
  @Override
  public void suiteSucceeded(final TestSuite testSuite) {
    this.setStatus(testSuite, ExecutionStatus.SUCCEEDED);
  }
  
  private void setStatus(final Identifiable testItem, final ExecutionStatus executionStatus) {
    if ((this.executionPlanRoot == null)) {
      return;
    }
    final TreeItem<Identifiable> item = this.search(this.executionPlanRoot, testItem);
    if ((item instanceof AbstractCheckBoxTreeItem<?>)) {
      final Runnable _function = () -> {
        ((AbstractCheckBoxTreeItem<?>)item).setExecutionStatus(executionStatus);
        Object _object = new Object();
        this.refreshTableObservable.setValue(_object);
      };
      Platform.runLater(_function);
    } else {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Test item with ");
      Integer _id = testItem.getId();
      _builder.append(_id);
      _builder.append(" could not be found in the execution plan");
      GUIExecutionListener.logger.warn(_builder);
    }
  }
  
  private TreeItem<Identifiable> search(final TreeItem<Identifiable> treeItem, final Identifiable testItem) {
    Identifiable _value = treeItem.getValue();
    boolean _equals = Objects.equal(_value, testItem);
    if (_equals) {
      return treeItem;
    } else {
      ObservableList<TreeItem<Identifiable>> _children = treeItem.getChildren();
      for (final TreeItem<Identifiable> child : _children) {
        {
          final TreeItem<Identifiable> result = this.search(child, testItem);
          if ((result != null)) {
            return result;
          }
        }
      }
    }
    return null;
  }
  
  public AbstractCheckBoxTreeItem<ExecutionPlan> setExecutionPlanRoot(final AbstractCheckBoxTreeItem<ExecutionPlan> executionPlanRoot) {
    return this.executionPlanRoot = executionPlanRoot;
  }
  
  public ObjectProperty<Object> setRefreshTableObservable(final ObjectProperty<Object> refreshTableObservable) {
    return this.refreshTableObservable = refreshTableObservable;
  }
}
