/*
 * Decompiled with CFR 0.152.
 */
package io.scalecube.metrics.loki;

import io.scalecube.metrics.Delay;
import io.scalecube.metrics.MetricNames;
import io.scalecube.metrics.loki.LokiPublisher;
import io.scalecube.metrics.loki.WriteRequest;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.time.Duration;
import java.time.Instant;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import org.agrona.CloseHelper;
import org.agrona.concurrent.Agent;
import org.agrona.concurrent.AgentInvoker;
import org.agrona.concurrent.AgentTerminationException;
import org.agrona.concurrent.EpochClock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JvmSafepointExporter
implements Agent {
    private static final Logger LOGGER = LoggerFactory.getLogger(JvmSafepointExporter.class);
    private static final Duration READ_INTERVAL = Duration.ofSeconds(1L);
    private static final int DEFAULT_CHUNK_SIZE = 65536;
    private static final DateTimeFormatter GC_LOG_TIMESTAMP_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss.SSSZ");
    private static final Pattern SAFEPOINT_PATTERN = Pattern.compile("\\[(?<timestamp>[^]]+)] Safepoint \"(?<reason>[^\"]+)\", Time since last: (?<sinceLast>\\d+) ns, Reaching safepoint: (?<reaching>\\d+) ns, Cleanup: (?<cleanup>\\d+) ns, At safepoint: (?<at>\\d+) ns, Total: (?<total>\\d+) ns");
    private final File gcLogDir;
    private final Map<String, String> labels;
    private final LokiPublisher.WriteProxy writeProxy;
    private final AgentInvoker publisherInvoker;
    private final Delay retryInterval;
    private final Delay readInterval;
    private FileChannel fileChannel;
    private final ByteBuffer chunkBuffer = ByteBuffer.allocate(65536);
    private final StringBuilder lineBuffer = new StringBuilder();
    private State state = State.CLOSED;

    public JvmSafepointExporter(File gcLogDir, Map<String, String> labels, LokiPublisher.WriteProxy writeProxy, AgentInvoker publisherInvoker, EpochClock epochClock, Duration retryInterval) {
        this.gcLogDir = gcLogDir;
        this.labels = labels;
        this.writeProxy = writeProxy;
        this.publisherInvoker = publisherInvoker;
        this.retryInterval = new Delay(epochClock, retryInterval.toMillis());
        this.readInterval = new Delay(epochClock, READ_INTERVAL.toMillis());
    }

    public String roleName() {
        return "JvmSafepointExporter";
    }

    public void onStart() {
        if (this.state != State.CLOSED) {
            throw new AgentTerminationException("Illegal state: " + String.valueOf((Object)this.state));
        }
        this.state(State.INIT);
    }

    public int doWork() throws Exception {
        try {
            if (this.publisherInvoker != null) {
                this.publisherInvoker.invoke();
            }
            return switch (this.state) {
                case State.INIT -> this.init();
                case State.RUNNING -> this.running();
                case State.CLEANUP -> this.cleanup();
                default -> throw new AgentTerminationException("Unknown state: " + String.valueOf((Object)this.state));
            };
        }
        catch (AgentTerminationException e) {
            throw e;
        }
        catch (Exception e) {
            this.state(State.CLEANUP);
            throw e;
        }
    }

    private int init() throws IOException {
        if (this.retryInterval.isNotOverdue()) {
            return 0;
        }
        Path filePath = JvmSafepointExporter.findLatestGcLog(this.gcLogDir.toPath());
        if (!Files.exists(filePath, new LinkOption[0]) || Files.isDirectory(filePath, new LinkOption[0])) {
            throw new IllegalArgumentException("Wrong file: " + String.valueOf(filePath));
        }
        this.fileChannel = FileChannel.open(filePath, new OpenOption[0]);
        this.state(State.RUNNING);
        return 1;
    }

    private static Path findLatestGcLog(Path dir) throws IOException {
        try (Stream<Path> files = Files.list(dir);){
            Path path = files.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).filter(p -> p.getFileName().toString().contains("gc.log")).max(Comparator.comparingLong(p -> p.toFile().lastModified())).orElseThrow(() -> new FileNotFoundException("No matching gc.log files found in " + String.valueOf(dir)));
            return path;
        }
    }

    private int running() throws IOException {
        int lineEnd;
        if (this.readInterval.isOverdue()) {
            int read = this.fileChannel.read(this.chunkBuffer.clear());
            if (read > 0) {
                byte[] bytes = new byte[this.chunkBuffer.flip().remaining()];
                this.chunkBuffer.get(bytes);
                this.lineBuffer.append(new String(bytes, StandardCharsets.UTF_8));
            } else {
                this.readInterval.delay();
            }
        }
        int workCount = 0;
        ArrayList<SafepointEvent> events = new ArrayList<SafepointEvent>();
        while ((lineEnd = this.lineBuffer.indexOf("\n")) >= 0) {
            String line = this.lineBuffer.substring(0, lineEnd).trim();
            this.lineBuffer.delete(0, lineEnd + 1);
            SafepointEvent event = JvmSafepointExporter.processLine(line);
            if (event == null) continue;
            ++workCount;
            events.add(event);
        }
        if (!events.isEmpty()) {
            this.writeProxy.push(this.toWriteRequest(events));
        }
        return workCount;
    }

    private static SafepointEvent processLine(String line) {
        Matcher matcher = SAFEPOINT_PATTERN.matcher(line);
        if (!matcher.find()) {
            return null;
        }
        return new SafepointEvent(ZonedDateTime.parse(matcher.group("timestamp"), GC_LOG_TIMESTAMP_FORMATTER).toInstant(), matcher.group("reason"), Long.parseLong(matcher.group("sinceLast")), Long.parseLong(matcher.group("reaching")), Long.parseLong(matcher.group("cleanup")), Long.parseLong(matcher.group("at")), Long.parseLong(matcher.group("total")));
    }

    private WriteRequest toWriteRequest(List<SafepointEvent> events) {
        Map<String, String> streamLabels = JvmSafepointExporter.streamLabels(this.labels);
        streamLabels.put("metric_name", "jvm_safepoint");
        List<String[]> values = events.stream().map(JvmSafepointExporter::toLogEntry).toList();
        return new WriteRequest(List.of(new WriteRequest.Stream(streamLabels, values)));
    }

    private static Map<String, String> streamLabels(Map<String, String> map) {
        HashMap<String, String> labels = new HashMap<String, String>();
        if (map != null) {
            map.forEach((key, value) -> labels.put(MetricNames.sanitizeName((String)key), (String)value));
        }
        return labels;
    }

    private static String[] toLogEntry(SafepointEvent event) {
        Instant ts = event.timestamp();
        String timestamp = String.valueOf(TimeUnit.SECONDS.toNanos(ts.getEpochSecond()) + (long)ts.getNano());
        String logLine = String.format("reason=\"%s\" sinceLast=%.3f reaching=%.3f cleanup=%.3f at=%.3f total=%.3f", event.reason(), JvmSafepointExporter.toMicros(event.sinceLastNs()), JvmSafepointExporter.toMicros(event.reachingNs()), JvmSafepointExporter.toMicros(event.cleanupNs()), JvmSafepointExporter.toMicros(event.atSafepointNs()), JvmSafepointExporter.toMicros(event.totalNs()));
        return new String[]{timestamp, logLine};
    }

    private static double toMicros(long nanos) {
        return (double)nanos / (double)TimeUnit.MICROSECONDS.toNanos(1L);
    }

    private int cleanup() {
        CloseHelper.quietClose((AutoCloseable)this.fileChannel);
        this.lineBuffer.setLength(0);
        State previous = this.state;
        if (previous != State.CLOSED) {
            this.retryInterval.delay();
            this.state(State.INIT);
        }
        return 1;
    }

    public void onClose() {
        this.state(State.CLOSED);
        this.cleanup();
    }

    private void state(State state) {
        LOGGER.debug("[{}][state] {}->{}", new Object[]{this.roleName(), this.state, state});
        this.state = state;
    }

    public static enum State {
        INIT,
        RUNNING,
        CLEANUP,
        CLOSED;

    }

    record SafepointEvent(Instant timestamp, String reason, long sinceLastNs, long reachingNs, long cleanupNs, long atSafepointNs, long totalNs) {
    }
}

