/*
 * Decompiled with CFR 0.152.
 */
package io.opentelemetry.sdk.logging.export;

import io.opentelemetry.api.metrics.BoundLongCounter;
import io.opentelemetry.api.metrics.GlobalMetricsProvider;
import io.opentelemetry.api.metrics.LongCounter;
import io.opentelemetry.api.metrics.Meter;
import io.opentelemetry.api.metrics.common.Labels;
import io.opentelemetry.sdk.common.CompletableResultCode;
import io.opentelemetry.sdk.internal.DaemonThreadFactory;
import io.opentelemetry.sdk.logging.LogProcessor;
import io.opentelemetry.sdk.logging.data.LogRecord;
import io.opentelemetry.sdk.logging.export.BatchLogProcessorBuilder;
import io.opentelemetry.sdk.logging.export.LogExporter;
import java.util.ArrayList;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;

public final class BatchLogProcessor
implements LogProcessor {
    private static final String WORKER_THREAD_NAME = BatchLogProcessor.class.getSimpleName() + "_WorkerThread";
    private final Worker worker;
    private final Thread workerThread;

    BatchLogProcessor(int maxQueueSize, long scheduleDelayMillis, int maxExportBatchSize, long exporterTimeoutMillis, LogExporter logExporter) {
        this.worker = new Worker(logExporter, scheduleDelayMillis, maxExportBatchSize, exporterTimeoutMillis, new ArrayBlockingQueue(maxQueueSize));
        this.workerThread = new DaemonThreadFactory(WORKER_THREAD_NAME).newThread((Runnable)this.worker);
        this.workerThread.start();
    }

    public static BatchLogProcessorBuilder builder(LogExporter logExporter) {
        return new BatchLogProcessorBuilder(logExporter);
    }

    @Override
    public void addLogRecord(LogRecord record) {
        this.worker.addLogRecord(record);
    }

    @Override
    public CompletableResultCode shutdown() {
        this.workerThread.interrupt();
        return this.worker.shutdown();
    }

    @Override
    public CompletableResultCode forceFlush() {
        return this.worker.forceFlush();
    }

    private static class Worker
    implements Runnable {
        private static final BoundLongCounter exporterFailureCounter;
        private static final BoundLongCounter queueFullRecordCounter;
        private static final BoundLongCounter successCounter;
        private final long scheduleDelayNanos;
        private final int maxExportBatchSize;
        private final LogExporter logExporter;
        private final long exporterTimeoutMillis;
        private final ArrayList<LogRecord> batch;
        private final BlockingQueue<LogRecord> queue;
        private final AtomicReference<CompletableResultCode> flushRequested = new AtomicReference();
        private volatile boolean continueWork = true;
        private long nextExportTime;

        private Worker(LogExporter logExporter, long scheduleDelayMillis, int maxExportBatchSize, long exporterTimeoutMillis, BlockingQueue<LogRecord> queue) {
            this.logExporter = logExporter;
            this.maxExportBatchSize = maxExportBatchSize;
            this.exporterTimeoutMillis = exporterTimeoutMillis;
            this.scheduleDelayNanos = TimeUnit.MILLISECONDS.toNanos(scheduleDelayMillis);
            this.queue = queue;
            this.batch = new ArrayList(this.maxExportBatchSize);
        }

        @Override
        public void run() {
            this.updateNextExportTime();
            while (this.continueWork) {
                if (this.flushRequested.get() != null) {
                    this.flush();
                }
                try {
                    LogRecord lastElement = this.queue.poll(100L, TimeUnit.MILLISECONDS);
                    if (lastElement != null) {
                        this.batch.add(lastElement);
                    }
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    return;
                }
                if (this.batch.size() < this.maxExportBatchSize && System.nanoTime() < this.nextExportTime) continue;
                this.exportCurrentBatch();
                this.updateNextExportTime();
            }
        }

        private void flush() {
            for (int recordsToFlush = this.queue.size(); recordsToFlush > 0; --recordsToFlush) {
                LogRecord record = (LogRecord)this.queue.poll();
                assert (record != null);
                this.batch.add(record);
                if (this.batch.size() < this.maxExportBatchSize) continue;
                this.exportCurrentBatch();
            }
            this.exportCurrentBatch();
            CompletableResultCode result = this.flushRequested.get();
            assert (result != null);
            this.flushRequested.set(null);
        }

        private void updateNextExportTime() {
            this.nextExportTime = System.nanoTime() + this.scheduleDelayNanos;
        }

        private void exportCurrentBatch() {
            if (this.batch.isEmpty()) {
                return;
            }
            try {
                CompletableResultCode result = this.logExporter.export(this.batch);
                result.join(this.exporterTimeoutMillis, TimeUnit.MILLISECONDS);
                if (result.isSuccess()) {
                    successCounter.add((long)this.batch.size());
                } else {
                    exporterFailureCounter.add(1L);
                }
            }
            catch (RuntimeException t) {
                exporterFailureCounter.add((long)this.batch.size());
            }
            finally {
                this.batch.clear();
            }
        }

        private CompletableResultCode shutdown() {
            final CompletableResultCode result = new CompletableResultCode();
            final CompletableResultCode flushResult = this.forceFlush();
            flushResult.whenComplete(new Runnable(){

                @Override
                public void run() {
                    continueWork = false;
                    final CompletableResultCode shutdownResult = logExporter.shutdown();
                    shutdownResult.whenComplete(new Runnable(){

                        @Override
                        public void run() {
                            if (flushResult.isSuccess() && shutdownResult.isSuccess()) {
                                result.succeed();
                            } else {
                                result.fail();
                            }
                        }
                    });
                }
            });
            return result;
        }

        private CompletableResultCode forceFlush() {
            CompletableResultCode flushResult = new CompletableResultCode();
            this.flushRequested.compareAndSet(null, flushResult);
            return this.flushRequested.get();
        }

        public void addLogRecord(LogRecord record) {
            if (!this.queue.offer(record)) {
                queueFullRecordCounter.add(1L);
            }
        }

        static {
            Meter meter = GlobalMetricsProvider.getMeter((String)"io.opentelemetry.sdk.logging");
            LongCounter logRecordsProcessed = meter.longCounterBuilder("logRecordsProcessed").setUnit("1").setDescription("Number of records processed").build();
            successCounter = logRecordsProcessed.bind(Labels.of((String)"result", (String)"success"));
            exporterFailureCounter = logRecordsProcessed.bind(Labels.of((String)"result", (String)"dropped record", (String)"cause", (String)"exporter failure"));
            queueFullRecordCounter = logRecordsProcessed.bind(Labels.of((String)"result", (String)"dropped record", (String)"cause", (String)"queue full"));
        }
    }
}

