package io.shiftleft.semanticcpg.language

import io.shiftleft.codepropertygraph.generated.Cpg
import io.shiftleft.codepropertygraph.generated.nodes.*
import io.shiftleft.codepropertygraph.generated.{NodeTypes, Properties}
import io.shiftleft.codepropertygraph.generated.help.{Doc, TraversalSource}
import io.shiftleft.semanticcpg.language.*

/** Starting point for a new traversal, e.g.
  *   - `cpg.method`, `cpg.call` etc. - these are generated by the flatgraph codegenerator and automatically inherited
  *   - `cpg.method.name`
  */
@TraversalSource
class NodeTypeStarters(cpg: Cpg) {

  /** Traverse to all arguments passed to methods */
  @Doc(info = "All arguments (actual parameters)")
  def argument: Iterator[Expression] =
    cpg.call.argument

  /** Shorthand for `cpg.argument.code(code)` */
  def argument(code: String): Iterator[Expression] =
    cpg.argument.code(code)

  @Doc(info = "All breaks (`ControlStructure` nodes)")
  def break: Iterator[ControlStructure] =
    cpg.controlStructure.isBreak

  @Doc(info = "All continues (`ControlStructure` nodes)")
  def continue: Iterator[ControlStructure] =
    cpg.controlStructure.isContinue

  @Doc(info = "All do blocks (`ControlStructure` nodes)")
  def doBlock: Iterator[ControlStructure] =
    cpg.controlStructure.isDo

  @Doc(info = "All else blocks (`ControlStructure` nodes)")
  def elseBlock: Iterator[ControlStructure] =
    cpg.controlStructure.isElse

  @Doc(info = "All throws (`ControlStructure` nodes)")
  def throws: Iterator[ControlStructure] =
    cpg.controlStructure.isThrow

  @Doc(info = "All for blocks (`ControlStructure` nodes)")
  def forBlock: Iterator[ControlStructure] =
    cpg.controlStructure.isFor

  @Doc(info = "All gotos (`ControlStructure` nodes)")
  def goto: Iterator[ControlStructure] =
    cpg.controlStructure.isGoto

  @Doc(info = "All if blocks (`ControlStructure` nodes)")
  def ifBlock: Iterator[ControlStructure] =
    cpg.controlStructure.isIf

  /** Shorthand for `cpg.methodRef.where(_.referencedMethod.name(name))`
    *
    * Note re API design: this step was supposed to be called `methodRef(name: String)`, but due to limitations in
    * Scala's implicit resolution (and the setup of our implicit steps) we have to disambiguate it from `.methodRef` by
    * name.
    *
    * More precisely: Scala's implicit resolution reports 'ambiguous implicits' if two methods with the same name but
    * different parameters are defined in two different (implicitly reachable) classes. The `.methodRef` step is defined
    * in `generated.CpgNodeStarter`. This step (filter by name) doesn't get generated by the codegen because it's more
    * complex than the other 'filter by primary key' starter steps.
    */
  def methodRefWithName(name: String): Iterator[MethodRef] =
    cpg.methodRef.where(_.referencedMethod.name(name))

  /** Traverse to all input parameters */
  @Doc(info = "All parameters")
  def parameter: Iterator[MethodParameterIn] =
    cpg.methodParameterIn

  /** Shorthand for `cpg.parameter.name(name)` */
  def parameter(name: String): Iterator[MethodParameterIn] =
    parameter.name(name)

  @Doc(info = "All switch blocks (`ControlStructure` nodes)")
  def switchBlock: Iterator[ControlStructure] =
    cpg.controlStructure.isSwitch

  @Doc(info = "All try blocks (`ControlStructure` nodes)")
  def tryBlock: Iterator[ControlStructure] =
    cpg.controlStructure.isTry

  @Doc(info = "All while blocks (`ControlStructure` nodes)")
  def whileBlock: Iterator[ControlStructure] =
    cpg.controlStructure.isWhile

}
