/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.cattleframework.utils.http;

import java.net.ConnectException;
import java.util.Iterator;
import java.util.Map;

import org.apache.commons.collections4.MapUtils;
import org.cattleframework.exception.CattleException;
import org.cattleframework.exception.CattleResponseException;
import org.cattleframework.exception.ExceptionConstants;
import org.cattleframework.exception.ExceptionWrapUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.ResourceAccessException;
import org.springframework.web.client.RestClientResponseException;
import org.springframework.web.client.RestTemplate;

/**
 * Rest模板工具
 * 
 * @author orange
 *
 */
public class RestTemplateUtils {

    private static final Logger logger = LoggerFactory.getLogger(RestTemplateUtils.class);

    private static final String MESSAGE = "message";

    private static final String EXCEPTION_CODE = "error-code";

    private static final String EXCEPTION_RESPONSE = "response";

    private static final String MESSAGE_SOURCE = "source";

    private static final String SOURCE_OWNER = "cattle";

    private final RestTemplate restTemplate;

    public RestTemplateUtils(RestTemplate restTemplate) {
	this.restTemplate = restTemplate;
    }

    public <T, R> T post(String url, MediaType type, Map<String, String> headers, R request,
	    ParameterizedTypeReference<T> responseType) {
	HttpHeaders httpHeaders = getHeaders(type, headers);
	HttpEntity<R> requestEntity = new HttpEntity<R>(request, httpHeaders);
	try {
	    ResponseEntity<T> responseEntity = restTemplate.exchange(url, HttpMethod.POST, requestEntity, responseType);
	    return responseEntity.getBody();
	} catch (Throwable e) {
	    throw throwException(e);
	}
    }

    public <T, R> T post(String url, MediaType type, Map<String, String> headers, R request, Class<T> responseType) {
	HttpHeaders httpHeaders = getHeaders(type, headers);
	HttpEntity<R> requestEntity = new HttpEntity<R>(request, httpHeaders);
	try {
	    ResponseEntity<T> responseEntity = restTemplate.exchange(url, HttpMethod.POST, requestEntity, responseType);
	    return responseEntity.getBody();
	} catch (Throwable e) {
	    throw throwException(e);
	}
    }

    private HttpHeaders getHeaders(MediaType type, Map<String, String> headers) {
	HttpHeaders httpHeaders = new HttpHeaders();
	if (type != null) {
	    httpHeaders.setContentType(type);
	}
	if (MapUtils.isNotEmpty(headers)) {
	    Iterator<String> iterator = headers.keySet().iterator();
	    while (iterator.hasNext()) {
		String header = iterator.next();
		httpHeaders.add(header, headers.get(header));
	    }
	}
	return httpHeaders;
    }

    public <T> T get(String url, MediaType type, Map<String, String> headers,
	    ParameterizedTypeReference<T> responseType) {
	return get(url, type, headers, responseType, null);
    }

    public <T> T get(String url, MediaType type, Map<String, String> headers,
	    ParameterizedTypeReference<T> responseType, Map<String, ?> uriVariables) {
	HttpHeaders httpHeaders = getHeaders(type, headers);
	HttpEntity<Object> requestEntity = new HttpEntity<Object>(httpHeaders);
	try {
	    ResponseEntity<T> responseEntity;
	    if (MapUtils.isNotEmpty(uriVariables)) {
		responseEntity = restTemplate.exchange(url, HttpMethod.GET, requestEntity, responseType, uriVariables);
	    } else {
		responseEntity = restTemplate.exchange(url, HttpMethod.GET, requestEntity, responseType);
	    }
	    return responseEntity.getBody();
	} catch (Throwable e) {
	    throw throwException(e);
	}
    }

    public <T> T get(String url, MediaType type, Map<String, String> headers, Class<T> responseType) {
	return get(url, type, headers, responseType, null);
    }

    public <T> T get(String url, MediaType type, Map<String, String> headers, Class<T> responseType,
	    Map<String, ?> uriVariables) {
	HttpHeaders httpHeaders = getHeaders(type, headers);
	HttpEntity<Object> requestEntity = new HttpEntity<Object>(httpHeaders);
	try {
	    ResponseEntity<T> responseEntity;
	    if (MapUtils.isNotEmpty(uriVariables)) {
		responseEntity = restTemplate.exchange(url, HttpMethod.GET, requestEntity, responseType, uriVariables);
	    } else {
		responseEntity = restTemplate.exchange(url, HttpMethod.GET, requestEntity, responseType);
	    }
	    return responseEntity.getBody();
	} catch (Throwable e) {
	    throw throwException(e);
	}
    }

    private CattleException throwException(Throwable e) {
	if (e instanceof RestClientResponseException) {
	    RestClientResponseException ex = (RestClientResponseException) e;
	    return throwRestClientResponseException(ex);
	} else if (e instanceof IllegalStateException) {
	    return new CattleException(ExceptionConstants.SERVICE_UNAVAILABLE, e.getMessage(), e);
	} else if (e instanceof ResourceAccessException) {
	    ResourceAccessException ex = (ResourceAccessException) e;
	    if (ex.getCause() != null && ex.getCause() instanceof ConnectException) {
		return new CattleException(ExceptionConstants.SERVICE_UNAVAILABLE, e.getMessage(), e);
	    }
	}
	return ExceptionWrapUtils.wrap(e);
    }

    private CattleException throwRestClientResponseException(RestClientResponseException e) {
	CattleException exception = null;
	int exceptionCode = ExceptionConstants.GENERAL;
	if (e.getStatusCode().isSameCodeAs(HttpStatus.SERVICE_UNAVAILABLE)) {
	    exceptionCode = ExceptionConstants.SERVICE_UNAVAILABLE;
	} else {
	    logger.error(e.getMessage(), e);
	}
	if (e.getResponseHeaders().containsKey(MESSAGE_SOURCE)
		&& SOURCE_OWNER.equals(e.getResponseHeaders().getFirst(MESSAGE_SOURCE))) {
	    Map<String, Object> errorResponse = e
		    .getResponseBodyAs(new ParameterizedTypeReference<Map<String, Object>>() {
		    });
	    String message = (String) errorResponse.get(MESSAGE);
	    Integer code = null;
	    if (errorResponse.containsKey(EXCEPTION_CODE)) {
		code = (Integer) errorResponse.get(EXCEPTION_CODE);
	    }
	    if (errorResponse.containsKey(EXCEPTION_RESPONSE)) {
		String response = (String) errorResponse.get(EXCEPTION_RESPONSE);
		exception = code != null ? new CattleResponseException(code, message, response)
			: new CattleResponseException(message, response);
	    } else {
		exception = code != null ? new CattleException(code, message) : new CattleException(message);
	    }
	}
	if (exception == null) {
	    String responseBody = e.getResponseBodyAsString();
	    exception = new CattleResponseException(exceptionCode, e.getMessage(), responseBody);
	}
	return exception;
    }
}