/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.cattleframework.utils.concurrent;

import static java.util.Objects.requireNonNull;

import java.lang.Thread.UncaughtExceptionHandler;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicLong;

import org.apache.commons.lang3.StringUtils;
import org.cattleframework.exception.CattleException;

/**
 * 命名线程工厂生成器
 * 
 * @author orange
 *
 */
public final class NamedThreadFactoryBuilder {

    private String component = null;

    private Boolean daemon = null;

    private Integer priority = null;

    private UncaughtExceptionHandler uncaughtExceptionHandler = null;

    private ThreadFactory backingThreadFactory = null;

    public NamedThreadFactoryBuilder() {
    }

    public NamedThreadFactoryBuilder setComponent(String component) {
	this.component = component;
	return this;
    }

    public NamedThreadFactoryBuilder setDaemon(Boolean daemon) {
	this.daemon = daemon;
	return this;
    }

    public NamedThreadFactoryBuilder setPriority(Integer priority) {
	if (priority < Thread.MIN_PRIORITY) {
	    throw new CattleException(String.format("线程优先级(%s)必须大于等于%s", priority, Thread.MIN_PRIORITY));
	}
	if (priority > Thread.MAX_PRIORITY) {
	    throw new CattleException(String.format("线程优先级(%s)必须小于等于%s", priority, Thread.MAX_PRIORITY));
	}
	this.priority = priority;
	return this;
    }

    public NamedThreadFactoryBuilder setUncaughtExceptionHandler(UncaughtExceptionHandler uncaughtExceptionHandler) {
	this.uncaughtExceptionHandler = uncaughtExceptionHandler;
	return this;
    }

    public NamedThreadFactoryBuilder setThreadFactory(ThreadFactory backingThreadFactory) {
	this.backingThreadFactory = backingThreadFactory;
	return this;
    }

    public ThreadFactory build() {
	return doBuild(this);
    }

    private static ThreadFactory doBuild(NamedThreadFactoryBuilder builder) {
	String component = builder.component;
	Boolean daemon = builder.daemon;
	Integer priority = builder.priority;
	UncaughtExceptionHandler uncaughtExceptionHandler = builder.uncaughtExceptionHandler;
	ThreadFactory backingThreadFactory = builder.backingThreadFactory != null ? builder.backingThreadFactory
		: Executors.defaultThreadFactory();
	AtomicLong count = new AtomicLong(0);
	return r -> {
	    Thread thread = backingThreadFactory.newThread(r);
	    if (thread == null) {
		throw new CattleException("线程为空");
	    }
	    if (StringUtils.isNotBlank(component)) {
		thread.setName(String.format("cattle-%s-%d", component, requireNonNull(count).getAndIncrement()));
	    }
	    if (daemon != null) {
		thread.setDaemon(daemon);
	    }
	    if (priority != null) {
		thread.setPriority(priority);
	    }
	    if (uncaughtExceptionHandler != null) {
		thread.setUncaughtExceptionHandler(uncaughtExceptionHandler);
	    }
	    return thread;
	};
    }
}