/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.cattleframework.utils.cache;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import org.apache.commons.lang3.StringUtils;
import org.cattleframework.exception.CattleException;

/**
 * 缓存管理器抽象
 * 
 * @author orange
 *
 */
public abstract class AbstractCacheManager implements CacheManager {

    private final ConcurrentMap<String, Cache> caches = new ConcurrentHashMap<String, Cache>();

    /**
     * 创建缓存
     * 
     * @param name  名称
     * @param props 属性
     * @return 缓存
     */
    protected abstract Cache createCache(String name, Map<String, String> props);

    @Override
    public Cache getCache(String name, Map<String, String> props) {
	if (StringUtils.isBlank(name)) {
	    throw new CattleException("缓存分类键值为空");
	}
	if (!caches.containsKey(name)) {
	    putCache(name, props);
	}
	return caches.get(name);
    }

    private synchronized void putCache(String name, Map<String, String> props) {
	if (props == null) {
	    throw new CattleException("缓存属性为空");
	}
	Cache cache = createCache(name, props);
	if (cache == null) {
	    throw new CattleException("缓存分类键值'" + name + "'创建失败");
	}
	caches.put(name, cache);
    }

    @Override
    public void removeCache(String name) {
	if (StringUtils.isBlank(name)) {
	    throw new CattleException("缓存分类键值为空");
	}
	if (caches.containsKey(name)) {
	    Cache cache = caches.get(name);
	    cache.clear();
	}
    }
}