/*
 * Decompiled with CFR 0.152.
 */
package io.datarouter.filesystem.snapshot.writer;

import io.datarouter.filesystem.snapshot.compress.CompressedBlock;
import io.datarouter.filesystem.snapshot.path.SnapshotPaths;
import io.datarouter.filesystem.snapshot.storage.file.FileKey;
import io.datarouter.filesystem.snapshot.storage.file.SnapshotFileStorage;
import io.datarouter.filesystem.snapshot.writer.BlockQueue;
import io.datarouter.filesystem.snapshot.writer.SnapshotWriterConfig;
import io.datarouter.filesystem.snapshot.writer.SnapshotWriterTracker;
import io.datarouter.util.number.NumberFormatter;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;
import java.util.stream.IntStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SnapshotFileWriter {
    private static final Logger logger = LoggerFactory.getLogger(SnapshotFileWriter.class);
    private static final boolean LOG_FLUSH_START = false;
    private static final boolean LOG_QUEUE_STATS = true;
    private final SnapshotWriterTracker tracker;
    private final SnapshotFileStorage fileStorage;
    private final SnapshotWriterConfig config;
    private final SnapshotPaths paths;
    private final Consumer<Void> onValueFileWriteCallback;
    private final Consumer<Void> onLeafFileWriteCallback;
    private final Consumer<Integer> onBranchFileWriteCallback;
    public final Map<Integer, BlockQueue> branchBlockQueueByLevel;
    public final BlockQueue leafBlockQueue;
    public final List<BlockQueue> valueBlockQueueByColumn;

    public SnapshotFileWriter(SnapshotWriterTracker tracker, SnapshotFileStorage snapshotFileStorage, SnapshotWriterConfig config, Consumer<Void> onValueFileWriteCallback, Consumer<Void> onLeafFileWriteCallback, Consumer<Integer> onBranchFileWriteCallback) {
        this.tracker = tracker;
        this.fileStorage = snapshotFileStorage;
        this.config = config;
        this.paths = config.pathsSupplier.get();
        this.onValueFileWriteCallback = onValueFileWriteCallback;
        this.onLeafFileWriteCallback = onLeafFileWriteCallback;
        this.onBranchFileWriteCallback = onBranchFileWriteCallback;
        this.branchBlockQueueByLevel = new ConcurrentHashMap<Integer, BlockQueue>();
        this.leafBlockQueue = new BlockQueue("leaf", config.leafBytesPerFile, config.leafBlocksPerFile);
        this.valueBlockQueueByColumn = config.columnIds().map(columnId -> new BlockQueue("branch-" + columnId, snapshotWriterConfig.valueBytesPerFile, snapshotWriterConfig.valueBlocksPerFile)).list();
    }

    public void addBranchBlock(int level, int blockId, CompressedBlock block) {
        this.tracker.branchMemory(true, 1, block.totalLength);
        BlockQueue queue = this.branchBlockQueueByLevel.computeIfAbsent(level, $ -> new BlockQueue("branch-" + level, this.config.branchBytesPerFile, this.config.branchBlocksPerFile));
        queue.submit(blockId, block).forEach(file -> this.writeBranchFile(level, (BlockQueue.SnapshotFile)file));
    }

    private void writeBranchFile(int level, BlockQueue.SnapshotFile file) {
        if (this.config.persist) {
            this.logWriteStart(file);
            long startMs = System.currentTimeMillis();
            this.fileStorage.addBranchFile(this.paths, FileKey.branch(level, file.id), file.compressedBlocks);
            long ms = System.currentTimeMillis() - startMs;
            this.logWriteEnd(file, ms);
        }
        this.onBranchFileWriteCallback.accept(level);
        this.tracker.branchMemory(false, file.compressedBlocks.count, file.compressedBlocks.totalLength);
    }

    public void completeBranches(int level) {
        BlockQueue queue = this.branchBlockQueueByLevel.get(level);
        queue.takeLastFiles().forEach(file -> this.writeBranchFile(level, (BlockQueue.SnapshotFile)file));
        queue.assertEmpty();
    }

    public void addLeafBlock(int blockId, CompressedBlock block) {
        this.tracker.leafMemory(true, 1, block.totalLength);
        this.leafBlockQueue.submit(blockId, block).forEach(this::writeLeafFile);
    }

    private void writeLeafFile(BlockQueue.SnapshotFile file) {
        if (this.config.persist) {
            this.logWriteStart(file);
            long startMs = System.currentTimeMillis();
            this.fileStorage.addLeafFile(this.paths, FileKey.leaf(file.id), file.compressedBlocks);
            long ms = System.currentTimeMillis() - startMs;
            this.logWriteEnd(file, ms);
        }
        this.onLeafFileWriteCallback.accept(null);
        this.tracker.leafMemory(false, file.compressedBlocks.count, file.compressedBlocks.totalLength);
    }

    public void completeLeaves() {
        this.leafBlockQueue.takeLastFiles().forEach(this::writeLeafFile);
        this.leafBlockQueue.assertEmpty();
    }

    public void addValueBlock(int column, int blockId, CompressedBlock block) {
        this.tracker.valueMemory(true, 1, block.totalLength);
        this.valueBlockQueueByColumn.get(column).submit(blockId, block).forEach(file -> this.writeValueFile(column, (BlockQueue.SnapshotFile)file));
    }

    private void writeValueFile(int column, BlockQueue.SnapshotFile file) {
        if (this.config.persist) {
            this.logWriteStart(file);
            long startMs = System.currentTimeMillis();
            this.fileStorage.addValueFile(this.paths, FileKey.value(column, file.id), file.compressedBlocks);
            long ms = System.currentTimeMillis() - startMs;
            this.logWriteEnd(file, ms);
        }
        this.onValueFileWriteCallback.accept(null);
        this.tracker.valueMemory(false, file.compressedBlocks.count, file.compressedBlocks.totalLength);
    }

    public void completeValues() {
        this.config.columnIds().forEach(column -> {
            BlockQueue queue = this.valueBlockQueueByColumn.get((int)column);
            queue.takeLastFiles().forEach(file -> this.writeValueFile((int)column, (BlockQueue.SnapshotFile)file));
            queue.assertEmpty();
        });
    }

    private void logWriteStart(BlockQueue.SnapshotFile file) {
    }

    private void logWriteEnd(BlockQueue.SnapshotFile file, long ms) {
        double megabytes = (double)file.compressedBlocks.totalLength / 1024.0 / 1024.0;
        double seconds = (double)ms / 1000.0;
        double megabytesPerSecond = megabytes / seconds;
        logger.info("wrote BlockFile {}, {} ms, {} MBps", new Object[]{file.getFlushLog(), ms, NumberFormatter.format((Number)megabytesPerSecond, (int)3)});
    }

    public void logQueueStats() {
        IntStream.range(0, this.valueBlockQueueByColumn.size()).forEach(column -> logger.info("column={}, valueSingleEndingChecks={}, valueMultiEndingChecks={}", new Object[]{column, this.valueBlockQueueByColumn.get((int)column).numSingleEndingChecks, this.valueBlockQueueByColumn.get((int)column).numMultiEndingChecks}));
        logger.info("leafSingleEndingChecks={}, leafMultiEndingChecks={}", (Object)this.leafBlockQueue.numSingleEndingChecks, (Object)this.leafBlockQueue.numMultiEndingChecks);
        IntStream.range(0, this.branchBlockQueueByLevel.size()).forEach(level -> logger.info("level={}, branchSingleEndingChecks={}, branchMultiEndingChecks={}", new Object[]{level, this.branchBlockQueueByLevel.get((Object)Integer.valueOf((int)level)).numSingleEndingChecks, this.branchBlockQueueByLevel.get((Object)Integer.valueOf((int)level)).numMultiEndingChecks}));
    }

    public boolean valueFileInfoReady(int column, int firstBlockId, int numBlocks) {
        return this.valueBlockQueueByColumn.get(column).isReady(firstBlockId, numBlocks);
    }

    public BlockQueue.FileIdsAndEndings valueFileInfo(int column, int firstBlockId, int numBlocks) {
        return this.valueBlockQueueByColumn.get(column).fileIdsAndEndings(firstBlockId, numBlocks);
    }

    public boolean leafFileInfoReady(int firstBlockId, int numBlocks) {
        return this.leafBlockQueue.isReady(firstBlockId, numBlocks);
    }

    public BlockQueue.FileIdsAndEndings leafFileInfo(int firstBlockId, int numBlocks) {
        return this.leafBlockQueue.fileIdsAndEndings(firstBlockId, numBlocks);
    }

    public boolean branchFileInfoReady(int level, int firstBlockId, int numBlocks) {
        return this.branchBlockQueueByLevel.get(level).isReady(firstBlockId, numBlocks);
    }

    public BlockQueue.FileIdsAndEndings branchFileInfo(int level, int firstBlockId, int numBlocks) {
        return this.branchBlockQueueByLevel.get(level).fileIdsAndEndings(firstBlockId, numBlocks);
    }

    public int rootBranchEnding() {
        int maxLevel = this.branchBlockQueueByLevel.size() - 1;
        return this.branchBlockQueueByLevel.get(maxLevel).ending(0);
    }
}

