package io.ciogram.types

import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.InternalSerializationApi
import kotlinx.serialization.KSerializer
import kotlinx.serialization.Serializable
import kotlinx.serialization.descriptors.PolymorphicKind
import kotlinx.serialization.descriptors.PrimitiveKind
import kotlinx.serialization.descriptors.PrimitiveSerialDescriptor
import kotlinx.serialization.descriptors.buildSerialDescriptor
import kotlinx.serialization.encoding.Decoder
import kotlinx.serialization.encoding.Encoder
import okio.Source

@Serializable(with = InputFileSerializer::class)
public sealed class InputFile

@OptIn(InternalSerializationApi::class, ExperimentalSerializationApi::class)
internal object InputFileSerializer : KSerializer<InputFile> {
    override val descriptor = buildSerialDescriptor("InputFile", PolymorphicKind.SEALED)

    override fun serialize(encoder: Encoder, value: InputFile) {
        when (value) {
            is RemoteInputFile -> RemoteInputFileSerializer.serialize(encoder, value)
            is LocalInputFile -> LocalInputFileSerializer.serialize(encoder, value)
        }
    }

    override fun deserialize(decoder: Decoder): InputFile {
        throw UnsupportedOperationException("Deserialization is not supported for this type")
    }
}

@Serializable(with = RemoteInputFileSerializer::class)
public data class RemoteInputFile(public val fileId: String) : InputFile()

internal object RemoteInputFileSerializer : KSerializer<RemoteInputFile> {
    override val descriptor = PrimitiveSerialDescriptor("RemoteInputFile", PrimitiveKind.STRING)

    override fun serialize(encoder: Encoder, value: RemoteInputFile) {
        encoder.encodeString(value.fileId)
    }

    override fun deserialize(decoder: Decoder): RemoteInputFile {
        throw UnsupportedOperationException("Deserialization is not supported for this type")
    }
}

@Serializable(with = LocalInputFileSerializer::class)
public data class LocalInputFile(public val fileName: String, public val sourceProvider: () -> Source) : InputFile()

internal object LocalInputFileSerializer : KSerializer<LocalInputFile> {
    override val descriptor = PrimitiveSerialDescriptor("LocalInputFile", PrimitiveKind.STRING)

    override fun serialize(encoder: Encoder, value: LocalInputFile) {
        encoder.encodeString("attach://${value.fileName}")
    }

    override fun deserialize(decoder: Decoder): LocalInputFile {
        throw UnsupportedOperationException("Deserialization is not supported for this type")
    }
}