package io.ciogram.types

import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonContentPolymorphicSerializer
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.jsonObject
import kotlinx.serialization.json.jsonPrimitive

@Serializable(with = InlineQueryResultSerializer::class)
public sealed class InlineQueryResult {
    public abstract val type: String
}

internal object InlineQueryResultSerializer : JsonContentPolymorphicSerializer<InlineQueryResult>(InlineQueryResult::class) {
    override fun selectDeserializer(element: JsonElement): DeserializationStrategy<out InlineQueryResult> = when (val type = element.jsonObject["type"]?.jsonPrimitive?.content) {
        "article" -> InlineQueryResultArticle.serializer()
        "audio" -> if ("audio_url" in element.jsonObject) InlineQueryResultAudio.serializer() else InlineQueryResultCachedAudio.serializer()
        "contact" -> InlineQueryResultContact.serializer()
        "game" -> InlineQueryResultGame.serializer()
        "document" -> if ("document_url" in element.jsonObject) InlineQueryResultDocument.serializer() else InlineQueryResultCachedDocument.serializer()
        "gif" -> if ("gif_url" in element.jsonObject) InlineQueryResultGif.serializer() else InlineQueryResultCachedGif.serializer()
        "location" -> InlineQueryResultLocation.serializer()
        "mpeg4_gif" -> if ("mpeg4_url" in element.jsonObject) InlineQueryResultMpeg4Gif.serializer() else InlineQueryResultCachedMpeg4Gif.serializer()
        "photo" -> if ("photo_url" in element.jsonObject) InlineQueryResultPhoto.serializer() else InlineQueryResultCachedPhoto.serializer()
        "venue" -> InlineQueryResultVenue.serializer()
        "video" -> if ("video_url" in element.jsonObject) InlineQueryResultVideo.serializer() else InlineQueryResultCachedVideo.serializer()
        "voice" -> if ("voice_url" in element.jsonObject) InlineQueryResultVoice.serializer() else InlineQueryResultCachedVoice.serializer()
        "sticker" -> InlineQueryResultCachedSticker.serializer()
        else -> throw IllegalArgumentException("Type \"$type\" is unsupported for the inline query result")
    }
}

@Serializable
public data class InlineQueryResultArticle(
    public val id: String,
    public val title: String,
    @SerialName("input_message_content") public val inputMessageContent: InputMessageContent,
    @SerialName("reply_markup") public val replyMarkup: InlineKeyboardMarkup? = null,
    public val url: String? = null,
    @SerialName("hide_url") public val hideUrl: Boolean? = null,
    public val description: String? = null,
    @SerialName("thumb_url") public val thumbUrl: String? = null,
    @SerialName("thumb_width") public val thumbWidth: Int? = null,
    @SerialName("thumb_height") public val thumbHeight: Int? = null
) : InlineQueryResult() {
    public override val type: String = "article"
}

@Serializable
public data class InlineQueryResultPhoto(
    public val id: String,
    @SerialName("photo_url") public val photoUrl: String,
    @SerialName("thumb_url") public val thumbUrl: String,
    @SerialName("photo_width") public val photoWidth: Int? = null,
    @SerialName("photo_height") public val photoHeight: Int? = null,
    public val title: String? = null,
    public val description: String? = null,
    public val caption: String? = null,
    @SerialName("parse_mode") public val parseMode: String? = null,
    @SerialName("caption_entities") public val captionEntities: List<MessageEntity>? = null,
    @SerialName("reply_markup") public val replyMarkup: InlineKeyboardMarkup? = null,
    @SerialName("input_message_content") public val inputMessageContent: InputMessageContent? = null
) : InlineQueryResult() {
    public override val type: String = "photo"
}

@Serializable
public data class InlineQueryResultGif(
    public val id: String,
    @SerialName("gif_url") public val gifUrl: String,
    @SerialName("gif_width") public val gifWidth: Int? = null,
    @SerialName("gif_height") public val gifHeight: Int? = null,
    @SerialName("gif_duration") public val gifDuration: Int? = null,
    @SerialName("thumb_url") public val thumbUrl: String,
    @SerialName("thumb_mime_type") public val thumbMimeType: String? = null,
    public val title: String? = null,
    public val caption: String? = null,
    @SerialName("parse_mode") public val parseMode: String? = null,
    @SerialName("caption_entities") public val captionEntities: List<MessageEntity>? = null,
    @SerialName("reply_markup") public val replyMarkup: InlineKeyboardMarkup? = null,
    @SerialName("input_message_content") public val inputMessageContent: InputMessageContent? = null
) : InlineQueryResult() {
    public override val type: String = "gif"
}

@Serializable
public data class InlineQueryResultMpeg4Gif(
    public val id: String,
    @SerialName("mpeg4_url") public val mpeg4Url: String,
    @SerialName("mpeg4_width") public val mpeg4Width: Int? = null,
    @SerialName("mpeg4_height") public val mpeg4Height: Int? = null,
    @SerialName("mpeg4_duration") public val mpeg4Duration: Int? = null,
    @SerialName("thumb_url") public val thumbUrl: String,
    @SerialName("thumb_mime_type") public val thumbMimeType: String? = null,
    public val title: String? = null,
    public val caption: String? = null,
    @SerialName("parse_mode") public val parseMode: String? = null,
    @SerialName("caption_entities") public val captionEntities: List<MessageEntity>? = null,
    @SerialName("reply_markup") public val replyMarkup: InlineKeyboardMarkup? = null,
    @SerialName("input_message_content") public val inputMessageContent: InputMessageContent? = null
) : InlineQueryResult() {
    public override val type: String = "mpeg4_gif"
}

@Serializable
public data class InlineQueryResultVideo(
    public val id: String,
    @SerialName("video_url") public val videoUrl: String,
    @SerialName("mime_type") public val mimeType: String,
    @SerialName("thumb_url") public val thumbUrl: String,
    public val title: String,
    public val caption: String? = null,
    @SerialName("parse_mode") public val parseMode: String? = null,
    @SerialName("caption_entities") public val captionEntities: List<MessageEntity>? = null,
    @SerialName("video_width") public val videoWidth: Int? = null,
    @SerialName("video_height") public val videoHeight: Int? = null,
    @SerialName("video_duration") public val videoDuration: Int? = null,
    public val description: String? = null,
    @SerialName("reply_markup") public val replyMarkup: InlineKeyboardMarkup? = null,
    @SerialName("input_message_content") public val inputMessageContent: InputMessageContent? = null
) : InlineQueryResult() {
    public override val type: String = "video"
}

@Serializable
public data class InlineQueryResultAudio(
    public val id: String,
    @SerialName("audio_url") public val audioUrl: String,
    public val title: String,
    public val caption: String? = null,
    @SerialName("parse_mode") public val parseMode: String? = null,
    @SerialName("caption_entities") public val captionEntities: List<MessageEntity>? = null,
    public val performer: String? = null,
    @SerialName("audio_duration") public val audioDuration: Int? = null,
    @SerialName("reply_markup") public val replyMarkup: InlineKeyboardMarkup? = null,
    @SerialName("input_message_content") public val inputMessageContent: InputMessageContent? = null
) : InlineQueryResult() {
    public override val type: String = "audio"
}

@Serializable
public data class InlineQueryResultVoice(
    public val id: String,
    @SerialName("voice_url") public val voiceUrl: String,
    public val title: String,
    public val caption: String? = null,
    @SerialName("parse_mode") public val parseMode: String? = null,
    @SerialName("caption_entities") public val captionEntities: List<MessageEntity>? = null,
    @SerialName("voice_duration") public val voiceDuration: Int? = null,
    @SerialName("reply_markup") public val replyMarkup: InlineKeyboardMarkup? = null,
    @SerialName("input_message_content") public val inputMessageContent: InputMessageContent? = null
) : InlineQueryResult() {
    public override val type: String = "voice"
}

@Serializable
public data class InlineQueryResultDocument(
    public val id: String,
    public val title: String,
    public val caption: String? = null,
    @SerialName("parse_mode") public val parseMode: String? = null,
    @SerialName("caption_entities") public val captionEntities: List<MessageEntity>? = null,
    @SerialName("document_url") public val documentUrl: String,
    @SerialName("mime_type") public val mimeType: String,
    public val description: String? = null,
    @SerialName("reply_markup") public val replyMarkup: InlineKeyboardMarkup? = null,
    @SerialName("input_message_content") public val inputMessageContent: InputMessageContent? = null,
    @SerialName("thumb_url") public val thumbUrl: String? = null,
    @SerialName("thumb_width") public val thumbWidth: Int? = null,
    @SerialName("thumb_height") public val thumbHeight: Int? = null
) : InlineQueryResult() {
    public override val type: String = "document"
}

@Serializable
public data class InlineQueryResultLocation(
    public val id: String,
    public val latitude: Float,
    public val longitude: Float,
    public val title: String,
    @SerialName("horizontal_accuracy") public val horizontalAccuracy: Float? = null,
    @SerialName("live_period") public val livePeriod: Int? = null,
    public val heading: Int? = null,
    @SerialName("proximity_alert_radius") public val proximityAlertRadius: Int? = null,
    @SerialName("reply_markup") public val replyMarkup: InlineKeyboardMarkup? = null,
    @SerialName("input_message_content") public val inputMessageContent: InputMessageContent? = null,
    @SerialName("thumb_url") public val thumbUrl: String? = null,
    @SerialName("thumb_width") public val thumbWidth: Int? = null,
    @SerialName("thumb_height") public val thumbHeight: Int? = null
) : InlineQueryResult() {
    public override val type: String = "location"
}

@Serializable
public data class InlineQueryResultVenue(
    public val id: String,
    public val latitude: Float,
    public val longitude: Float,
    public val title: String,
    public val address: String,
    @SerialName("foursquare_id") public val foursquareId: String? = null,
    @SerialName("foursquare_type") public val foursquareType: String? = null,
    @SerialName("google_place_id") public val googlePlaceId: String? = null,
    @SerialName("google_place_type") public val googlePlaceType: String? = null,
    @SerialName("reply_markup") public val replyMarkup: InlineKeyboardMarkup? = null,
    @SerialName("input_message_content") public val inputMessageContent: InputMessageContent? = null,
    @SerialName("thumb_url") public val thumbUrl: String? = null,
    @SerialName("thumb_width") public val thumbWidth: Int? = null,
    @SerialName("thumb_height") public val thumbHeight: Int? = null
) : InlineQueryResult() {
    public override val type: String = "venue"
}

@Serializable
public data class InlineQueryResultContact(
    public val id: String,
    @SerialName("phone_number") public val phoneNumber: String,
    @SerialName("first_name") public val firstName: String,
    @SerialName("last_name") public val lastName: String? = null,
    public val vcard: String? = null,
    @SerialName("reply_markup") public val replyMarkup: InlineKeyboardMarkup? = null,
    @SerialName("input_message_content") public val inputMessageContent: InputMessageContent? = null,
    @SerialName("thumb_url") public val thumbUrl: String? = null,
    @SerialName("thumb_width") public val thumbWidth: Int? = null,
    @SerialName("thumb_height") public val thumbHeight: Int? = null
) : InlineQueryResult() {
    public override val type: String = "contact"
}

@Serializable
public data class InlineQueryResultGame(
    public val id: String,
    @SerialName("game_short_name") public val gameShortName: String,
    @SerialName("reply_markup") public val replyMarkup: InlineKeyboardMarkup? = null
) : InlineQueryResult() {
    public override val type: String = "game"
}

@Serializable
public data class InlineQueryResultCachedPhoto(
    public val id: String,
    @SerialName("photo_file_id") public val photoFileId: String,
    public val title: String? = null,
    public val description: String? = null,
    public val caption: String? = null,
    @SerialName("parse_mode") public val parseMode: String? = null,
    @SerialName("caption_entities") public val captionEntities: List<MessageEntity>? = null,
    @SerialName("reply_markup") public val replyMarkup: InlineKeyboardMarkup? = null,
    @SerialName("input_message_content") public val inputMessageContent: InputMessageContent? = null
) : InlineQueryResult() {
    public override val type: String = "photo"
}

@Serializable
public data class InlineQueryResultCachedGif(
    public val id: String,
    @SerialName("gif_file_id") public val gifFileId: String,
    public val title: String? = null,
    public val caption: String? = null,
    @SerialName("parse_mode") public val parseMode: String? = null,
    @SerialName("caption_entities") public val captionEntities: List<MessageEntity>? = null,
    @SerialName("reply_markup") public val replyMarkup: InlineKeyboardMarkup? = null,
    @SerialName("input_message_content") public val inputMessageContent: InputMessageContent? = null
) : InlineQueryResult() {
    public override val type: String = "gif"
}

@Serializable
public data class InlineQueryResultCachedMpeg4Gif(
    public val id: String,
    @SerialName("mpeg4_file_id") public val mpeg4FileId: String,
    public val title: String? = null,
    public val caption: String? = null,
    @SerialName("parse_mode") public val parseMode: String? = null,
    @SerialName("caption_entities") public val captionEntities: List<MessageEntity>? = null,
    @SerialName("reply_markup") public val replyMarkup: InlineKeyboardMarkup? = null,
    @SerialName("input_message_content") public val inputMessageContent: InputMessageContent? = null
) : InlineQueryResult() {
    public override val type: String = "mpeg4_gif"
}

@Serializable
public data class InlineQueryResultCachedSticker(
    public val id: String,
    @SerialName("sticker_file_id") public val stickerFileId: String,
    @SerialName("reply_markup") public val replyMarkup: InlineKeyboardMarkup? = null,
    @SerialName("input_message_content") public val inputMessageContent: InputMessageContent? = null
) : InlineQueryResult() {
    public override val type: String = "sticker"
}

@Serializable
public data class InlineQueryResultCachedDocument(
    public val id: String,
    public val title: String,
    @SerialName("document_file_id") public val documentFileId: String,
    public val description: String? = null,
    public val caption: String? = null,
    @SerialName("parse_mode") public val parseMode: String? = null,
    @SerialName("caption_entities") public val captionEntities: List<MessageEntity>? = null,
    @SerialName("reply_markup") public val replyMarkup: InlineKeyboardMarkup? = null,
    @SerialName("input_message_content") public val inputMessageContent: InputMessageContent? = null
) : InlineQueryResult() {
    public override val type: String = "document"
}

@Serializable
public data class InlineQueryResultCachedVideo(
    public val id: String,
    @SerialName("video_file_id") public val videoFileId: String,
    public val title: String,
    public val description: String? = null,
    public val caption: String? = null,
    @SerialName("parse_mode") public val parseMode: String? = null,
    @SerialName("caption_entities") public val captionEntities: List<MessageEntity>? = null,
    @SerialName("reply_markup") public val replyMarkup: InlineKeyboardMarkup? = null,
    @SerialName("input_message_content") public val inputMessageContent: InputMessageContent? = null
) : InlineQueryResult() {
    public override val type: String = "video"
}

@Serializable
public data class InlineQueryResultCachedVoice(
    public val id: String,
    @SerialName("voice_file_id") public val voiceFileId: String,
    public val title: String,
    public val caption: String? = null,
    @SerialName("parse_mode") public val parseMode: String? = null,
    @SerialName("caption_entities") public val captionEntities: List<MessageEntity>? = null,
    @SerialName("reply_markup") public val replyMarkup: InlineKeyboardMarkup? = null,
    @SerialName("input_message_content") public val inputMessageContent: InputMessageContent? = null
) : InlineQueryResult() {
    public override val type: String = "voice"
}

@Serializable
public data class InlineQueryResultCachedAudio(
    public val id: String,
    @SerialName("audio_file_id") public val audioFileId: String,
    public val caption: String? = null,
    @SerialName("parse_mode") public val parseMode: String? = null,
    @SerialName("caption_entities") public val captionEntities: List<MessageEntity>? = null,
    @SerialName("reply_markup") public val replyMarkup: InlineKeyboardMarkup? = null,
    @SerialName("input_message_content") public val inputMessageContent: InputMessageContent? = null
) : InlineQueryResult() {
    public override val type: String = "audio"
}