/*
 * Decompiled with CFR 0.152.
 */
package io.atomix.copycat.server.storage;

import io.atomix.catalyst.buffer.FileBuffer;
import io.atomix.catalyst.buffer.HeapBuffer;
import io.atomix.catalyst.buffer.MappedBuffer;
import io.atomix.catalyst.serializer.Serializer;
import io.atomix.catalyst.util.Assert;
import io.atomix.copycat.server.storage.Segment;
import io.atomix.copycat.server.storage.SegmentDescriptor;
import io.atomix.copycat.server.storage.SegmentFile;
import io.atomix.copycat.server.storage.Storage;
import io.atomix.copycat.server.storage.util.OffsetIndex;
import io.atomix.copycat.server.storage.util.OffsetPredicate;
import java.io.File;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.NavigableMap;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentSkipListMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SegmentManager
implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(SegmentManager.class);
    private static final int DEFAULT_BUFFER_SIZE = 0x100000;
    private final String name;
    private final Storage storage;
    private final Serializer serializer;
    private final NavigableMap<Long, Segment> segments = new ConcurrentSkipListMap<Long, Segment>();
    private Segment currentSegment;
    private long commitIndex;

    public SegmentManager(String name, Storage storage, Serializer serializer) {
        this.name = Assert.notNull(name, "name");
        this.storage = Assert.notNull(storage, "storage");
        this.serializer = Assert.notNull(serializer, "serializer");
        this.open();
    }

    public Serializer serializer() {
        return this.serializer;
    }

    SegmentManager commitIndex(long commitIndex) {
        this.commitIndex = Math.max(this.commitIndex, commitIndex);
        return this;
    }

    public long commitIndex() {
        return this.commitIndex;
    }

    private void open() {
        for (Segment segment : this.loadSegments()) {
            this.segments.put(segment.descriptor().index(), segment);
        }
        if (!this.segments.isEmpty()) {
            this.currentSegment = this.segments.lastEntry().getValue();
        } else {
            SegmentDescriptor descriptor = SegmentDescriptor.builder().withId(1L).withVersion(1L).withIndex(1L).withMaxSegmentSize(this.storage.maxSegmentSize()).withMaxEntries(this.storage.maxEntriesPerSegment()).build();
            descriptor.lock();
            this.currentSegment = this.createSegment(descriptor);
            this.currentSegment.descriptor().update(System.currentTimeMillis());
            this.currentSegment.descriptor().lock();
            this.segments.put(1L, this.currentSegment);
        }
    }

    private void assertOpen() {
        Assert.state(this.currentSegment != null, "segment manager not open", new Object[0]);
    }

    public Segment currentSegment() {
        return this.currentSegment != null ? this.currentSegment : this.lastSegment();
    }

    private void resetCurrentSegment() {
        Segment lastSegment = this.lastSegment();
        if (lastSegment != null) {
            this.currentSegment = lastSegment;
        } else {
            SegmentDescriptor descriptor = SegmentDescriptor.builder().withId(1L).withVersion(1L).withIndex(1L).withMaxSegmentSize(this.storage.maxSegmentSize()).withMaxEntries(this.storage.maxEntriesPerSegment()).build();
            descriptor.lock();
            this.currentSegment = this.createSegment(descriptor);
            this.segments.put(1L, this.currentSegment);
        }
    }

    public Segment firstSegment() {
        this.assertOpen();
        Map.Entry<Long, Segment> segment = this.segments.firstEntry();
        return segment != null ? segment.getValue() : null;
    }

    public Segment lastSegment() {
        this.assertOpen();
        Map.Entry<Long, Segment> segment = this.segments.lastEntry();
        return segment != null ? segment.getValue() : null;
    }

    public Segment nextSegment() {
        this.assertOpen();
        Segment lastSegment = this.lastSegment();
        SegmentDescriptor descriptor = SegmentDescriptor.builder().withId(lastSegment != null ? lastSegment.descriptor().id() + 1L : 1L).withVersion(1L).withIndex(this.currentSegment.lastIndex() + 1L).withMaxSegmentSize(this.storage.maxSegmentSize()).withMaxEntries(this.storage.maxEntriesPerSegment()).build();
        descriptor.lock();
        this.currentSegment = this.createSegment(descriptor);
        this.segments.put(descriptor.index(), this.currentSegment);
        return this.currentSegment;
    }

    public Collection<Segment> segments() {
        return this.segments.values();
    }

    public Collection<Segment> reverseSegments() {
        return this.segments.descendingMap().values();
    }

    public Segment segment(long index) {
        this.assertOpen();
        if (this.currentSegment != null && this.currentSegment.validIndex(index)) {
            return this.currentSegment;
        }
        Map.Entry<Long, Segment> segment = this.segments.floorEntry(index);
        return segment != null ? segment.getValue() : null;
    }

    public synchronized void replaceSegments(Collection<Segment> segments, Segment segment) {
        segment.descriptor().update(System.currentTimeMillis());
        segment.descriptor().lock();
        for (Segment oldSegment : segments) {
            if (!this.segments.containsKey(oldSegment.index())) {
                throw new IllegalArgumentException("unknown segment at index: " + oldSegment.index());
            }
            this.segments.remove(oldSegment.index());
        }
        this.segments.put(segment.index(), segment);
        this.resetCurrentSegment();
    }

    public synchronized void removeSegment(Segment segment) {
        this.segments.remove(segment.index());
        segment.close();
        segment.delete();
        this.resetCurrentSegment();
    }

    public Segment createSegment(SegmentDescriptor descriptor) {
        switch (this.storage.level()) {
            case MEMORY: {
                return this.createMemorySegment(descriptor);
            }
            case MAPPED: {
                if (descriptor.version() == 1L) {
                    return this.createMappedSegment(descriptor);
                }
                return this.createDiskSegment(descriptor);
            }
            case DISK: {
                return this.createDiskSegment(descriptor);
            }
        }
        throw new AssertionError();
    }

    private Segment createDiskSegment(SegmentDescriptor descriptor) {
        File segmentFile = SegmentFile.createSegmentFile(this.name, this.storage.directory(), descriptor.id(), descriptor.version());
        FileBuffer buffer = FileBuffer.allocate(segmentFile, Math.min(0x100000L, descriptor.maxSegmentSize()), Integer.MAX_VALUE);
        descriptor.copyTo(buffer);
        Segment segment = new Segment(new SegmentFile(segmentFile), buffer.slice(), descriptor, this.createIndex(descriptor), new OffsetPredicate(), this.serializer.clone(), this);
        LOGGER.debug("Created segment: {}", (Object)segment);
        return segment;
    }

    private Segment createMappedSegment(SegmentDescriptor descriptor) {
        File segmentFile = SegmentFile.createSegmentFile(this.name, this.storage.directory(), descriptor.id(), descriptor.version());
        MappedBuffer buffer = MappedBuffer.allocate(segmentFile, Math.min(0x100000L, descriptor.maxSegmentSize()), Integer.MAX_VALUE);
        descriptor.copyTo(buffer);
        Segment segment = new Segment(new SegmentFile(segmentFile), buffer.slice(), descriptor, this.createIndex(descriptor), new OffsetPredicate(), this.serializer.clone(), this);
        LOGGER.debug("Created segment: {}", (Object)segment);
        return segment;
    }

    private Segment createMemorySegment(SegmentDescriptor descriptor) {
        File segmentFile = SegmentFile.createSegmentFile(this.name, this.storage.directory(), descriptor.id(), descriptor.version());
        HeapBuffer buffer = HeapBuffer.allocate(Math.min(0x100000L, descriptor.maxSegmentSize()), Integer.MAX_VALUE);
        descriptor.copyTo(buffer);
        Segment segment = new Segment(new SegmentFile(segmentFile), buffer.slice(), descriptor, this.createIndex(descriptor), new OffsetPredicate(), this.serializer.clone(), this);
        LOGGER.debug("Created segment: {}", (Object)segment);
        return segment;
    }

    public Segment loadSegment(long segmentId, long segmentVersion) {
        switch (this.storage.level()) {
            case MEMORY: {
                return this.loadMemorySegment(segmentId, segmentVersion);
            }
            case MAPPED: {
                return this.loadMappedSegment(segmentId, segmentVersion);
            }
            case DISK: {
                return this.loadDiskSegment(segmentId, segmentVersion);
            }
        }
        throw new AssertionError();
    }

    private Segment loadDiskSegment(long segmentId, long segmentVersion) {
        File file = SegmentFile.createSegmentFile(this.name, this.storage.directory(), segmentId, segmentVersion);
        FileBuffer buffer = FileBuffer.allocate(file, Math.min(0x100000, this.storage.maxSegmentSize()), Integer.MAX_VALUE);
        SegmentDescriptor descriptor = new SegmentDescriptor(buffer);
        Segment segment = new Segment(new SegmentFile(file), buffer.position(64L).slice(), descriptor, this.createIndex(descriptor), new OffsetPredicate(), this.serializer.clone(), this);
        LOGGER.debug("Loaded file segment: {} ({})", (Object)descriptor.id(), (Object)file.getName());
        return segment;
    }

    private Segment loadMappedSegment(long segmentId, long segmentVersion) {
        File file = SegmentFile.createSegmentFile(this.name, this.storage.directory(), segmentId, segmentVersion);
        MappedBuffer buffer = MappedBuffer.allocate(file, Math.min(0x100000, this.storage.maxSegmentSize()), Integer.MAX_VALUE);
        SegmentDescriptor descriptor = new SegmentDescriptor(buffer);
        Segment segment = new Segment(new SegmentFile(file), buffer.position(64L).slice(), descriptor, this.createIndex(descriptor), new OffsetPredicate(), this.serializer.clone(), this);
        LOGGER.debug("Loaded mapped segment: {} ({})", (Object)descriptor.id(), (Object)file.getName());
        return segment;
    }

    private Segment loadMemorySegment(long segmentId, long segmentVersion) {
        File file = SegmentFile.createSegmentFile(this.name, this.storage.directory(), segmentId, segmentVersion);
        HeapBuffer buffer = HeapBuffer.allocate(Math.min(0x100000, this.storage.maxSegmentSize()), Integer.MAX_VALUE);
        SegmentDescriptor descriptor = new SegmentDescriptor(buffer);
        Segment segment = new Segment(new SegmentFile(file), buffer.position(64L).slice(), descriptor, this.createIndex(descriptor), new OffsetPredicate(), this.serializer.clone(), this);
        LOGGER.debug("Loaded memory segment: {}", (Object)descriptor.id());
        return segment;
    }

    private OffsetIndex createIndex(SegmentDescriptor descriptor) {
        return new OffsetIndex(HeapBuffer.allocate(Math.min(0x100000, descriptor.maxEntries()), OffsetIndex.size(descriptor.maxEntries())));
    }

    /*
     * Enabled aggressive block sorting
     */
    protected Collection<Segment> loadSegments() {
        this.storage.directory().mkdirs();
        TreeMap<Long, Segment> segments = new TreeMap<Long, Segment>();
        for (File file : this.storage.directory().listFiles(File::isFile)) {
            if (!SegmentFile.isSegmentFile(this.name, file)) continue;
            SegmentFile segmentFile = new SegmentFile(file);
            SegmentDescriptor descriptor = new SegmentDescriptor(FileBuffer.allocate(file, 64L));
            if (descriptor.locked()) {
                Segment segment;
                block10: {
                    segment = this.loadSegment(descriptor.id(), descriptor.version());
                    Map.Entry previousEntry = segments.floorEntry(segment.index());
                    if (previousEntry != null) {
                        Segment previousSegment = (Segment)previousEntry.getValue();
                        if (previousSegment.index() == segment.index()) {
                            if (segment.descriptor().version() > previousSegment.descriptor().version()) {
                                LOGGER.debug("Replaced segment {} with newer version: {} ({})", previousSegment.descriptor().id(), segment.descriptor().version(), segmentFile.file().getName());
                                segments.remove(previousEntry.getKey());
                                previousSegment.close();
                                previousSegment.delete();
                                break block10;
                            } else {
                                segment.close();
                                segment.delete();
                                continue;
                            }
                        }
                        if (previousSegment.index() + previousSegment.length() > segment.index()) {
                            segment.close();
                            segment.delete();
                            continue;
                        }
                    }
                }
                LOGGER.debug("Found segment: {} ({})", (Object)segment.descriptor().id(), (Object)segmentFile.file().getName());
                segments.put(segment.index(), segment);
                Map.Entry nextEntry = segments.higherEntry(segment.index());
                while (nextEntry != null && ((Segment)nextEntry.getValue()).index() < segment.index() + segment.length()) {
                    segments.remove(nextEntry.getKey());
                    nextEntry = segments.higherEntry(segment.index());
                }
                descriptor.close();
                continue;
            }
            LOGGER.debug("Deleting unlocked segment: {}-{} ({})", descriptor.id(), descriptor.version(), segmentFile.file().getName());
            descriptor.close();
            descriptor.delete();
        }
        Iterator iterator = segments.keySet().iterator();
        while (iterator.hasNext()) {
            Segment previousSegment;
            Long segmentId = (Long)iterator.next();
            Segment segment = (Segment)segments.get(segmentId);
            Map.Entry previousEntry = segments.floorEntry(segmentId - 1L);
            if (previousEntry == null || (previousSegment = (Segment)previousEntry.getValue()).index() + previousSegment.length() - 1L >= segment.index()) continue;
            previousSegment.skip(segment.index() - (previousSegment.index() + previousSegment.length()));
        }
        return segments.values();
    }

    @Override
    public void close() {
        this.segments.values().forEach(s -> {
            LOGGER.debug("Closing segment: {}", (Object)s.descriptor().id());
            s.close();
        });
        this.currentSegment = null;
    }

    public String toString() {
        return String.format("%s[directory=%s, segments=%d]", this.getClass().getSimpleName(), this.storage.directory(), this.segments.size());
    }
}

