/*
 * Copyright (c) 2020 GitLive Ltd.  Use of this source code is governed by the Apache 2.0 license.
 */

package dev.gitlive.firebase

import kotlinx.serialization.encoding.CompositeDecoder
import kotlinx.serialization.descriptors.PolymorphicKind
import kotlinx.serialization.descriptors.SerialDescriptor
import kotlinx.serialization.descriptors.StructureKind

actual fun FirebaseDecoder.structureDecoder(descriptor: SerialDescriptor): CompositeDecoder = when(descriptor.kind) {
        StructureKind.CLASS, StructureKind.OBJECT, PolymorphicKind.SEALED -> (value as Map<*, *>).let { map ->
            FirebaseClassDecoder(map.size, { map.containsKey(it) }) { desc, index ->
                val elementName = desc.getElementName(index)
                if (desc.kind is PolymorphicKind && elementName == "value") {
                    map
                } else {
                    map[desc.getElementName(index)]
                }
            }
        }
        StructureKind.LIST ->
            when(value) {
                is List<*> -> value
                is Map<*, *> -> value.asSequence()
                    .sortedBy { (it) -> it.toString().toIntOrNull() }
                    .map { (_, it) -> it }
                    .toList()
                else -> error("unexpected type, got $value when expecting a list")
            }
            .let { FirebaseCompositeDecoder(it.size) { _, index -> it[index] } }
        StructureKind.MAP -> (value as Map<*, *>).entries.toList().let {
            FirebaseCompositeDecoder(it.size) { _, index -> it[index/2].run { if(index % 2 == 0) key else value }  }
        }
        else -> TODO("The firebase-kotlin-sdk does not support $descriptor for serialization yet")
    }

actual fun getPolymorphicType(value: Any?, discriminator: String): String =
    (value as Map<*,*>)[discriminator] as String