/*
 * Copyright © 2018-2019 Apple Inc. and the ServiceTalk project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.servicetalk.concurrent.api;

import java.util.function.Function;

import static io.servicetalk.concurrent.api.AsyncContextMapThreadLocal.contextThreadLocal;
import static java.util.Objects.requireNonNull;

final class ContextPreservingFunction<T, U> implements Function<T, U> {
    private final AsyncContextMap saved;
    private final Function<T, U> delegate;

    ContextPreservingFunction(Function<T, U> delegate, AsyncContextMap contextMap) {
        this.saved = requireNonNull(contextMap);
        this.delegate = requireNonNull(delegate);
    }

    @Override
    public U apply(T t) {
        final Thread currentThread = Thread.currentThread();
        if (currentThread instanceof AsyncContextMapHolder) {
            final AsyncContextMapHolder asyncContextMapHolder = (AsyncContextMapHolder) currentThread;
            AsyncContextMap prev = asyncContextMapHolder.asyncContextMap();
            try {
                asyncContextMapHolder.asyncContextMap(saved);
                return delegate.apply(t);
            } finally {
                asyncContextMapHolder.asyncContextMap(prev);
            }
        } else {
            return slowPath(t);
        }
    }

    private U slowPath(T t) {
        AsyncContextMap prev = contextThreadLocal.get();
        try {
            contextThreadLocal.set(saved);
            return delegate.apply(t);
        } finally {
            contextThreadLocal.set(prev);
        }
    }
}
