/*
 * Decompiled with CFR 0.152.
 */
package io.netty.handler.codec.http;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelInboundHandlerAdapter;
import io.netty.channel.embedded.EmbeddedChannel;
import io.netty.handler.codec.CodecException;
import io.netty.handler.codec.DecoderException;
import io.netty.handler.codec.compression.Brotli;
import io.netty.handler.codec.compression.ZlibCodecFactory;
import io.netty.handler.codec.compression.ZlibDecoder;
import io.netty.handler.codec.compression.ZlibEncoder;
import io.netty.handler.codec.compression.ZlibWrapper;
import io.netty.handler.codec.http.DefaultHttpContent;
import io.netty.handler.codec.http.DefaultHttpRequest;
import io.netty.handler.codec.http.DefaultHttpResponse;
import io.netty.handler.codec.http.FullHttpRequest;
import io.netty.handler.codec.http.FullHttpResponse;
import io.netty.handler.codec.http.HttpContent;
import io.netty.handler.codec.http.HttpContentDecoder;
import io.netty.handler.codec.http.HttpContentDecompressor;
import io.netty.handler.codec.http.HttpHeaderNames;
import io.netty.handler.codec.http.HttpHeaderValues;
import io.netty.handler.codec.http.HttpMessage;
import io.netty.handler.codec.http.HttpMethod;
import io.netty.handler.codec.http.HttpObjectAggregator;
import io.netty.handler.codec.http.HttpRequest;
import io.netty.handler.codec.http.HttpRequestDecoder;
import io.netty.handler.codec.http.HttpResponse;
import io.netty.handler.codec.http.HttpResponseDecoder;
import io.netty.handler.codec.http.HttpStatusClass;
import io.netty.handler.codec.http.HttpVersion;
import io.netty.handler.codec.http.LastHttpContent;
import io.netty.util.CharsetUtil;
import io.netty.util.ReferenceCountUtil;
import io.netty.util.internal.PlatformDependent;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Queue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.condition.DisabledIf;

public class HttpContentDecoderTest {
    private static final String HELLO_WORLD = "hello, world";
    private static final byte[] GZ_HELLO_WORLD = new byte[]{31, -117, 8, 8, 12, 3, -74, 84, 0, 3, 50, 0, -53, 72, -51, -55, -55, -41, 81, 40, -49, 47, -54, 73, 1, 0, 58, 114, -85, -1, 12, 0, 0, 0};
    private static final byte[] SNAPPY_HELLO_WORLD = new byte[]{-1, 6, 0, 0, 115, 78, 97, 80, 112, 89, 1, 16, 0, 0, 11, -66, -63, -22, 104, 101, 108, 108, 111, 44, 32, 119, 111, 114, 108, 100};
    private static final String SAMPLE_STRING = "Hello, I am Meow!. A small kitten. :)I sleep all day, and meow all night.";
    private static final byte[] SAMPLE_BZ_BYTES = new byte[]{27, 72, 0, 0, -60, -102, 91, -86, 103, 20, -28, -23, 54, -101, 11, -106, -16, -32, -95, -61, -37, 94, -16, 97, -40, -93, -56, 18, 21, 86, -110, 82, -41, 102, -89, 20, 11, 10, -68, -31, 96, -116, -55, -80, -31, -91, 96, -64, 83, 51, -39, 13, -21, 92, -16, -119, 124, -31, 18, 78, -1, 91, 82, 105, -116, -95, -22, -11, -70, -45, 0};
    private static final byte[] SAMPLE_ZSTD_BYTES = new byte[]{40, -75, 47, -3, 32, 73, 45, 2, 0, 2, -124, 14, 22, -112, -75, 109, 11, -112, 113, 101, -65, 53, 59, -25, -51, -51, -78, 40, -120, -69, -32, 110, -7, -81, 28, -90, -112, 3, -23, -38, -89, -113, -98, 75, -32, -114, 1, 63, 38, 27, 97, -80, -23, -107, 66, -119, 17, 47, -109, 9, 91, -80, -10, -122, -13, 108, 92, 22, -15, 69, 2, 0, 53, -27, 55, -125, 89, 6};

    @Test
    public void testBinaryDecompression() throws Exception {
        byte[] helloWorld = HttpContentDecoderTest.gzDecompress(GZ_HELLO_WORLD);
        Assertions.assertEquals((int)HELLO_WORLD.length(), (int)helloWorld.length);
        Assertions.assertEquals((Object)HELLO_WORLD, (Object)new String(helloWorld, CharsetUtil.US_ASCII));
        String fullCycleTest = "full cycle test";
        byte[] compressed = HttpContentDecoderTest.gzCompress(fullCycleTest.getBytes(CharsetUtil.US_ASCII));
        byte[] decompressed = HttpContentDecoderTest.gzDecompress(compressed);
        Assertions.assertEquals((int)decompressed.length, (int)fullCycleTest.length());
        Assertions.assertEquals((Object)fullCycleTest, (Object)new String(decompressed, CharsetUtil.US_ASCII));
    }

    @Test
    public void testRequestDecompression() {
        HttpRequestDecoder decoder = new HttpRequestDecoder();
        HttpContentDecompressor decompressor = new HttpContentDecompressor(0);
        HttpObjectAggregator aggregator = new HttpObjectAggregator(1024);
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{decoder, decompressor, aggregator});
        String headers = "POST / HTTP/1.1\r\nContent-Length: " + GZ_HELLO_WORLD.length + "\r\nContent-Encoding: gzip\r\n\r\n";
        ByteBuf buf = Unpooled.copiedBuffer((byte[][])new byte[][]{headers.getBytes(CharsetUtil.US_ASCII), GZ_HELLO_WORLD});
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{buf}));
        Object o = channel.readInbound();
        Assertions.assertInstanceOf(FullHttpRequest.class, (Object)o);
        FullHttpRequest req = (FullHttpRequest)o;
        Assertions.assertEquals((int)HELLO_WORLD.length(), (int)req.headers().getInt((CharSequence)HttpHeaderNames.CONTENT_LENGTH));
        Assertions.assertEquals((Object)HELLO_WORLD, (Object)req.content().toString(CharsetUtil.US_ASCII));
        req.release();
        HttpContentDecoderTest.assertHasInboundMessages(channel, false);
        HttpContentDecoderTest.assertHasOutboundMessages(channel, false);
        Assertions.assertFalse((boolean)channel.finish());
    }

    @Test
    public void testChunkedRequestDecompression() {
        HttpResponseDecoder decoder = new HttpResponseDecoder();
        HttpContentDecompressor decompressor = new HttpContentDecompressor(0);
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{decoder, decompressor, null});
        String headers = "HTTP/1.1 200 OK\r\nTransfer-Encoding: chunked\r\nTrailer: My-Trailer\r\nContent-Encoding: gzip\r\n\r\n";
        channel.writeInbound(new Object[]{Unpooled.copiedBuffer((byte[])headers.getBytes(CharsetUtil.US_ASCII))});
        String chunkLength = Integer.toHexString(GZ_HELLO_WORLD.length);
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{Unpooled.copiedBuffer((CharSequence)(chunkLength + "\r\n"), (Charset)CharsetUtil.US_ASCII)}));
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{Unpooled.copiedBuffer((byte[])GZ_HELLO_WORLD)}));
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{Unpooled.copiedBuffer((byte[])"\r\n".getBytes(CharsetUtil.US_ASCII))}));
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{Unpooled.copiedBuffer((CharSequence)"0\r\n", (Charset)CharsetUtil.US_ASCII)}));
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{Unpooled.copiedBuffer((CharSequence)"My-Trailer: 42\r\n\r\n\r\n", (Charset)CharsetUtil.US_ASCII)}));
        Object ob1 = channel.readInbound();
        Assertions.assertInstanceOf(DefaultHttpResponse.class, (Object)ob1);
        Object ob2 = channel.readInbound();
        Assertions.assertInstanceOf(HttpContent.class, (Object)ob2);
        HttpContent content = (HttpContent)ob2;
        Assertions.assertEquals((Object)HELLO_WORLD, (Object)content.content().toString(CharsetUtil.US_ASCII));
        content.release();
        Object ob3 = channel.readInbound();
        Assertions.assertInstanceOf(LastHttpContent.class, (Object)ob3);
        LastHttpContent lastContent = (LastHttpContent)ob3;
        Assertions.assertNotNull((Object)lastContent.decoderResult());
        Assertions.assertTrue((boolean)lastContent.decoderResult().isSuccess());
        Assertions.assertFalse((boolean)lastContent.trailingHeaders().isEmpty());
        Assertions.assertEquals((Object)"42", (Object)lastContent.trailingHeaders().get("My-Trailer"));
        HttpContentDecoderTest.assertHasInboundMessages(channel, false);
        HttpContentDecoderTest.assertHasOutboundMessages(channel, false);
        Assertions.assertFalse((boolean)channel.finish());
    }

    @Test
    public void testSnappyResponseDecompression() {
        HttpResponseDecoder decoder = new HttpResponseDecoder();
        HttpContentDecompressor decompressor = new HttpContentDecompressor(0);
        HttpObjectAggregator aggregator = new HttpObjectAggregator(1024);
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{decoder, decompressor, aggregator});
        String headers = "HTTP/1.1 200 OK\r\nContent-Length: " + SNAPPY_HELLO_WORLD.length + "\r\nContent-Encoding: snappy\r\n\r\n";
        ByteBuf buf = Unpooled.copiedBuffer((byte[][])new byte[][]{headers.getBytes(CharsetUtil.UTF_8), SNAPPY_HELLO_WORLD});
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{buf}));
        Object o = channel.readInbound();
        Assertions.assertInstanceOf(FullHttpResponse.class, (Object)o);
        FullHttpResponse resp = (FullHttpResponse)o;
        Assertions.assertEquals((int)HELLO_WORLD.length(), (int)resp.headers().getInt((CharSequence)HttpHeaderNames.CONTENT_LENGTH));
        Assertions.assertEquals((Object)HELLO_WORLD, (Object)resp.content().toString(CharsetUtil.UTF_8));
        resp.release();
        HttpContentDecoderTest.assertHasInboundMessages(channel, false);
        HttpContentDecoderTest.assertHasOutboundMessages(channel, false);
        Assertions.assertFalse((boolean)channel.finish());
    }

    @Test
    public void testResponseDecompression() {
        HttpResponseDecoder decoder = new HttpResponseDecoder();
        HttpContentDecompressor decompressor = new HttpContentDecompressor(0);
        HttpObjectAggregator aggregator = new HttpObjectAggregator(1024);
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{decoder, decompressor, aggregator});
        String headers = "HTTP/1.1 200 OK\r\nContent-Length: " + GZ_HELLO_WORLD.length + "\r\nContent-Encoding: gzip\r\n\r\n";
        ByteBuf buf = Unpooled.copiedBuffer((byte[][])new byte[][]{headers.getBytes(CharsetUtil.US_ASCII), GZ_HELLO_WORLD});
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{buf}));
        Object o = channel.readInbound();
        Assertions.assertInstanceOf(FullHttpResponse.class, (Object)o);
        FullHttpResponse resp = (FullHttpResponse)o;
        Assertions.assertEquals((int)HELLO_WORLD.length(), (int)resp.headers().getInt((CharSequence)HttpHeaderNames.CONTENT_LENGTH));
        Assertions.assertEquals((Object)HELLO_WORLD, (Object)resp.content().toString(CharsetUtil.US_ASCII));
        resp.release();
        HttpContentDecoderTest.assertHasInboundMessages(channel, false);
        HttpContentDecoderTest.assertHasOutboundMessages(channel, false);
        Assertions.assertFalse((boolean)channel.finish());
    }

    @DisabledIf(value="isNotSupported", disabledReason="Brotli is not supported on this platform")
    @Test
    public void testResponseBrotliDecompression() throws Throwable {
        Brotli.ensureAvailability();
        HttpResponseDecoder decoder = new HttpResponseDecoder();
        HttpContentDecompressor decompressor = new HttpContentDecompressor(0);
        HttpObjectAggregator aggregator = new HttpObjectAggregator(Integer.MAX_VALUE);
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{decoder, decompressor, aggregator});
        String headers = "HTTP/1.1 200 OK\r\nContent-Length: " + SAMPLE_BZ_BYTES.length + "\r\nContent-Encoding: br\r\n\r\n";
        ByteBuf buf = Unpooled.wrappedBuffer((byte[][])new byte[][]{headers.getBytes(CharsetUtil.US_ASCII), SAMPLE_BZ_BYTES});
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{buf}));
        Object o = channel.readInbound();
        Assertions.assertInstanceOf(FullHttpResponse.class, (Object)o);
        FullHttpResponse resp = (FullHttpResponse)o;
        Assertions.assertNull((Object)resp.headers().get((CharSequence)HttpHeaderNames.CONTENT_ENCODING), (String)"Content-Encoding header should be removed");
        Assertions.assertEquals((Object)SAMPLE_STRING, (Object)resp.content().toString(CharsetUtil.UTF_8), (String)"Response body should match uncompressed string");
        resp.release();
        HttpContentDecoderTest.assertHasInboundMessages(channel, false);
        HttpContentDecoderTest.assertHasOutboundMessages(channel, false);
        Assertions.assertFalse((boolean)channel.finish());
    }

    @DisabledIf(value="isNotSupported", disabledReason="Brotli is not supported on this platform")
    @Test
    public void testResponseChunksBrotliDecompression() throws Throwable {
        Brotli.ensureAvailability();
        HttpResponseDecoder decoder = new HttpResponseDecoder();
        HttpContentDecompressor decompressor = new HttpContentDecompressor(0);
        HttpObjectAggregator aggregator = new HttpObjectAggregator(Integer.MAX_VALUE);
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{decoder, decompressor, aggregator});
        String headers = "HTTP/1.1 200 OK\r\nContent-Length: " + SAMPLE_BZ_BYTES.length + "\r\nContent-Encoding: br\r\n\r\n";
        Assertions.assertFalse((boolean)channel.writeInbound(new Object[]{Unpooled.wrappedBuffer((byte[])headers.getBytes(CharsetUtil.US_ASCII))}));
        int offset = 0;
        while (offset < SAMPLE_BZ_BYTES.length) {
            int len = Math.min(1500, SAMPLE_BZ_BYTES.length - offset);
            boolean available = channel.writeInbound(new Object[]{Unpooled.wrappedBuffer((byte[])SAMPLE_BZ_BYTES, (int)offset, (int)len)});
            if ((offset += 1500) < SAMPLE_BZ_BYTES.length) {
                Assertions.assertFalse((boolean)available);
                continue;
            }
            Assertions.assertTrue((boolean)available);
        }
        Object o = channel.readInbound();
        Assertions.assertInstanceOf(FullHttpResponse.class, (Object)o);
        FullHttpResponse resp = (FullHttpResponse)o;
        Assertions.assertEquals((Object)SAMPLE_STRING, (Object)resp.content().toString(CharsetUtil.UTF_8), (String)"Response body should match uncompressed string");
        resp.release();
        HttpContentDecoderTest.assertHasInboundMessages(channel, false);
        HttpContentDecoderTest.assertHasOutboundMessages(channel, false);
        Assertions.assertFalse((boolean)channel.finish());
    }

    @Test
    public void testResponseZstdDecompression() throws Throwable {
        HttpResponseDecoder decoder = new HttpResponseDecoder();
        HttpContentDecompressor decompressor = new HttpContentDecompressor(0);
        HttpObjectAggregator aggregator = new HttpObjectAggregator(Integer.MAX_VALUE);
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{decoder, decompressor, aggregator});
        String headers = "HTTP/1.1 200 OK\r\nContent-Length: " + SAMPLE_ZSTD_BYTES.length + "\r\nContent-Encoding: zstd\r\n\r\n";
        ByteBuf buf = Unpooled.wrappedBuffer((byte[][])new byte[][]{headers.getBytes(CharsetUtil.US_ASCII), SAMPLE_ZSTD_BYTES});
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{buf}));
        Object o = channel.readInbound();
        Assertions.assertInstanceOf(FullHttpResponse.class, (Object)o);
        FullHttpResponse resp = (FullHttpResponse)o;
        Assertions.assertNull((Object)resp.headers().get((CharSequence)HttpHeaderNames.CONTENT_ENCODING), (String)"Content-Encoding header should be removed");
        Assertions.assertEquals((Object)SAMPLE_STRING, (Object)resp.content().toString(CharsetUtil.UTF_8), (String)"Response body should match uncompressed string");
        resp.release();
        HttpContentDecoderTest.assertHasInboundMessages(channel, false);
        HttpContentDecoderTest.assertHasOutboundMessages(channel, false);
        Assertions.assertFalse((boolean)channel.finish());
    }

    @Test
    public void testResponseChunksZstdDecompression() throws Throwable {
        HttpResponseDecoder decoder = new HttpResponseDecoder();
        HttpContentDecompressor decompressor = new HttpContentDecompressor(0);
        HttpObjectAggregator aggregator = new HttpObjectAggregator(Integer.MAX_VALUE);
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{decoder, decompressor, aggregator});
        String headers = "HTTP/1.1 200 OK\r\nContent-Length: " + SAMPLE_ZSTD_BYTES.length + "\r\nContent-Encoding: zstd\r\n\r\n";
        Assertions.assertFalse((boolean)channel.writeInbound(new Object[]{Unpooled.wrappedBuffer((byte[])headers.getBytes(CharsetUtil.US_ASCII))}));
        int offset = 0;
        while (offset < SAMPLE_ZSTD_BYTES.length) {
            int len = Math.min(1500, SAMPLE_ZSTD_BYTES.length - offset);
            boolean available = channel.writeInbound(new Object[]{Unpooled.wrappedBuffer((byte[])SAMPLE_ZSTD_BYTES, (int)offset, (int)len)});
            if ((offset += 1500) < SAMPLE_ZSTD_BYTES.length) {
                Assertions.assertFalse((boolean)available);
                continue;
            }
            Assertions.assertTrue((boolean)available);
        }
        Object o = channel.readInbound();
        Assertions.assertInstanceOf(FullHttpResponse.class, (Object)o);
        FullHttpResponse resp = (FullHttpResponse)o;
        Assertions.assertEquals((Object)SAMPLE_STRING, (Object)resp.content().toString(CharsetUtil.UTF_8), (String)"Response body should match uncompressed string");
        resp.release();
        HttpContentDecoderTest.assertHasInboundMessages(channel, false);
        HttpContentDecoderTest.assertHasOutboundMessages(channel, false);
        Assertions.assertFalse((boolean)channel.finish());
    }

    @Test
    public void testExpectContinueResponse1() {
        HttpRequestDecoder decoder = new HttpRequestDecoder();
        HttpObjectAggregator aggregator = new HttpObjectAggregator(1024);
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{decoder, aggregator});
        String req = "POST / HTTP/1.1\r\nContent-Length: " + GZ_HELLO_WORLD.length + "\r\nExpect: 100-continue\r\n\r\n";
        Assertions.assertFalse((boolean)channel.writeInbound(new Object[]{Unpooled.wrappedBuffer((byte[])req.getBytes())}));
        Object o = channel.readOutbound();
        Assertions.assertInstanceOf(FullHttpResponse.class, (Object)o);
        FullHttpResponse r = (FullHttpResponse)o;
        Assertions.assertEquals((int)100, (int)r.status().code());
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{Unpooled.copiedBuffer((byte[])GZ_HELLO_WORLD)}));
        r.release();
        HttpContentDecoderTest.assertHasInboundMessages(channel, true);
        HttpContentDecoderTest.assertHasOutboundMessages(channel, false);
        Assertions.assertFalse((boolean)channel.finish());
    }

    @Test
    public void testExpectContinueResponse2() {
        HttpRequestDecoder decoder = new HttpRequestDecoder();
        HttpContentDecompressor decompressor = new HttpContentDecompressor(0);
        HttpObjectAggregator aggregator = new HttpObjectAggregator(1024);
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{decoder, decompressor, aggregator});
        String req = "POST / HTTP/1.1\r\nContent-Length: " + GZ_HELLO_WORLD.length + "\r\nExpect: 100-continue\r\n\r\n";
        Assertions.assertFalse((boolean)channel.writeInbound(new Object[]{Unpooled.wrappedBuffer((byte[])req.getBytes())}));
        Object o = channel.readOutbound();
        Assertions.assertInstanceOf(FullHttpResponse.class, (Object)o);
        FullHttpResponse r = (FullHttpResponse)o;
        Assertions.assertEquals((int)100, (int)r.status().code());
        r.release();
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{Unpooled.copiedBuffer((byte[])GZ_HELLO_WORLD)}));
        HttpContentDecoderTest.assertHasInboundMessages(channel, true);
        HttpContentDecoderTest.assertHasOutboundMessages(channel, false);
        Assertions.assertFalse((boolean)channel.finish());
    }

    @Test
    public void testExpectContinueResponse3() {
        HttpRequestDecoder decoder = new HttpRequestDecoder();
        HttpContentDecompressor decompressor = new HttpContentDecompressor(0);
        HttpObjectAggregator aggregator = new HttpObjectAggregator(1024);
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{decoder, decompressor, aggregator});
        String req = "POST / HTTP/1.1\r\nContent-Length: " + GZ_HELLO_WORLD.length + "\r\nExpect: 100-continue\r\nContent-Encoding: gzip\r\n\r\n";
        Assertions.assertFalse((boolean)channel.writeInbound(new Object[]{Unpooled.wrappedBuffer((byte[])req.getBytes())}));
        Object o = channel.readOutbound();
        Assertions.assertInstanceOf(FullHttpResponse.class, (Object)o);
        FullHttpResponse r = (FullHttpResponse)o;
        Assertions.assertEquals((int)100, (int)r.status().code());
        r.release();
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{Unpooled.copiedBuffer((byte[])GZ_HELLO_WORLD)}));
        HttpContentDecoderTest.assertHasInboundMessages(channel, true);
        HttpContentDecoderTest.assertHasOutboundMessages(channel, false);
        Assertions.assertFalse((boolean)channel.finish());
    }

    @Test
    public void testExpectContinueResponse4() {
        HttpRequestDecoder decoder = new HttpRequestDecoder();
        HttpObjectAggregator aggregator = new HttpObjectAggregator(1024);
        HttpContentDecompressor decompressor = new HttpContentDecompressor(0);
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{decoder, aggregator, decompressor});
        String req = "POST / HTTP/1.1\r\nContent-Length: " + GZ_HELLO_WORLD.length + "\r\nExpect: 100-continue\r\nContent-Encoding: gzip\r\n\r\n";
        Assertions.assertFalse((boolean)channel.writeInbound(new Object[]{Unpooled.wrappedBuffer((byte[])req.getBytes())}));
        Object o = channel.readOutbound();
        Assertions.assertInstanceOf(FullHttpResponse.class, (Object)o);
        FullHttpResponse r = (FullHttpResponse)o;
        Assertions.assertEquals((int)100, (int)r.status().code());
        r.release();
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{Unpooled.copiedBuffer((byte[])GZ_HELLO_WORLD)}));
        HttpContentDecoderTest.assertHasInboundMessages(channel, true);
        HttpContentDecoderTest.assertHasOutboundMessages(channel, false);
        Assertions.assertFalse((boolean)channel.finish());
    }

    @Test
    public void testExpectContinueResetHttpObjectDecoder() {
        HttpRequestDecoder decoder = new HttpRequestDecoder();
        int maxBytes = 10;
        HttpObjectAggregator aggregator = new HttpObjectAggregator(10);
        final AtomicReference secondRequestRef = new AtomicReference();
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{decoder, aggregator, new ChannelInboundHandlerAdapter(){

            public void channelRead(ChannelHandlerContext ctx, Object msg) throws Exception {
                if (msg instanceof FullHttpRequest) {
                    if (!secondRequestRef.compareAndSet(null, (FullHttpRequest)msg)) {
                        ((FullHttpRequest)msg).release();
                    }
                } else {
                    ReferenceCountUtil.release((Object)msg);
                }
            }
        }});
        String req1 = "POST /1 HTTP/1.1\r\nContent-Length: 11\r\nExpect: 100-continue\r\n\r\n";
        Assertions.assertFalse((boolean)channel.writeInbound(new Object[]{Unpooled.wrappedBuffer((byte[])req1.getBytes(CharsetUtil.US_ASCII))}));
        FullHttpResponse resp = (FullHttpResponse)channel.readOutbound();
        Assertions.assertEquals((Object)HttpStatusClass.CLIENT_ERROR, (Object)resp.status().codeClass());
        resp.release();
        String req2 = "POST /2 HTTP/1.1\r\nContent-Length: 10\r\nExpect: 100-continue\r\n\r\n";
        Assertions.assertFalse((boolean)channel.writeInbound(new Object[]{Unpooled.wrappedBuffer((byte[])req2.getBytes(CharsetUtil.US_ASCII))}));
        resp = (FullHttpResponse)channel.readOutbound();
        Assertions.assertEquals((int)100, (int)resp.status().code());
        resp.release();
        byte[] content = new byte[10];
        Assertions.assertFalse((boolean)channel.writeInbound(new Object[]{Unpooled.wrappedBuffer((byte[])content)}));
        FullHttpRequest req = (FullHttpRequest)secondRequestRef.get();
        Assertions.assertNotNull((Object)req);
        Assertions.assertEquals((Object)"/2", (Object)req.uri());
        Assertions.assertEquals((int)10, (int)req.content().readableBytes());
        req.release();
        HttpContentDecoderTest.assertHasInboundMessages(channel, false);
        HttpContentDecoderTest.assertHasOutboundMessages(channel, false);
        Assertions.assertFalse((boolean)channel.finish());
    }

    @Test
    public void testRequestContentLength1() {
        HttpRequestDecoder decoder = new HttpRequestDecoder(4096, 4096, 5);
        HttpContentDecompressor decompressor = new HttpContentDecompressor(0);
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{decoder, decompressor});
        String headers = "POST / HTTP/1.1\r\nContent-Length: " + GZ_HELLO_WORLD.length + "\r\nContent-Encoding: gzip\r\n\r\n";
        ByteBuf buf = Unpooled.copiedBuffer((byte[][])new byte[][]{headers.getBytes(CharsetUtil.US_ASCII), GZ_HELLO_WORLD});
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{buf}));
        Queue req = channel.inboundMessages();
        Assertions.assertTrue((req.size() >= 1 ? 1 : 0) != 0);
        Object o = req.peek();
        Assertions.assertInstanceOf(HttpRequest.class, o);
        HttpRequest r = (HttpRequest)o;
        String v = r.headers().get((CharSequence)HttpHeaderNames.CONTENT_LENGTH);
        Long value = v == null ? null : Long.valueOf(Long.parseLong(v));
        Assertions.assertTrue((value == null || value == (long)HELLO_WORLD.length() ? 1 : 0) != 0);
        HttpContentDecoderTest.assertHasInboundMessages(channel, true);
        HttpContentDecoderTest.assertHasOutboundMessages(channel, false);
        Assertions.assertFalse((boolean)channel.finish());
    }

    @Test
    public void testRequestContentLength2() {
        HttpRequestDecoder decoder = new HttpRequestDecoder(4096, 4096, 5);
        HttpContentDecompressor decompressor = new HttpContentDecompressor(0);
        HttpObjectAggregator aggregator = new HttpObjectAggregator(1024);
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{decoder, decompressor, aggregator});
        String headers = "POST / HTTP/1.1\r\nContent-Length: " + GZ_HELLO_WORLD.length + "\r\nContent-Encoding: gzip\r\n\r\n";
        ByteBuf buf = Unpooled.copiedBuffer((byte[][])new byte[][]{headers.getBytes(CharsetUtil.US_ASCII), GZ_HELLO_WORLD});
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{buf}));
        Object o = channel.readInbound();
        Assertions.assertInstanceOf(FullHttpRequest.class, (Object)o);
        FullHttpRequest r = (FullHttpRequest)o;
        String v = r.headers().get((CharSequence)HttpHeaderNames.CONTENT_LENGTH);
        Long value = v == null ? null : Long.valueOf(Long.parseLong(v));
        r.release();
        Assertions.assertNotNull((Object)value);
        Assertions.assertEquals((long)HELLO_WORLD.length(), (long)value);
        HttpContentDecoderTest.assertHasInboundMessages(channel, false);
        HttpContentDecoderTest.assertHasOutboundMessages(channel, false);
        Assertions.assertFalse((boolean)channel.finish());
    }

    @Test
    public void testResponseContentLength1() {
        HttpResponseDecoder decoder = new HttpResponseDecoder(4096, 4096, 5);
        HttpContentDecompressor decompressor = new HttpContentDecompressor(0);
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{decoder, decompressor});
        String headers = "HTTP/1.1 200 OK\r\nContent-Length: " + GZ_HELLO_WORLD.length + "\r\nContent-Encoding: gzip\r\n\r\n";
        ByteBuf buf = Unpooled.copiedBuffer((byte[][])new byte[][]{headers.getBytes(CharsetUtil.US_ASCII), GZ_HELLO_WORLD});
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{buf}));
        Queue resp = channel.inboundMessages();
        Assertions.assertTrue((resp.size() >= 1 ? 1 : 0) != 0);
        Object o = resp.peek();
        Assertions.assertInstanceOf(HttpResponse.class, o);
        HttpResponse r = (HttpResponse)o;
        Assertions.assertFalse((boolean)r.headers().contains((CharSequence)HttpHeaderNames.CONTENT_LENGTH), (String)"Content-Length header not removed.");
        String transferEncoding = r.headers().get((CharSequence)HttpHeaderNames.TRANSFER_ENCODING);
        Assertions.assertNotNull((Object)transferEncoding, (String)"Content-length as well as transfer-encoding not set.");
        Assertions.assertEquals((Object)HttpHeaderValues.CHUNKED.toString(), (Object)transferEncoding, (String)"Unexpected transfer-encoding value.");
        HttpContentDecoderTest.assertHasInboundMessages(channel, true);
        HttpContentDecoderTest.assertHasOutboundMessages(channel, false);
        Assertions.assertFalse((boolean)channel.finish());
    }

    @Test
    public void testResponseContentLength2() {
        HttpResponseDecoder decoder = new HttpResponseDecoder(4096, 4096, 5);
        HttpContentDecompressor decompressor = new HttpContentDecompressor(0);
        HttpObjectAggregator aggregator = new HttpObjectAggregator(1024);
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{decoder, decompressor, aggregator});
        String headers = "HTTP/1.1 200 OK\r\nContent-Length: " + GZ_HELLO_WORLD.length + "\r\nContent-Encoding: gzip\r\n\r\n";
        ByteBuf buf = Unpooled.copiedBuffer((byte[][])new byte[][]{headers.getBytes(CharsetUtil.US_ASCII), GZ_HELLO_WORLD});
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{buf}));
        Object o = channel.readInbound();
        Assertions.assertInstanceOf(FullHttpResponse.class, (Object)o);
        FullHttpResponse r = (FullHttpResponse)o;
        String v = r.headers().get((CharSequence)HttpHeaderNames.CONTENT_LENGTH);
        Long value = v == null ? null : Long.valueOf(Long.parseLong(v));
        Assertions.assertNotNull((Object)value);
        Assertions.assertEquals((long)HELLO_WORLD.length(), (long)value);
        r.release();
        HttpContentDecoderTest.assertHasInboundMessages(channel, false);
        HttpContentDecoderTest.assertHasOutboundMessages(channel, false);
        Assertions.assertFalse((boolean)channel.finish());
    }

    @Test
    public void testFullHttpRequest() {
        HttpRequestDecoder decoder = new HttpRequestDecoder(4096, 4096, 5);
        HttpObjectAggregator aggregator = new HttpObjectAggregator(1024);
        HttpContentDecompressor decompressor = new HttpContentDecompressor(0);
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{decoder, aggregator, decompressor});
        String headers = "POST / HTTP/1.1\r\nContent-Length: " + GZ_HELLO_WORLD.length + "\r\nContent-Encoding: gzip\r\n\r\n";
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{Unpooled.copiedBuffer((byte[][])new byte[][]{headers.getBytes(), GZ_HELLO_WORLD})}));
        Queue req = channel.inboundMessages();
        Assertions.assertTrue((req.size() > 1 ? 1 : 0) != 0);
        int contentLength = 0;
        contentLength = HttpContentDecoderTest.calculateContentLength(req, contentLength);
        byte[] receivedContent = HttpContentDecoderTest.readContent(req, contentLength, true);
        Assertions.assertEquals((Object)HELLO_WORLD, (Object)new String(receivedContent, CharsetUtil.US_ASCII));
        HttpContentDecoderTest.assertHasInboundMessages(channel, true);
        HttpContentDecoderTest.assertHasOutboundMessages(channel, false);
        Assertions.assertFalse((boolean)channel.finish());
    }

    @Test
    public void testFullHttpResponse() {
        HttpResponseDecoder decoder = new HttpResponseDecoder(4096, 4096, 5);
        HttpObjectAggregator aggregator = new HttpObjectAggregator(1024);
        HttpContentDecompressor decompressor = new HttpContentDecompressor(0);
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{decoder, aggregator, decompressor});
        String headers = "HTTP/1.1 200 OK\r\nContent-Length: " + GZ_HELLO_WORLD.length + "\r\nContent-Encoding: gzip\r\n\r\n";
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{Unpooled.copiedBuffer((byte[][])new byte[][]{headers.getBytes(), GZ_HELLO_WORLD})}));
        Queue resp = channel.inboundMessages();
        Assertions.assertTrue((resp.size() > 1 ? 1 : 0) != 0);
        int contentLength = 0;
        contentLength = HttpContentDecoderTest.calculateContentLength(resp, contentLength);
        byte[] receivedContent = HttpContentDecoderTest.readContent(resp, contentLength, true);
        Assertions.assertEquals((Object)HELLO_WORLD, (Object)new String(receivedContent, CharsetUtil.US_ASCII));
        HttpContentDecoderTest.assertHasInboundMessages(channel, true);
        HttpContentDecoderTest.assertHasOutboundMessages(channel, false);
        Assertions.assertFalse((boolean)channel.finish());
    }

    @Test
    public void testFullHttpResponseEOF() {
        HttpResponseDecoder decoder = new HttpResponseDecoder(4096, 4096, 5);
        HttpContentDecompressor decompressor = new HttpContentDecompressor(0);
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{decoder, decompressor});
        String headers = "HTTP/1.1 200 OK\r\nContent-Encoding: gzip\r\n\r\n";
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{Unpooled.copiedBuffer((byte[][])new byte[][]{headers.getBytes(), GZ_HELLO_WORLD})}));
        Assertions.assertTrue((boolean)channel.finish());
        Queue resp = channel.inboundMessages();
        Assertions.assertTrue((resp.size() > 1 ? 1 : 0) != 0);
        int contentLength = 0;
        contentLength = HttpContentDecoderTest.calculateContentLength(resp, contentLength);
        byte[] receivedContent = HttpContentDecoderTest.readContent(resp, contentLength, false);
        Assertions.assertEquals((Object)HELLO_WORLD, (Object)new String(receivedContent, CharsetUtil.US_ASCII));
        HttpContentDecoderTest.assertHasInboundMessages(channel, true);
        HttpContentDecoderTest.assertHasOutboundMessages(channel, false);
        Assertions.assertFalse((boolean)channel.finish());
    }

    @Test
    public void testCleanupThrows() {
        HttpContentDecoder decoder = new HttpContentDecoder(){

            protected EmbeddedChannel newContentDecoder(String contentEncoding) throws Exception {
                return new EmbeddedChannel(new ChannelHandler[]{new ChannelInboundHandlerAdapter(){

                    public void channelInactive(ChannelHandlerContext ctx) throws Exception {
                        ctx.fireExceptionCaught((Throwable)new DecoderException());
                        ctx.fireChannelInactive();
                    }
                }});
            }
        };
        final AtomicBoolean channelInactiveCalled = new AtomicBoolean();
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{decoder, new ChannelInboundHandlerAdapter(){

            public void channelInactive(ChannelHandlerContext ctx) throws Exception {
                Assertions.assertTrue((boolean)channelInactiveCalled.compareAndSet(false, true));
                super.channelInactive(ctx);
            }
        }});
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{new DefaultHttpRequest(HttpVersion.HTTP_1_1, HttpMethod.GET, "/")}));
        DefaultHttpContent content = new DefaultHttpContent(Unpooled.buffer().writeZero(10));
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{content}));
        Assertions.assertEquals((int)1, (int)content.refCnt());
        try {
            channel.finishAndReleaseAll();
            Assertions.fail();
        }
        catch (CodecException codecException) {
            // empty catch block
        }
        Assertions.assertTrue((boolean)channelInactiveCalled.get());
        Assertions.assertEquals((int)0, (int)content.refCnt());
    }

    @Test
    public void testTransferCodingGZIP() {
        String requestStr = "POST / HTTP/1.1\r\nContent-Length: " + GZ_HELLO_WORLD.length + "\r\nTransfer-Encoding: gzip\r\n\r\n";
        HttpRequestDecoder decoder = new HttpRequestDecoder();
        HttpContentDecompressor decompressor = new HttpContentDecompressor(0);
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{decoder, decompressor});
        channel.writeInbound(new Object[]{Unpooled.copiedBuffer((byte[])requestStr.getBytes())});
        channel.writeInbound(new Object[]{Unpooled.copiedBuffer((byte[])GZ_HELLO_WORLD)});
        HttpRequest request = (HttpRequest)channel.readInbound();
        Assertions.assertTrue((boolean)request.decoderResult().isSuccess());
        Assertions.assertFalse((boolean)request.headers().contains((CharSequence)HttpHeaderNames.CONTENT_LENGTH));
        HttpContent content = (HttpContent)channel.readInbound();
        Assertions.assertTrue((boolean)content.decoderResult().isSuccess());
        Assertions.assertEquals((Object)HELLO_WORLD, (Object)content.content().toString(CharsetUtil.US_ASCII));
        content.release();
        LastHttpContent lastHttpContent = (LastHttpContent)channel.readInbound();
        Assertions.assertTrue((boolean)lastHttpContent.decoderResult().isSuccess());
        lastHttpContent.release();
        HttpContentDecoderTest.assertHasInboundMessages(channel, false);
        HttpContentDecoderTest.assertHasOutboundMessages(channel, false);
        Assertions.assertFalse((boolean)channel.finish());
        channel.releaseInbound();
    }

    @Test
    public void testTransferCodingGZIPAndChunked() {
        String requestStr = "POST / HTTP/1.1\r\nHost: example.com\r\nContent-Type: application/x-www-form-urlencoded\r\nTrailer: My-Trailer\r\nTransfer-Encoding: gzip, chunked\r\n\r\n";
        HttpRequestDecoder decoder = new HttpRequestDecoder();
        HttpContentDecompressor decompressor = new HttpContentDecompressor(0);
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{decoder, decompressor});
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{Unpooled.copiedBuffer((CharSequence)requestStr, (Charset)CharsetUtil.US_ASCII)}));
        String chunkLength = Integer.toHexString(GZ_HELLO_WORLD.length);
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{Unpooled.copiedBuffer((CharSequence)(chunkLength + "\r\n"), (Charset)CharsetUtil.US_ASCII)}));
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{Unpooled.copiedBuffer((byte[])GZ_HELLO_WORLD)}));
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{Unpooled.copiedBuffer((byte[])"\r\n".getBytes(CharsetUtil.US_ASCII))}));
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{Unpooled.copiedBuffer((CharSequence)"0\r\n", (Charset)CharsetUtil.US_ASCII)}));
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{Unpooled.copiedBuffer((CharSequence)"My-Trailer: 42\r\n\r\n", (Charset)CharsetUtil.US_ASCII)}));
        HttpRequest request = (HttpRequest)channel.readInbound();
        Assertions.assertTrue((boolean)request.decoderResult().isSuccess());
        Assertions.assertTrue((boolean)request.headers().containsValue((CharSequence)HttpHeaderNames.TRANSFER_ENCODING, (CharSequence)HttpHeaderValues.CHUNKED, true));
        Assertions.assertFalse((boolean)request.headers().contains((CharSequence)HttpHeaderNames.CONTENT_LENGTH));
        HttpContent chunk1 = (HttpContent)channel.readInbound();
        Assertions.assertTrue((boolean)chunk1.decoderResult().isSuccess());
        Assertions.assertEquals((Object)HELLO_WORLD, (Object)chunk1.content().toString(CharsetUtil.US_ASCII));
        chunk1.release();
        LastHttpContent chunk2 = (LastHttpContent)channel.readInbound();
        Assertions.assertTrue((boolean)chunk2.decoderResult().isSuccess());
        Assertions.assertEquals((Object)"42", (Object)chunk2.trailingHeaders().get("My-Trailer"));
        chunk2.release();
        Assertions.assertFalse((boolean)channel.finish());
        channel.releaseInbound();
    }

    private static byte[] gzDecompress(byte[] input) {
        ByteBuf o;
        ZlibDecoder decoder = ZlibCodecFactory.newZlibDecoder((ZlibWrapper)ZlibWrapper.GZIP, (int)0);
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{decoder});
        Assertions.assertTrue((boolean)channel.writeInbound(new Object[]{Unpooled.copiedBuffer((byte[])input)}));
        Assertions.assertTrue((boolean)channel.finish());
        int outputSize = 0;
        ArrayList<ByteBuf> inbound = new ArrayList<ByteBuf>();
        while ((o = (ByteBuf)channel.readInbound()) != null) {
            inbound.add(o);
            outputSize += o.readableBytes();
        }
        byte[] output = new byte[outputSize];
        int readCount = 0;
        for (ByteBuf b : inbound) {
            int readableBytes = b.readableBytes();
            b.readBytes(output, readCount, readableBytes);
            b.release();
            readCount += readableBytes;
        }
        Assertions.assertTrue((channel.inboundMessages().isEmpty() && channel.outboundMessages().isEmpty() ? 1 : 0) != 0);
        return output;
    }

    private static byte[] readContent(Queue<Object> req, int contentLength, boolean hasTransferEncoding) {
        byte[] receivedContent = new byte[contentLength];
        int readCount = 0;
        for (Object e : req) {
            if (e instanceof HttpContent) {
                ByteBuf b = ((HttpContent)e).content();
                int readableBytes = b.readableBytes();
                b.readBytes(receivedContent, readCount, readableBytes);
                readCount += readableBytes;
            }
            if (!(e instanceof HttpMessage)) continue;
            Assertions.assertEquals((Object)hasTransferEncoding, (Object)((HttpMessage)e).headers().contains((CharSequence)HttpHeaderNames.TRANSFER_ENCODING));
        }
        return receivedContent;
    }

    private static int calculateContentLength(Queue<Object> req, int contentLength) {
        for (Object e : req) {
            if (!(e instanceof HttpContent)) continue;
            Assertions.assertTrue((((HttpContent)e).refCnt() > 0 ? 1 : 0) != 0);
            ByteBuf b = ((HttpContent)e).content();
            contentLength += b.readableBytes();
        }
        return contentLength;
    }

    private static byte[] gzCompress(byte[] input) {
        ByteBuf o;
        ZlibEncoder encoder = ZlibCodecFactory.newZlibEncoder((ZlibWrapper)ZlibWrapper.GZIP);
        EmbeddedChannel channel = new EmbeddedChannel(new ChannelHandler[]{encoder});
        Assertions.assertTrue((boolean)channel.writeOutbound(new Object[]{Unpooled.wrappedBuffer((byte[])input)}));
        Assertions.assertTrue((boolean)channel.finish());
        int outputSize = 0;
        ArrayList<ByteBuf> outbound = new ArrayList<ByteBuf>();
        while ((o = (ByteBuf)channel.readOutbound()) != null) {
            outbound.add(o);
            outputSize += o.readableBytes();
        }
        byte[] output = new byte[outputSize];
        int readCount = 0;
        for (ByteBuf b : outbound) {
            int readableBytes = b.readableBytes();
            b.readBytes(output, readCount, readableBytes);
            b.release();
            readCount += readableBytes;
        }
        Assertions.assertTrue((channel.inboundMessages().isEmpty() && channel.outboundMessages().isEmpty() ? 1 : 0) != 0);
        return output;
    }

    private static void assertHasInboundMessages(EmbeddedChannel channel, boolean hasMessages) {
        if (hasMessages) {
            Object o;
            do {
                o = channel.readInbound();
                Assertions.assertNotNull((Object)o);
                ReferenceCountUtil.release((Object)o);
            } while (!(o instanceof LastHttpContent));
        } else {
            Object o = channel.readInbound();
            Assertions.assertNull((Object)o);
        }
    }

    private static void assertHasOutboundMessages(EmbeddedChannel channel, boolean hasMessages) {
        if (hasMessages) {
            Object o;
            do {
                o = channel.readOutbound();
                Assertions.assertNotNull((Object)o);
                ReferenceCountUtil.release((Object)o);
            } while (!(o instanceof LastHttpContent));
        } else {
            Object o = channel.readOutbound();
            Assertions.assertNull((Object)o);
        }
    }

    static boolean isNotSupported() {
        return PlatformDependent.isOsx() && "aarch_64".equals(PlatformDependent.normalizedArch());
    }
}

