package io.lsn.spring.issue.domain;

import io.lsn.spring.issue.configuration.IssueProperties;
import io.lsn.spring.issue.domain.entity.Comment;
import io.lsn.spring.issue.domain.entity.Issue;
import io.lsn.spring.issue.domain.entity.UserInterface;
import io.lsn.spring.issue.domain.entity.event.IssueCreatedEvent;
import io.lsn.spring.issue.domain.entity.event.IssueUpdatedEvent;
import io.lsn.spring.issue.domain.entity.event.WatcherAddedEvent;
import io.lsn.spring.issue.domain.postgres.PostgresIssueDao;
import io.lsn.spring.issue.domain.sqlserver.SqlServerIssueDao;
import org.springframework.util.CollectionUtils;

@SuppressWarnings("unchecked")
public abstract class IssueAbstractService {

    private PostgresIssueDao postgresIssueDao;
    private SqlServerIssueDao sqlServerIssueDao;
    private EventDispatcher dispatcher;
    private IssueProperties properties;

    public IssueAbstractService(PostgresIssueDao postgresIssueDao, IssueProperties properties, EventDispatcher dispatcher) {
        this.postgresIssueDao = postgresIssueDao;
        this.properties = properties;
        this.dispatcher = dispatcher;
    }

    public IssueAbstractService(SqlServerIssueDao sqlServerIssueDao, IssueProperties properties, EventDispatcher dispatcher) {
        this.sqlServerIssueDao = sqlServerIssueDao;
        this.properties = properties;
        this.dispatcher = dispatcher;
    }

    protected GenericIssueDao getDao() {
        switch (this.properties.getProvider()) {
            case SQLSERVER:
                return sqlServerIssueDao;
            case POSTGRESQL:
            default:
                return postgresIssueDao;
        }
    }

    /**
     * get one issue by id
     *
     * @param id   issue ud
     * @param user user context
     * @return Issue
     */
    public abstract <T extends Issue> T getOne(Long id, UserInterface user);


    /**
     * create new issue
     * id and uid will be returned with the issue
     *
     * @param issue issue
     * @param user  user context
     */
    public <T extends Issue> T create(T issue, UserInterface user) {
        // create issue
        getDao().create(issue, user);

        // add files
        issue.getFileHashList().forEach(f -> getDao().addIssueFile(issue, f, user));

        // add author as watcher
        getDao().addIssueWatcher(issue, issue.getAuthor().getId().longValue(), user);

        // add assignee
        issue.getAssigneeList().forEach(a -> addAssignee(issue, a.getId().longValue(), user));

        // attributes
        if (!CollectionUtils.isEmpty(issue.getAttributeList())) {
            updateAttributeList(issue, user);
        }

        T created = (T) getOne(issue.getId(), user);
        dispatcher.dispatch(new IssueCreatedEvent(created));
        return created;
    }


    /**
     * update attribute list
     *
     * @param issue issue data
     * @param user  user context
     * @param <T>   issue type
     */
    public <T extends Issue> void updateAttributeList(T issue, UserInterface user) {
        getDao().updateAttributeList(issue, user);
    }


    /**
     * update issue data
     *
     * @param issue issue data
     * @param user  user context
     * @return Issue
     */
    public <T extends Issue> T updateIssue(T issue, UserInterface user) {
        getDao().updateIssue(issue, user);
        dispatcher.dispatch(new IssueUpdatedEvent(issue));
        return (T) getOne(issue.getId(), user);
    }


    /**
     * add watcher to issue
     *
     * @param issue  issue
     * @param userId new watcher id
     * @param user   user context
     * @return Issue
     */
    public <T extends Issue> T addIssueWatcher(T issue, Long userId, UserInterface user) {
        getDao().addIssueWatcher(issue, userId, user);
        dispatcher.dispatch(new WatcherAddedEvent(issue, userId));
        return (T) getOne(issue.getId(), user);
    }


    /**
     * remove watcher from issue
     *
     * @param issue  issue
     * @param userId watcher id
     * @param user   user context
     */
    public <T extends Issue> T removeIssueWatcher(T issue, Long userId, UserInterface user) {
        getDao().removeIssueWatcher(issue, userId, user);
        return getOne(issue.getId(), user);
    }


    /**
     * add file to issue
     *
     * @param issue    issue
     * @param fileHash file hash
     */
    public <T extends Issue> T addIssueFile(T issue, String fileHash, UserInterface user) {
        issue.getFileHashList().add(fileHash);
        getDao().addIssueFile(issue, fileHash, user);
        return getOne(issue.getId(), user);
    }


    /**
     * remove file from issue
     *
     * @param issue    issue
     * @param fileHash file hash
     * @param user     user context
     */
    public <T extends Issue> T removeIssueFile(T issue, String fileHash, UserInterface user) {
        getDao().removeIssueFile(issue, fileHash, user);
        return getOne(issue.getId(), user);
    }


    /**
     * close issue
     *
     * @param issue issue
     * @param user  user context
     */
    public <T extends Issue> T closeIssue(T issue, UserInterface user) {
        getDao().updateStatus(issue, properties.getStatus().getClosed(), user);
        return getOne(issue.getId(), user);
    }


    /**
     * reopen issue
     *
     * @param issue issue
     * @param user  user context
     */
    public <T extends Issue> T openIssue(T issue, UserInterface user) {
        getDao().updateStatus(issue, properties.getStatus().getOpen(), user);
        return getOne(issue.getId(), user);
    }


    /**
     * add comment to issue
     *
     * @param issue   issue
     * @param comment comment
     * @param user    user context
     */
    public <T extends Issue> T addComment(T issue, Comment comment, UserInterface user) {
        // set issue id
        comment.setIssueId(issue.getId());

        // add comment
        getDao().addComment(comment, user);

        // add author of the comment as watcher
        addIssueWatcher(issue, comment.getAuthor().getId().longValue(), user);

        // add files
        comment.getFileHashList().forEach(f -> getDao().addCommentFile(comment, f, user));

        return getOne(issue.getId(), user);
    }


    /**
     * remove comment
     *
     * @param comment comment
     * @param user    user context
     */
    public <T extends Issue> T removeComment(Comment comment, UserInterface user) {
        // remove files
        getDao().removeCommentFile(comment, user);

        // remove comment
        getDao().removeComment(comment, user);

        return getOne(comment.getIssueId(), user);
    }


    /**
     * update comment
     *
     * @param comment comment
     * @param user    user context
     */
    public <T extends Issue> T updateComment(Comment comment, UserInterface user) {
        // remove old files
        getDao().removeCommentFile(comment, user);

        // add new files
        comment.getFileHashList().forEach(f -> getDao().addCommentFile(comment, f, user));

        // update comment itself
        getDao().updateComment(comment, user);

        return getOne(comment.getIssueId(), user);
    }


    /**
     * add user to assignee list
     *
     * @param issue      issue
     * @param assigneeId assigneeId
     * @param user       user context
     * @param <T>        Issue
     * @return Issue
     */
    public <T extends Issue> T addAssignee(T issue, Long assigneeId, UserInterface user) {
        getDao().addAssignee(issue, assigneeId, user);
        return getOne(issue.getId(), user);
    }


    /**
     * set assignee (remove all others)
     *
     * @param issue      issue
     * @param assigneeId assigneeId
     * @param user       user context
     * @param <T>        Issue
     * @return Issue
     */
    public <T extends Issue> T setAssignee(T issue, Long assigneeId, UserInterface user) {
        getDao().setAssignee(issue, assigneeId, user);
        return getOne(issue.getId(), user);
    }


    /**
     * clear assignee list for the issue
     *
     * @param issue issue
     * @param user  user context
     * @param <T>   Issue
     * @return Issue
     */
    public <T extends Issue> T clearAssignee(T issue, UserInterface user) {
        getDao().clearAssignee(issue, user);
        return getOne(issue.getId(), user);
    }


    /**
     * remove user from assignee list
     *
     * @param issue      issue
     * @param assigneeId assigneeId
     * @param user       user context
     * @param <T>        Issue
     * @return Issue
     */
    public <T extends Issue> T removeAssignee(T issue, Long assigneeId, UserInterface user) {
        getDao().removeAssignee(issue, assigneeId, user);
        return getOne(issue.getId(), user);
    }


    /**
     * close all with referenceId and type
     *
     * @param referenceId   object reference id
     * @param referenceType object reference type
     * @param user          user context
     */
    public void closeAllByReference(Long referenceId, String referenceType, UserInterface user) {
        getDao().closeAllByReference(referenceId, referenceType, properties.getStatus().getClosed(), user);
    }
}
