package io.lsn.spring.issue.domain;

import io.lsn.spring.issue.configuration.IssueProperties;
import io.lsn.spring.issue.domain.entity.Comment;
import io.lsn.spring.issue.domain.entity.Issue;
import io.lsn.spring.issue.domain.entity.UserInterface;
import io.lsn.spring.issue.domain.entity.event.IssueCreatedEvent;
import io.lsn.spring.issue.domain.entity.event.IssueUpdatedEvent;
import io.lsn.spring.issue.domain.entity.event.WatcherAddedEvent;
import org.springframework.util.CollectionUtils;

@SuppressWarnings("unchecked")
public abstract class IssueAbstractService {

    private IssueDao dao;
    private EventDispatcher dispatcher;
    private IssueProperties properties;

    public IssueAbstractService(IssueDao dao, IssueProperties properties, EventDispatcher dispatcher) {
        this.dao = dao;
        this.properties = properties;
        this.dispatcher = dispatcher;
    }


    /**
     * get one issue by id
     *
     * @param id   issue ud
     * @param user user context
     * @return Issue
     */
    public abstract <T extends Issue> T getOne(Long id, UserInterface user);


    /**
     * create new issue
     * id and uid will be returned with the issue
     *
     * @param issue issue
     * @param user  user context
     */
    public <T extends Issue> T create(T issue, UserInterface user) {
        // create issue
        dao.create(issue, user);

        // add files
        issue.getFileHashList().forEach(f -> dao.addIssueFile(issue, f, user));

        // add author as watcher
        dao.addIssueWatcher(issue, issue.getAuthor().getId().longValue(), user);

        // add assignee
        issue.getAssigneeList().forEach(a -> addAssignee(issue, a.getId().longValue(), user));

        // attributes
        if (!CollectionUtils.isEmpty(issue.getAttributeList())) {
            updateAttributeList(issue, user);
        }

        T created = (T) getOne(issue.getId(), user);
        dispatcher.dispatch(new IssueCreatedEvent(created));
        return created;
    }


    /**
     * update attribute list
     *
     * @param issue issue data
     * @param user  user context
     * @param <T>   issue type
     */
    public <T extends Issue> void updateAttributeList(T issue, UserInterface user) {
        dao.updateAttributeList(issue, user);
    }


    /**
     * update issue data
     *
     * @param issue issue data
     * @param user  user context
     * @return Issue
     */
    public <T extends Issue> T updateIssue(T issue, UserInterface user) {
        dao.updateIssue(issue, user);
        dispatcher.dispatch(new IssueUpdatedEvent(issue));
        return (T) getOne(issue.getId(), user);
    }


    /**
     * add watcher to issue
     *
     * @param issue  issue
     * @param userId new watcher id
     * @param user   user context
     * @return Issue
     */
    public <T extends Issue> T addIssueWatcher(T issue, Long userId, UserInterface user) {
        dao.addIssueWatcher(issue, userId, user);
        dispatcher.dispatch(new WatcherAddedEvent(issue, userId));
        return (T) getOne(issue.getId(), user);
    }


    /**
     * remove watcher from issue
     *
     * @param issue  issue
     * @param userId watcher id
     * @param user   user context
     */
    public <T extends Issue> T removeIssueWatcher(T issue, Long userId, UserInterface user) {
        dao.removeIssueWatcher(issue, userId, user);
        return getOne(issue.getId(), user);
    }


    /**
     * add file to issue
     *
     * @param issue    issue
     * @param fileHash file hash
     */
    public <T extends Issue> T addIssueFile(T issue, String fileHash, UserInterface user) {
        issue.getFileHashList().add(fileHash);
        dao.addIssueFile(issue, fileHash, user);
        return getOne(issue.getId(), user);
    }


    /**
     * remove file from issue
     *
     * @param issue    issue
     * @param fileHash file hash
     * @param user     user context
     */
    public <T extends Issue> T removeIssueFile(T issue, String fileHash, UserInterface user) {
        dao.removeIssueFile(issue, fileHash, user);
        return getOne(issue.getId(), user);
    }


    /**
     * close issue
     *
     * @param issue issue
     * @param user  user context
     */
    public <T extends Issue> T closeIssue(T issue, UserInterface user) {
        dao.updateStatus(issue, properties.getStatus().getClosed(), user);
        return getOne(issue.getId(), user);
    }


    /**
     * reopen issue
     *
     * @param issue issue
     * @param user  user context
     */
    public <T extends Issue> T openIssue(T issue, UserInterface user) {
        dao.updateStatus(issue, properties.getStatus().getOpen(), user);
        return getOne(issue.getId(), user);
    }


    /**
     * add comment to issue
     *
     * @param issue   issue
     * @param comment comment
     * @param user    user context
     */
    public <T extends Issue> T addComment(T issue, Comment comment, UserInterface user) {
        // set issue id
        comment.setIssueId(issue.getId());

        // add comment
        dao.addComment(comment, user);

        // add author of the comment as watcher
        addIssueWatcher(issue, comment.getAuthor().getId().longValue(), user);

        // add files
        comment.getFileHashList().forEach(f -> dao.addCommentFile(comment, f, user));

        return getOne(issue.getId(), user);
    }


    /**
     * remove comment
     *
     * @param comment comment
     * @param user    user context
     */
    public <T extends Issue> T removeComment(Comment comment, UserInterface user) {
        // remove files
        dao.removeCommentFile(comment, user);

        // remove comment
        dao.removeComment(comment, user);

        return getOne(comment.getIssueId(), user);
    }


    /**
     * update comment
     *
     * @param comment comment
     * @param user    user context
     */
    public <T extends Issue> T updateComment(Comment comment, UserInterface user) {
        // remove old files
        dao.removeCommentFile(comment, user);

        // add new files
        comment.getFileHashList().forEach(f -> dao.addCommentFile(comment, f, user));

        // update comment itself
        dao.updateComment(comment, user);

        return getOne(comment.getIssueId(), user);
    }


    /**
     * add user to assignee list
     *
     * @param issue      issue
     * @param assigneeId assigneeId
     * @param user       user context
     * @param <T>        Issue
     * @return Issue
     */
    public <T extends Issue> T addAssignee(T issue, Long assigneeId, UserInterface user) {
        dao.addAssignee(issue, assigneeId, user);
        return getOne(issue.getId(), user);
    }


    /**
     * set assignee (remove all others)
     *
     * @param issue      issue
     * @param assigneeId assigneeId
     * @param user       user context
     * @param <T>        Issue
     * @return Issue
     */
    public <T extends Issue> T setAssignee(T issue, Long assigneeId, UserInterface user) {
        dao.setAssignee(issue, assigneeId, user);
        return getOne(issue.getId(), user);
    }


    /**
     * clear assignee list for the issue
     *
     * @param issue issue
     * @param user  user context
     * @param <T>   Issue
     * @return Issue
     */
    public <T extends Issue> T clearAssignee(T issue, UserInterface user) {
        dao.clearAssignee(issue, user);
        return getOne(issue.getId(), user);
    }


    /**
     * remove user from assignee list
     *
     * @param issue      issue
     * @param assigneeId assigneeId
     * @param user       user context
     * @param <T>        Issue
     * @return Issue
     */
    public <T extends Issue> T removeAssignee(T issue, Long assigneeId, UserInterface user) {
        dao.removeAssignee(issue, assigneeId, user);
        return getOne(issue.getId(), user);
    }


    /**
     * close all with referenceId and type
     *
     * @param referenceId   object reference id
     * @param referenceType object reference type
     * @param user          user context
     */
    public void closeAllByReference(Long referenceId, String referenceType, UserInterface user) {
        dao.closeAllByReference(referenceId, referenceType, properties.getStatus().getClosed(), user);
    }
}
