/*
 * Copyright 2016 Game Server Services, Inc. or its affiliates. All Rights
 * Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */

package io.gs2.identifier;

import java.util.ArrayList;
import java.util.List;

import io.gs2.model.Region;
import io.gs2.util.EncodingUtil;
import org.apache.http.NameValuePair;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.utils.URLEncodedUtils;
import org.apache.http.message.BasicNameValuePair;

import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.JsonNode;
import org.json.JSONObject;

import io.gs2.AbstractGs2Client;
import io.gs2.Gs2Constant;
import io.gs2.model.IGs2Credential;
import io.gs2.identifier.request.*;
import io.gs2.identifier.result.*;
import io.gs2.identifier.model.*;

/**
 * GS2 Identifier API クライアント
 *
 * @author Game Server Services, Inc.
 *
 */
public class Gs2IdentifierRestClient extends AbstractGs2Client<Gs2IdentifierRestClient> {

	public static String ENDPOINT = "identifier";

	/**
	 * コンストラクタ。
	 *
	 * @param credential 認証情報
	 */
	public Gs2IdentifierRestClient(IGs2Credential credential) {
		super(credential);
	}

	/**
	 * コンストラクタ。
	 *
	 * @param credential 認証情報
	 * @param region リージョン
	 */
	public Gs2IdentifierRestClient(IGs2Credential credential, Region region) {
		super(credential, region);
	}

	/**
	 * コンストラクタ。
	 *
	 * @param credential 認証情報
	 * @param region リージョン
	 */
	public Gs2IdentifierRestClient(IGs2Credential credential, String region) {
		super(credential, region);

	}

	/**
	 * ユーザの一覧を取得します<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DescribeUsersResult describeUsers(DescribeUsersRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/identifier-handler?handler=gs2_identifier%2Fhandler%2FUserFunctionHandler.describeUsers";
        String pageToken = String.valueOf(request.getPageToken());
        String limit = String.valueOf(request.getLimit());
        List<NameValuePair> _queryString = new ArrayList<>();
        if(pageToken != null) {
            _queryString.add(new BasicNameValuePair("pageToken", String.valueOf(pageToken)));
        }
        if(limit != null) {
            _queryString.add(new BasicNameValuePair("limit", String.valueOf(limit)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DescribeUsersResult.class);
    }

	/**
	 * ユーザを新規作成します<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public CreateUserResult createUser(CreateUserRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/identifier-handler?handler=gs2_identifier%2Fhandler%2FUserFunctionHandler.createUser";
        String name = null;
        if(request.getName() != null) {
            name = request.getName();
        }
        String description = null;
        if(request.getDescription() != null) {
            description = request.getDescription();
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(name != null) {
            _body.put("name", name);
        }
        if(description != null) {
            _body.put("description", description);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, CreateUserResult.class);
    }

	/**
	 * ユーザを更新します<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public UpdateUserResult updateUser(UpdateUserRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/identifier-handler?handler=gs2_identifier%2Fhandler%2FUserFunctionHandler.updateUser";
        String userName = null;
        if(request.getUserName() != null) {
            userName = request.getUserName();
        }
        String description = null;
        if(request.getDescription() != null) {
            description = request.getDescription();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(userName != null) {
            _queryString.add(new BasicNameValuePair("userName", String.valueOf(userName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(description != null) {
            _body.put("description", description);
        }
		HttpPut http = createHttpPut(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, UpdateUserResult.class);
    }

	/**
	 * ユーザを取得します<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetUserResult getUser(GetUserRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/identifier-handler?handler=gs2_identifier%2Fhandler%2FUserFunctionHandler.getUser";
        String userName = null;
        if(request.getUserName() != null) {
            userName = request.getUserName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(userName != null) {
            _queryString.add(new BasicNameValuePair("userName", String.valueOf(userName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, GetUserResult.class);
    }

	/**
	 * ユーザを削除します<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DeleteUserResult deleteUser(DeleteUserRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/identifier-handler?handler=gs2_identifier%2Fhandler%2FUserFunctionHandler.deleteUser";
        String userName = null;
        if(request.getUserName() != null) {
            userName = request.getUserName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(userName != null) {
            _queryString.add(new BasicNameValuePair("userName", String.valueOf(userName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpDelete http = createHttpDelete(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DeleteUserResult.class);
    }

	/**
	 * セキュリティポリシーの一覧を取得します<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DescribeSecurityPoliciesResult describeSecurityPolicies(DescribeSecurityPoliciesRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/identifier-handler?handler=gs2_identifier%2Fhandler%2FSecurityPolicyFunctionHandler.describeSecurityPolicies";
        String pageToken = String.valueOf(request.getPageToken());
        String limit = String.valueOf(request.getLimit());
        List<NameValuePair> _queryString = new ArrayList<>();
        if(pageToken != null) {
            _queryString.add(new BasicNameValuePair("pageToken", String.valueOf(pageToken)));
        }
        if(limit != null) {
            _queryString.add(new BasicNameValuePair("limit", String.valueOf(limit)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DescribeSecurityPoliciesResult.class);
    }

	/**
	 * オーナーIDを指定してセキュリティポリシーの一覧を取得します<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DescribeCommonSecurityPoliciesResult describeCommonSecurityPolicies(DescribeCommonSecurityPoliciesRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/identifier-handler?handler=gs2_identifier%2Fhandler%2FSecurityPolicyFunctionHandler.describeCommonSecurityPolicies";
        String ownerId = null;
        if(request.getOwnerId() != null) {
            ownerId = request.getOwnerId();
        }
        String pageToken = String.valueOf(request.getPageToken());
        String limit = String.valueOf(request.getLimit());
        List<NameValuePair> _queryString = new ArrayList<>();
        if(pageToken != null) {
            _queryString.add(new BasicNameValuePair("pageToken", String.valueOf(pageToken)));
        }
        if(limit != null) {
            _queryString.add(new BasicNameValuePair("limit", String.valueOf(limit)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DescribeCommonSecurityPoliciesResult.class);
    }

	/**
	 * セキュリティポリシーを新規作成します<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public CreateSecurityPolicyResult createSecurityPolicy(CreateSecurityPolicyRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/identifier-handler?handler=gs2_identifier%2Fhandler%2FSecurityPolicyFunctionHandler.createSecurityPolicy";
        String name = null;
        if(request.getName() != null) {
            name = request.getName();
        }
        String description = null;
        if(request.getDescription() != null) {
            description = request.getDescription();
        }
        String policy = null;
        if(request.getPolicy() != null) {
            policy = request.getPolicy();
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(name != null) {
            _body.put("name", name);
        }
        if(description != null) {
            _body.put("description", description);
        }
        if(policy != null) {
            _body.put("policy", policy);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, CreateSecurityPolicyResult.class);
    }

	/**
	 * セキュリティポリシーを更新します<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public UpdateSecurityPolicyResult updateSecurityPolicy(UpdateSecurityPolicyRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/identifier-handler?handler=gs2_identifier%2Fhandler%2FSecurityPolicyFunctionHandler.updateSecurityPolicy";
        String securityPolicyName = null;
        if(request.getSecurityPolicyName() != null) {
            securityPolicyName = request.getSecurityPolicyName();
        }
        String description = null;
        if(request.getDescription() != null) {
            description = request.getDescription();
        }
        String policy = null;
        if(request.getPolicy() != null) {
            policy = request.getPolicy();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(securityPolicyName != null) {
            _queryString.add(new BasicNameValuePair("securityPolicyName", String.valueOf(securityPolicyName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(description != null) {
            _body.put("description", description);
        }
        if(policy != null) {
            _body.put("policy", policy);
        }
		HttpPut http = createHttpPut(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, UpdateSecurityPolicyResult.class);
    }

	/**
	 * セキュリティポリシーを取得します<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetSecurityPolicyResult getSecurityPolicy(GetSecurityPolicyRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/identifier-handler?handler=gs2_identifier%2Fhandler%2FSecurityPolicyFunctionHandler.getSecurityPolicy";
        String securityPolicyName = null;
        if(request.getSecurityPolicyName() != null) {
            securityPolicyName = request.getSecurityPolicyName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(securityPolicyName != null) {
            _queryString.add(new BasicNameValuePair("securityPolicyName", String.valueOf(securityPolicyName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, GetSecurityPolicyResult.class);
    }

	/**
	 * セキュリティポリシーを削除します<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DeleteSecurityPolicyResult deleteSecurityPolicy(DeleteSecurityPolicyRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/identifier-handler?handler=gs2_identifier%2Fhandler%2FSecurityPolicyFunctionHandler.deleteSecurityPolicy";
        String securityPolicyName = null;
        if(request.getSecurityPolicyName() != null) {
            securityPolicyName = request.getSecurityPolicyName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(securityPolicyName != null) {
            _queryString.add(new BasicNameValuePair("securityPolicyName", String.valueOf(securityPolicyName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpDelete http = createHttpDelete(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DeleteSecurityPolicyResult.class);
    }

	/**
	 * クレデンシャルの一覧を取得します<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DescribeIdentifiersResult describeIdentifiers(DescribeIdentifiersRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/identifier-handler?handler=gs2_identifier%2Fhandler%2FIdentifierFunctionHandler.describeIdentifiers";
        String userName = null;
        if(request.getUserName() != null) {
            userName = request.getUserName();
        }
        String pageToken = String.valueOf(request.getPageToken());
        String limit = String.valueOf(request.getLimit());
        List<NameValuePair> _queryString = new ArrayList<>();
        if(userName != null) {
            _queryString.add(new BasicNameValuePair("userName", String.valueOf(userName)));
        }
        if(pageToken != null) {
            _queryString.add(new BasicNameValuePair("pageToken", String.valueOf(pageToken)));
        }
        if(limit != null) {
            _queryString.add(new BasicNameValuePair("limit", String.valueOf(limit)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DescribeIdentifiersResult.class);
    }

	/**
	 * クレデンシャルを新規作成します<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public CreateIdentifierResult createIdentifier(CreateIdentifierRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/identifier-handler?handler=gs2_identifier%2Fhandler%2FIdentifierFunctionHandler.createIdentifier";
        String userName = null;
        if(request.getUserName() != null) {
            userName = request.getUserName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(userName != null) {
            _queryString.add(new BasicNameValuePair("userName", String.valueOf(userName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, CreateIdentifierResult.class);
    }

	/**
	 * クレデンシャルを取得します<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetIdentifierResult getIdentifier(GetIdentifierRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/identifier-handler?handler=gs2_identifier%2Fhandler%2FIdentifierFunctionHandler.getIdentifier";
        String userName = null;
        if(request.getUserName() != null) {
            userName = request.getUserName();
        }
        String clientId = null;
        if(request.getClientId() != null) {
            clientId = request.getClientId();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(userName != null) {
            _queryString.add(new BasicNameValuePair("userName", String.valueOf(userName)));
        }
        if(clientId != null) {
            _queryString.add(new BasicNameValuePair("clientId", String.valueOf(clientId)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, GetIdentifierResult.class);
    }

	/**
	 * クレデンシャルを削除します<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DeleteIdentifierResult deleteIdentifier(DeleteIdentifierRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/identifier-handler?handler=gs2_identifier%2Fhandler%2FIdentifierFunctionHandler.deleteIdentifier";
        String userName = null;
        if(request.getUserName() != null) {
            userName = request.getUserName();
        }
        String clientId = null;
        if(request.getClientId() != null) {
            clientId = request.getClientId();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(userName != null) {
            _queryString.add(new BasicNameValuePair("userName", String.valueOf(userName)));
        }
        if(clientId != null) {
            _queryString.add(new BasicNameValuePair("clientId", String.valueOf(clientId)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpDelete http = createHttpDelete(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DeleteIdentifierResult.class);
    }

	/**
	 * 割り当てられたセキュリティポリシーの一覧を取得します<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetHasSecurityPolicyResult getHasSecurityPolicy(GetHasSecurityPolicyRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/identifier-handler?handler=gs2_identifier%2Fhandler%2FAttachSecurityPolicyFunctionHandler.getHasSecurityPolicy";
        String userName = null;
        if(request.getUserName() != null) {
            userName = request.getUserName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(userName != null) {
            _queryString.add(new BasicNameValuePair("userName", String.valueOf(userName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, GetHasSecurityPolicyResult.class);
    }

	/**
	 * 割り当てられたセキュリティポリシーを新しくユーザーに割り当てます<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public AttachSecurityPolicyResult attachSecurityPolicy(AttachSecurityPolicyRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/identifier-handler?handler=gs2_identifier%2Fhandler%2FAttachSecurityPolicyFunctionHandler.attachSecurityPolicy";
        String userName = null;
        if(request.getUserName() != null) {
            userName = request.getUserName();
        }
        String securityPolicyId = null;
        if(request.getSecurityPolicyId() != null) {
            securityPolicyId = request.getSecurityPolicyId();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(userName != null) {
            _queryString.add(new BasicNameValuePair("userName", String.valueOf(userName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(securityPolicyId != null) {
            _body.put("securityPolicyId", securityPolicyId);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, AttachSecurityPolicyResult.class);
    }

	/**
	 * 割り当てられたセキュリティポリシーをユーザーから外します<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DetachSecurityPolicyResult detachSecurityPolicy(DetachSecurityPolicyRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/identifier-handler?handler=gs2_identifier%2Fhandler%2FAttachSecurityPolicyFunctionHandler.detachSecurityPolicy";
        String userName = null;
        if(request.getUserName() != null) {
            userName = request.getUserName();
        }
        String securityPolicyId = null;
        if(request.getSecurityPolicyId() != null) {
            securityPolicyId = request.getSecurityPolicyId();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(userName != null) {
            _queryString.add(new BasicNameValuePair("userName", String.valueOf(userName)));
        }
        if(securityPolicyId != null) {
            _queryString.add(new BasicNameValuePair("securityPolicyId", String.valueOf(securityPolicyId)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpDelete http = createHttpDelete(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DetachSecurityPolicyResult.class);
    }

	/**
	 * プロジェクトトークン を取得します<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public LoginResult login(LoginRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/identifier-handler?handler=gs2_identifier%2Fhandler%2FProjectTokenFunctionHandler.login";
        String clientId = null;
        if(request.getClientId() != null) {
            clientId = request.getClientId();
        }
        String clientSecret = null;
        if(request.getClientSecret() != null) {
            clientSecret = request.getClientSecret();
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(clientId != null) {
            _body.put("clientId", clientId);
        }
        if(clientSecret != null) {
            _body.put("clientSecret", clientSecret);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, LoginResult.class);
    }
}