/*
 * Copyright © 2009 HotPads (admin@hotpads.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.datarouter.storage.serialize.fieldcache;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import io.datarouter.model.databean.Databean;
import io.datarouter.model.field.Field;
import io.datarouter.model.field.FieldKey;
import io.datarouter.model.field.encoding.FieldGeneratorType;
import io.datarouter.model.key.primary.EntityPrimaryKey;
import io.datarouter.model.key.primary.PrimaryKey;
import io.datarouter.model.serialize.fielder.DatabeanFielder;
import io.datarouter.storage.node.NodeParams;
import io.datarouter.util.bytes.ByteTool;
import io.datarouter.util.bytes.StringByteTool;
import io.datarouter.util.lang.ReflectionTool;
import io.datarouter.util.string.StringTool;

public class DatabeanFieldInfo<
		PK extends PrimaryKey<PK>,
		D extends Databean<PK,D>,
		F extends DatabeanFielder<PK,D>>{

	private static final byte ENTITY_PREFIX_TERMINATOR = 0;

	private final Supplier<PK> primaryKeySupplier;
	private final PK samplePrimaryKey;
	private final Supplier<D> databeanSupplier;
	private final D sampleDatabean;
	private final java.lang.reflect.Field keyJavaField;
	private final Map<String,List<Field<?>>> uniqueIndexes;

	private final boolean subEntity;
	private final String entityNodePrefix;
	private final byte[] entityNodeColumnPrefixBytes;
	private final List<Field<?>> ekFields;//for accessing the EK directly
	private final List<Field<?>> ekPkFields;//for accessing the EK via a PK
	private final List<Field<?>> postEkPkKeyFields;
	private final boolean singleDatabeanEntity;

	private final Supplier<F> fielderSupplier;
	private final F sampleFielder;

	private final FieldGeneratorType autoGeneratedType;
	private String autoGeneratedFieldName;
	private final Optional<Long> ttlMs;
	//these hold separate deep-copies of the fields
	private final List<Field<?>> primaryKeyFields;//no prefixes
	private final List<Field<?>> nonKeyFields;
	private final List<Field<?>> fields;//PK fields will have prefixes in this Collection

	private final Supplier<Boolean> recordCallsite;

	private final Map<String,Field<?>> nonKeyFieldByColumnName = new HashMap<>();
	private final Map<String,Field<?>> fieldByColumnName = new HashMap<>();
	private final Map<String,Field<?>> fieldByPrefixedName = new HashMap<>();

	private final List<String> primaryKeyFieldColumnNames;
	private final List<String> fieldColumnNames;
	private final List<String> nonKeyFieldColumnNames;

	public DatabeanFieldInfo(NodeParams<PK,D,F> params){
		this.databeanSupplier = params.getDatabeanSupplier();
		this.sampleDatabean = databeanSupplier.get();
		this.primaryKeySupplier = sampleDatabean.getKeySupplier();
		this.samplePrimaryKey = primaryKeySupplier.get();
		this.fielderSupplier = params.getFielderSupplier();
		this.sampleFielder = fielderSupplier.get();
		this.autoGeneratedType = detectAutoGeneratedType();
		this.ttlMs = sampleFielder.getTtlMs();
		this.recordCallsite = params.getRecordCallsites();
		this.primaryKeyFields = samplePrimaryKey.getFields();
		this.fields = sampleFielder.getFields(sampleDatabean);//make sure there is a PK or this will NPE
		addFieldsToCollections();
		//only do these if the previous fields succeeded
		this.nonKeyFields = sampleFielder.getNonKeyFields(sampleDatabean);
		addNonKeyFieldsToCollections();
		this.uniqueIndexes = new HashMap<>(sampleFielder.getUniqueIndexes(sampleDatabean));
		this.primaryKeyFieldColumnNames = createFieldColumnNames(this.primaryKeyFields);
		this.fieldColumnNames = createFieldColumnNames(this.fields);
		this.nonKeyFieldColumnNames = createFieldColumnNames(this.nonKeyFields);
		this.keyJavaField = ReflectionTool.getDeclaredFieldFromAncestors(sampleDatabean.getClass(), sampleDatabean
				.getKeyFieldName());
		this.subEntity = StringTool.notEmpty(params.getEntityNodePrefix());
		this.entityNodePrefix = params.getEntityNodePrefix();
		this.entityNodeColumnPrefixBytes = ByteTool.concatenate(StringByteTool.getUtf8Bytes(entityNodePrefix),
				new byte[]{ENTITY_PREFIX_TERMINATOR});
		EntityPrimaryKey<?,?> sampleEntityPrimaryKey = (EntityPrimaryKey<?,?>)samplePrimaryKey;
		this.ekFields = sampleEntityPrimaryKey.getEntityKey().getFields();
		this.ekPkFields = sampleEntityPrimaryKey.getEntityKeyFields();
		this.postEkPkKeyFields = sampleEntityPrimaryKey.getPostEntityKeyFields();
		this.singleDatabeanEntity = postEkPkKeyFields.isEmpty();
	}

	/*----------------------------- methods ---------------------------------*/

	public Field<?> getFieldForColumnName(String columnName){
		return fieldByColumnName.get(columnName);
	}

	public List<Field<?>> getFieldsWithValues(D databean){
		return sampleFielder.getFields(databean);
	}

	public List<Field<?>> getNonKeyFieldsWithValues(D databean){
		return sampleFielder.getNonKeyFields(databean);
	}

	private static List<String> createFieldColumnNames(List<Field<?>> fields){
		return fields.stream()
				.map(Field::getKey)
				.map(FieldKey::getColumnName)
				.collect(Collectors.toList());
	}

	private void addNonKeyFieldsToCollections(){
		for(Field<?> field : nonKeyFields){
			this.nonKeyFieldByColumnName.put(field.getKey().getColumnName(), field);
		}
	}

	private void addFieldsToCollections(){
		for(Field<?> field : fields){
			this.fieldByColumnName.put(field.getKey().getColumnName(), field);
			this.fieldByPrefixedName.put(field.getPrefixedName(), field);
		}
	}

	private FieldGeneratorType detectAutoGeneratedType(){
		for(Field<?> field : samplePrimaryKey.getFields()){
			if(field.getKey().getAutoGeneratedType().isGenerated()){
				this.autoGeneratedFieldName = field.getKey().getName();
				return field.getKey().getAutoGeneratedType();
			}
		}
		return FieldGeneratorType.NONE;
	}


	/*----------------------------- get/set ---------------------------------*/

	public Supplier<PK> getPrimaryKeySupplier(){
		return primaryKeySupplier;
	}

	public PK getSamplePrimaryKey(){
		return samplePrimaryKey;
	}

	public Supplier<D> getDatabeanSupplier(){
		return databeanSupplier;
	}

	public D getSampleDatabean(){
		return sampleDatabean;
	}

	public Supplier<F> getFielderSupplier(){
		return fielderSupplier;
	}

	public F getSampleFielder(){
		return sampleFielder;
	}

	public List<Field<?>> getPrimaryKeyFields(){
		return primaryKeyFields;
	}

	public List<Field<?>> getNonKeyFields(){
		return nonKeyFields;
	}

	public List<Field<?>> getFields(){
		return fields;
	}

	public Map<String,Field<?>> getNonKeyFieldByColumnName(){
		return nonKeyFieldByColumnName;
	}

	public Map<String,Field<?>> getFieldByPrefixedName(){
		return fieldByPrefixedName;
	}

	public Map<String,List<Field<?>>> getUniqueIndexes(){
		return uniqueIndexes;
	}

	public FieldGeneratorType getAutoGeneratedType(){
		return autoGeneratedType;
	}

	public boolean isManagedAutoGeneratedId(){
		return FieldGeneratorType.MANAGED == autoGeneratedType;
	}

	public String getAutogeneratedFieldName(){
		return autoGeneratedFieldName;
	}

	public String getEntityNodePrefix(){
		return entityNodePrefix;
	}

	public byte[] getEntityColumnPrefixBytes(){
		return entityNodeColumnPrefixBytes;
	}

	public java.lang.reflect.Field getKeyJavaField(){
		return keyJavaField;
	}

	public boolean isSubEntity(){
		return subEntity;
	}

	public List<Field<?>> getEkFields(){
		return ekFields;
	}

	public List<Field<?>> getEkPkFields(){
		return ekPkFields;
	}

	public List<Field<?>> getPostEkPkKeyFields(){
		return postEkPkKeyFields;
	}

	public boolean isSingleDatabeanEntity(){
		return singleDatabeanEntity;
	}

	public Optional<Long> getTtlMs(){
		return ttlMs;
	}

	public Supplier<Boolean> getRecordCallsite(){
		return recordCallsite;
	}

	public List<String> getPrimaryKeyFieldColumnNames(){
		return primaryKeyFieldColumnNames;
	}

	public List<String> getFieldColumnNames(){
		return fieldColumnNames;
	}

	public List<String> getNonKeyFieldColumnNames(){
		return nonKeyFieldColumnNames;
	}

}
