package io.avaje.spi.internal;

import static java.util.stream.Collectors.*;

import java.io.*;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.processing.Generated;
import javax.lang.model.element.ModuleElement;
import javax.lang.model.element.ModuleElement.DirectiveKind;
import javax.lang.model.element.ModuleElement.RequiresDirective;
import javax.tools.StandardLocation;
import javax.lang.model.element.PackageElement;

/**
 * Helper Class to work with an application's root module-info.
 *
 * <p>Calling {@link ModuleElement#getDirectives()} on the application module can break compilation
 * in some situations, so this class helps parse the module source file and get the relevant
 * information without breaking anything.
 */
@Generated("avaje-prism-generator")
public class ModuleInfoReader {

  private static final String SPLIT_PATTERN = "\\s*,\\s*";
  private static final Pattern IMPORT_PATTERN = Pattern.compile("import\\s+([\\w.$]+);");
  private static final Pattern REQUIRES_PATTERN =
      Pattern.compile("requires\\s+(transitive\\s+)?(static\\s+)?([\\w.$]+);");
  private static final Pattern PROVIDES_PATTERN =
      Pattern.compile("provides\\s+([\\w.$]+)\\s+with\\s+([\\w.$,\\s]+);");
  private static final Pattern OPENS_PATTERN =
      Pattern.compile("opens\\s+([\\w.$]+)\\s+to\\s+([\\w.$,\\s]+);");
  private static final Pattern EXPORTS_PATTERN =
      Pattern.compile("exports\\s+([\\w.$]+)\\s+to\\s+([\\w.$,\\s]+);");
  private static final Pattern USES_PATTERN = Pattern.compile("uses\\s+([\\w.$]+);");

  private final List<Requires> requires = new ArrayList<>();
  private final List<Uses> uses = new ArrayList<>();
  private final List<Exports> exports = new ArrayList<>();
  private final List<Opens> opens = new ArrayList<>();
  private final List<Provides> provides = new ArrayList<>();
  private final ModuleElement moduleElement;

  /**
   * Parse a module-info and create a new instance
   *
   * @param moduleElement the element representing the root module
   * @param reader a reader for the contents of the module-info.java
   */
  public ModuleInfoReader(ModuleElement moduleElement, BufferedReader reader) throws IOException {
    this(moduleElement, reader.lines().collect(joining("\n")));
    reader.close();
  }

  /**
   * Parse a module-info and create a new instance
   *
   * @param moduleElement the element representing the root module
   * @param moduleString a string containing the contents of the module-info.java
   */
  public ModuleInfoReader(ModuleElement moduleElement, CharSequence moduleString) {
    this.moduleElement = moduleElement;
    Matcher importMatcher = IMPORT_PATTERN.matcher(moduleString);
    Matcher requiresMatcher = REQUIRES_PATTERN.matcher(moduleString);
    Matcher providesMatcher = PROVIDES_PATTERN.matcher(moduleString);
    Matcher opensMatcher = OPENS_PATTERN.matcher(moduleString);
    Matcher exportsMatcher = EXPORTS_PATTERN.matcher(moduleString);
    Matcher usesMatcher = USES_PATTERN.matcher(moduleString);

    while (requiresMatcher.find()) {
      boolean transitive = requiresMatcher.group(1) != null;
      boolean isStatic = requiresMatcher.group(2) != null;
      String dep = requiresMatcher.group(3);
      requires.add(new Requires(APContext.elements().getModuleElement(dep), transitive, isStatic));
    }

    while (opensMatcher.find()) {
      String packageName = opensMatcher.group(1);
      String targets = opensMatcher.group(2);
      List<ModuleElement> openTargets =
          Arrays.stream(targets.split(SPLIT_PATTERN))
              .map(APContext.elements()::getModuleElement)
              .collect(toList());
      opens.add(new Opens(moduleElement, packageName, openTargets));
    }

    while (exportsMatcher.find()) {
      String packageName = exportsMatcher.group(1);
      String targets = exportsMatcher.group(2);
      List<ModuleElement> exportTargets =
          Arrays.stream(targets.split(SPLIT_PATTERN))
              .map(APContext.elements()::getModuleElement)
              .collect(toList());
      exports.add(new Exports(moduleElement, packageName, exportTargets));
    }

    var imports = new ArrayList<String>();

    while (importMatcher.find()) {
      imports.add(importMatcher.group(1));
    }

    while (providesMatcher.find()) {
      String providedInterface = resolveImport(imports, providesMatcher.group(1));
      String implementationClasses = providesMatcher.group(2);

      List<String> implementations =
          Arrays.stream(implementationClasses.split(SPLIT_PATTERN))
              .map(s -> resolveImport(imports, s))
              .collect(toList());
      provides.add(new Provides(providedInterface, implementations));
    }
    while (usesMatcher.find()) {
      String usedInterface = resolveImport(imports, usesMatcher.group(1));

      uses.add(new Uses(usedInterface));
    }
  }

  private String resolveImport(List<String> imports, String providedInterface) {
    return imports.stream()
        .filter(s -> s.contains(providedInterface))
        .findFirst()
        .orElse(providedInterface)
        .replaceAll("\\s", "");
  }

  /**
   * Check to see whether the given module is on the module path as a non-static dependency
   *
   * @param moduleName
   * @return whether the given module is on the path
   */
  public boolean containsOnModulePath(String moduleName) {
    if (requires.isEmpty()) {
      return false;
    }
    var surfaceCheck =
        requires.stream()
            .filter(r -> !r.isStatic)
            .anyMatch(r -> r.dependency.getQualifiedName().contentEquals(moduleName));

    if (surfaceCheck) {
      return true;
    }

    var seen = new HashSet<String>();
    return requires.parallelStream()
        .filter(r -> !r.isStatic)
        .anyMatch(r -> hasNonStaticModule(moduleName, r.dependency, seen));
  }

  private boolean hasNonStaticModule(String name, ModuleElement element, Set<String> seen) {
    if (!seen.add(element.getQualifiedName().toString())) {
      return false;
    }

    var directives =
        element.getDirectives().stream()
            .filter(d -> d.getKind() == DirectiveKind.REQUIRES)
            .map(RequiresDirective.class::cast)
            .filter(r -> !r.isStatic())
            .collect(toList());
    if (directives.isEmpty()) {
      return false;
    }
    var surfaceCheck =
        directives.stream().anyMatch(r -> r.getDependency().getQualifiedName().contentEquals(name));

    if (surfaceCheck) {
      return true;
    }

    return requires.stream().anyMatch(r -> hasNonStaticModule(name, r.getDependency(), seen));
  }

  private String replace$(String k) {
    return k.replace('$', '.');
  }

  /**
   * Checks whether the module-info has the defined provides directive and all their implementations
   * Will register an error message compilation
   *
   * @param providesType the provides directive to check
   * @param implementations the implementations to verify the presence of
   */
  public void validateServices(String providesType, Collection<String> implementations) {
    if (buildPluginAvailable() || moduleElement.isUnnamed() || APContext.isTestCompilation()) {
      return;
    }
    var implSet = new TreeSet<>(implementations);
    try (final var file =
            APContext.filer()
                .getResource(StandardLocation.CLASS_OUTPUT, "", "META-INF/services/" + providesType)
                .toUri()
                .toURL()
                .openStream();
        final var buffer = new BufferedReader(new InputStreamReader(file)); ) {

      String line;
      while ((line = buffer.readLine()) != null) {
        line.replaceAll("\\s", "").replace(",", "\n").lines().forEach(implSet::add);
      }
    } catch (Exception e) {
      // not a critical error
    }
    final var missingImpls = implSet.stream().map(this::replace$).collect(toSet());

    provides()
        .forEach(
            p -> {
              final var contract = replace$(providesType);
              if (!providesType.equals(contract)) {
                return;
              }
              var impls = p.implementations();
              if (missingImpls.size() > impls.size()) {
                return;
              }
              impls.stream().map(this::replace$).forEach(missingImpls::remove);
            });

    if (!missingImpls.isEmpty()) {
      var message = implementations.stream().collect(joining(", "));

      APContext.logError(moduleElement, "Missing `provides %s with %s;`", providesType, message);
    }
  }

    private static boolean buildPluginAvailable() {
    return isPresent("avaje-plugin-exists.txt");
  }

  private static boolean isPresent(String path) {
    try {

      return APContext.getBuildResource(path).toFile().exists();
    } catch (Exception e) {
      return false;
    }
  }

  /** The requires directives associated with this module */
  public List<Requires> requires() {
    return requires;
  }

  /** The uses directives associated with this module */
  public List<Uses> uses() {
    return uses;
  }

  /** The exports directives associated with this module */
  public List<Exports> exports() {
    return exports;
  }

  /** The opens directives associated with this module */
  public List<Opens> opens() {
    return opens;
  }

  /** The provides directives associated with this module */
  public List<Provides> provides() {
    return provides;
  }

  /** A dependency of a module. */
  public static class Requires {
    private final ModuleElement dependency;
    private final boolean isTransitive;
    private final boolean isStatic;

    public Requires(ModuleElement dependency, boolean isTransitive, boolean isStatic) {
      this.dependency = dependency;
      this.isTransitive = isTransitive;
      this.isStatic = isStatic;
    }

    /** {@return whether or not this is a static dependency} */
    public boolean isStatic() {
      return isStatic;
    }

    /** {@return whether or not this is a transitive dependency} */
    public boolean isTransitive() {
      return isTransitive;
    }

    /** {@return the module that is required} */
    public ModuleElement getDependency() {
      return dependency;
    }
  }

  /** An implementation of a service provided by a module. */
  public static class Provides {

    private final String type;
    private final List<String> impls;

    public Provides(String type, List<String> impls) {
      this.type = type;
      this.impls = impls;
    }

    /** {@return the service being provided} */
    public String service() {
      return type;
    }

    /** {@return the implementations of the service being provided} */
    public List<String> implementations() {
      return impls;
    }
  }

  /** A reference to a service used by a module. */
  public static class Uses {
    private final String service;

    public Uses(String usedInterface) {
      this.service = usedInterface;
    }

    /** {@return the service that is used} */
    public String service() {
      return service;
    }
  }

  /** An opened package of a module. */
  public static class Opens {

    private final ModuleElement parent;

    private final String packageName;
    private final List<ModuleElement> targets;

    public Opens(ModuleElement parent, String packageName, List<ModuleElement> targets) {
      this.parent = parent;
      this.packageName = packageName;
      this.targets = targets.isEmpty() ? null : targets;
    }

    /** {@return the name of the package being opened} */
    public String packageName() {
      return packageName;
    }

    /** {@return the package being opened} */
    public PackageElement getPackage() {
      return APContext.elements().getPackageElement(parent, packageName);
    }

    /**
     * Returns the specific modules to which the package is being open or {@code null}, if the
     * package is open all modules which have readability to this module.
     *
     * @return the specific modules to which the package is being opened
     */
    public List<ModuleElement> getTargetModules() {
      return targets;
    }
  }

  /** An exported package of a module. */
  public static class Exports {

    private final ModuleElement parent;

    private final String packageName;
    private final List<ModuleElement> targets;

    public Exports(ModuleElement parent, String packageName, List<ModuleElement> targets) {
      this.parent = parent;
      this.packageName = packageName;
      this.targets = targets.isEmpty() ? null : targets;
    }

    /** {@return the name of the package being exported} */
    public String packageName() {
      return packageName;
    }

    /** {@return the package being exported} */
    public PackageElement getPackage() {
      return APContext.elements().getPackageElement(parent, packageName);
    }

    /**
     * Returns the specific modules to which the package is being exported, or {@code null}, if the
     * package is exported to all modules which have readability to this module.
     *
     * @return the specific modules to which the package is being exported
     */
    public List<ModuleElement> getTargetModules() {
      return targets;
    }
  }
}