package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** SecretKeySelector selects a key of a Secret. */
final case class SecretKeySelector(
  key : String,
  name : Option[String] = None,
  optional : Option[Boolean] = None
) {

  /** Returns a new data with key set to new value */
  def withKey(value: String) : SecretKeySelector = copy(key = value)
  /** transforms key to result of function */
  def mapKey(f: String => String) : SecretKeySelector = copy(key = f(key))

  /** Returns a new data with name set to new value */
  def withName(value: String) : SecretKeySelector = copy(name = Some(value))
  /** if name has a value, transforms to the result of function*/
  def mapName(f: String => String) : SecretKeySelector = copy(name = name.map(f))

  /** Returns a new data with optional set to new value */
  def withOptional(value: Boolean) : SecretKeySelector = copy(optional = Some(value))
  /** if optional has a value, transforms to the result of function*/
  def mapOptional(f: Boolean => Boolean) : SecretKeySelector = copy(optional = optional.map(f))
}

object SecretKeySelector {

    implicit val encoder : Encoder[io.k8s.api.core.v1.SecretKeySelector] = new Encoder[io.k8s.api.core.v1.SecretKeySelector] {
        def apply[T : Builder](o: io.k8s.api.core.v1.SecretKeySelector) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("key", o.key)
            .write("name", o.name)
            .write("optional", o.optional)
            .build
        }
    }

    implicit val decoder: Decoder[SecretKeySelector] = new Decoder[SecretKeySelector] {
      def apply[T : Reader](t: T): Either[String, SecretKeySelector] = for {
          obj <- ObjectReader(t)
          key <- obj.read[String]("key")
          name <- obj.readOpt[String]("name")
          optional <- obj.readOpt[Boolean]("optional")
      } yield SecretKeySelector (
          key = key,
          name = name,
          optional = optional
        )
    }
}

