package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** EndpointSubset is a group of addresses with a common set of ports. The expanded set of endpoints is the Cartesian product of Addresses x Ports. For example, given:

	{
	  Addresses: [{"ip": "10.10.1.1"}, {"ip": "10.10.2.2"}],
	  Ports:     [{"name": "a", "port": 8675}, {"name": "b", "port": 309}]
	}

The resulting set of endpoints can be viewed as:

	a: [ 10.10.1.1:8675, 10.10.2.2:8675 ],
	b: [ 10.10.1.1:309, 10.10.2.2:309 ] */
final case class EndpointSubset(
  addresses : Option[Seq[io.k8s.api.core.v1.EndpointAddress]] = None,
  notReadyAddresses : Option[Seq[io.k8s.api.core.v1.EndpointAddress]] = None,
  ports : Option[Seq[io.k8s.api.core.v1.EndpointPort]] = None
) {

  /** Returns a new data with addresses set to new value */
  def withAddresses(value: Seq[io.k8s.api.core.v1.EndpointAddress]) : EndpointSubset = copy(addresses = Some(value))
  /** Appends new values to addresses */
  def addAddresses(newValues: io.k8s.api.core.v1.EndpointAddress*) : EndpointSubset = copy(addresses = Some(addresses.fold(newValues)(_ ++ newValues)))
  /** if addresses has a value, transforms to the result of function*/
  def mapAddresses(f: Seq[io.k8s.api.core.v1.EndpointAddress] => Seq[io.k8s.api.core.v1.EndpointAddress]) : EndpointSubset = copy(addresses = addresses.map(f))

  /** Returns a new data with notReadyAddresses set to new value */
  def withNotReadyAddresses(value: Seq[io.k8s.api.core.v1.EndpointAddress]) : EndpointSubset = copy(notReadyAddresses = Some(value))
  /** Appends new values to notReadyAddresses */
  def addNotReadyAddresses(newValues: io.k8s.api.core.v1.EndpointAddress*) : EndpointSubset = copy(notReadyAddresses = Some(notReadyAddresses.fold(newValues)(_ ++ newValues)))
  /** if notReadyAddresses has a value, transforms to the result of function*/
  def mapNotReadyAddresses(f: Seq[io.k8s.api.core.v1.EndpointAddress] => Seq[io.k8s.api.core.v1.EndpointAddress]) : EndpointSubset = copy(notReadyAddresses = notReadyAddresses.map(f))

  /** Returns a new data with ports set to new value */
  def withPorts(value: Seq[io.k8s.api.core.v1.EndpointPort]) : EndpointSubset = copy(ports = Some(value))
  /** Appends new values to ports */
  def addPorts(newValues: io.k8s.api.core.v1.EndpointPort*) : EndpointSubset = copy(ports = Some(ports.fold(newValues)(_ ++ newValues)))
  /** if ports has a value, transforms to the result of function*/
  def mapPorts(f: Seq[io.k8s.api.core.v1.EndpointPort] => Seq[io.k8s.api.core.v1.EndpointPort]) : EndpointSubset = copy(ports = ports.map(f))
}

object EndpointSubset {

    implicit val encoder : Encoder[io.k8s.api.core.v1.EndpointSubset] = new Encoder[io.k8s.api.core.v1.EndpointSubset] {
        def apply[T : Builder](o: io.k8s.api.core.v1.EndpointSubset) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("addresses", o.addresses)
            .write("notReadyAddresses", o.notReadyAddresses)
            .write("ports", o.ports)
            .build
        }
    }

    implicit val decoder: Decoder[EndpointSubset] = new Decoder[EndpointSubset] {
      def apply[T : Reader](t: T): Either[String, EndpointSubset] = for {
          obj <- ObjectReader(t)
          addresses <- obj.readOpt[Seq[io.k8s.api.core.v1.EndpointAddress]]("addresses")
          notReadyAddresses <- obj.readOpt[Seq[io.k8s.api.core.v1.EndpointAddress]]("notReadyAddresses")
          ports <- obj.readOpt[Seq[io.k8s.api.core.v1.EndpointPort]]("ports")
      } yield EndpointSubset (
          addresses = addresses,
          notReadyAddresses = notReadyAddresses,
          ports = ports
        )
    }
}

